/*
Szymon Rusinkiewicz

graphics.cc
This encapsulates all the (GLUT) graphics routines.
*/

#include <glut.h>

#include "sys_dependent.h"
#include "bv.h"
#include "graphics.h"
#include "bv_brdf_panels.h"

static void GRreshape(int,int);

/* Visibility tracking */
static BOOL I_am_mapped=TRUE;
static void GRvisfunc(int state)
{
	if (state == GLUT_VISIBLE)
		I_am_mapped=TRUE;
		// GLUT will take care of redrawing
	else if (state == GLUT_NOT_VISIBLE)
		I_am_mapped=FALSE;
}

/* The idle/refresh functions */
static int mylastrefresh=0;
static int lastviewers=0;
static BOOL lastuselog=FALSE;
static usecos_options lastusecos=USECOS_NONE;

static inline void refreshall()
{
	int i;
        for (i=0; i<numwindows; i++) {
                GRsetwin(i+2);
		glutPostRedisplay();
        }
}

enum redraw_what_enum { REDRAW_MYSELF, REDRAW_WINDOW, REDRAW_ALL };
static inline void need_redraw(enum redraw_what_enum redraw_what)
{
	switch(redraw_what) {
		case REDRAW_MYSELF:
			glutPostRedisplay();
			return;
		case REDRAW_WINDOW:
			mylastrefresh = lastrefresh;
			refreshall();
			return;
		case REDRAW_ALL:
			lastrefresh++;
			return;
	}
}

static void do_idle()
{
	BOOL something_happened=FALSE;

	yield();

	if (bv_suggest_quit)
		cleanup();

	update_brdf_panel();
	if (BRDFchanged) {
		BRDFchanged = FALSE;
		need_redraw(REDRAW_WINDOW);
		changetitle();
		something_happened=TRUE;
	}

	if (lastviewers != whichviewers) {
		updateviewers();
		GRreshape(top_window_x,top_window_y);
		need_redraw(REDRAW_WINDOW);
		something_happened=TRUE;
	}

	if (lastuselog != uselog) {
		changetitle();
		lastuselog = uselog;
	}
	
	if (lastusecos != usecos) {
		changetitle();
		lastusecos = usecos;
	}
	
	if (lastrefresh != mylastrefresh) {
		need_redraw(REDRAW_WINDOW);
		something_happened=TRUE;
	}

	// Spin the 3D view...
	if (spin3Dview) {
		dospin(something_happened); // Sets current window
		need_redraw(REDRAW_MYSELF);
		something_happened=TRUE;
	}
		
	if (!something_happened)
		usleep(10000); // So we don't eat 100% of CPU
}

static int whichmousebutton; /* Which mouse button was last pressed */

/* This is called when the mouse moves with a button held down */
static void mousemotionfunc(int x, int y)
{
	need_redraw(updatemouse(x,y,whichmousebutton,MOUSE_DRAG)?REDRAW_ALL:REDRAW_MYSELF);
}

/* This is called when a mouse button is pressed or released */
static void mousefunc(int button, int state, int x, int y)
{
	if (button == GLUT_LEFT_BUTTON) {
		whichmousebutton=1;
	} else if (button == GLUT_MIDDLE_BUTTON) {
		whichmousebutton=2;
	} else
		return;
	if (state == GLUT_DOWN) {
		//GRstopidle();
		need_redraw(updatemouse(x,y,whichmousebutton,MOUSE_PRESS)?REDRAW_ALL:REDRAW_MYSELF);
	} else /* if (state == GLUT_UP) */ {
		GRstartidle();
		updatemouse(x,y,whichmousebutton,MOUSE_RELEASE);
	}
}

static void keyboardfunc(unsigned char key, int x, int y)
{
	switch(key) {
		case '1':
		case '2':
		case '3':
		case '4':
		case '5':
		case '6':
			whichviewers ^= (1 << (key - '1'));
			updateviewers();
			GRreshape(top_window_x,top_window_y);
			need_redraw(REDRAW_ALL);
			return;

		case '=':
		case '+':
			brighter();
			need_redraw(REDRAW_WINDOW);
			changetitle();
			return;

		case '_':
		case '-':
			dimmer();
			need_redraw(REDRAW_WINDOW);
			changetitle();
			return;

		case 'L':
		case 'l':
			togglelog();
			need_redraw(REDRAW_ALL);
			changetitle();
			return;

		case 'C':
		case 'c':
			togglecos();
			need_redraw(REDRAW_ALL);
			changetitle();
			return;

		case 'S':
		case 's':
			togglespin3D();
			return;

		case ' ':
		case 'R':
		case 'r':
			resetview();
			need_redraw(REDRAW_ALL);
			return;

		case 'N':
		case 'n':
			newviewer();
			return;

		case '\033': /* Esc */
		case 'Q':
		case 'q':
			cleanup();
			return;

		default:
			return;
	}
}

static void skeyboardfunc(int key, int x, int y)
{
	switch(key) {
		case GLUT_KEY_UP: keyboardfunc('+',x,y); return;
		case GLUT_KEY_DOWN: keyboardfunc('-',x,y); return;
		default: return;
	}
}

/* Just for fun, let's create a menu! */
static void menucallback(int val)
{
	/* Menu functions correspond to keyboard commands */
	keyboardfunc((unsigned char)val,0,0);
}

static int GRmakesubmenu();

static void GRmakemenu()
{
	int myid=glutCreateMenu(menucallback);
	int subid=GRmakesubmenu();
	glutSetMenu(myid);
	glutAddMenuEntry("Toggle Cosine (c)",'c');
	glutAddMenuEntry("Toggle Logarithm (l)",'l');
	glutAddMenuEntry("Brighter (+)",'+');
	glutAddMenuEntry("Dimmer (-)",'-');
	glutAddMenuEntry("Spin 3D view (s)",'s');
	glutAddSubMenu("Resolution",subid);
	glutAddMenuEntry("Reset View (r)",'r');
	glutAddMenuEntry("New Viewer (n)",'n');
	glutAddMenuEntry("Quit (q)",'q');
}

static void switchres(int r)
{
	if ((r == -999) || (r > MIN(screenx,screeny)))
		r = MIN(screenx,screeny);
	changeres(r);
	need_redraw(REDRAW_ALL);
}

static int GRmakesubmenu()
{
	int myid=glutCreateMenu(switchres);
	glutAddMenuEntry("48",48);
	glutAddMenuEntry("64",64);
	glutAddMenuEntry("96",96);
	glutAddMenuEntry("128",128);
	glutAddMenuEntry("192",192);
	glutAddMenuEntry("256",256);
	glutAddMenuEntry("384",384);
	glutAddMenuEntry("512",512);
	glutAddMenuEntry("Full",-999);
	return myid;
}

/* Initialization, etc. */
void GRinit(int *theargcp, char **theargv)
{
	glutInitDisplayMode(GLUT_DOUBLE | GLUT_RGBA | GLUT_DEPTH);
	glutInitWindowSize(top_window_x,top_window_y);
	glutInit(theargcp,theargv); /* Might change the above */
}

static void GRinitwindow()
{
	glutDisplayFunc(refresh);
	glutMouseFunc(mousefunc);
	glutMotionFunc(mousemotionfunc);
	glutKeyboardFunc(keyboardfunc);
	glutSpecialFunc(skeyboardfunc);
	glClearColor(0.0,0.0,0.0,1.0);
	glClearDepth(1.0);
	glLineWidth(3.0);
	glPointSize(5.0);
	glShadeModel(GL_FLAT);
	glDepthFunc(GL_LEQUAL);
	glEnable(GL_NORMALIZE);
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_BLEND);
	glutAttachMenu(GLUT_RIGHT_BUTTON);
}

void GRstart(char *name)
{
	glutCreateWindow(name);
	GRmakemenu();
	GRinitwindow();
	glutReshapeFunc(GRreshape);
	glutVisibilityFunc(GRvisfunc);
	def_top_window_x = top_window_x =
			glutGet(GLUT_WINDOW_WIDTH);
	def_top_window_y = top_window_y =
			glutGet(GLUT_WINDOW_HEIGHT);
	screenx = top_window_x/numwindows;
	screeny = top_window_y;
	
	lastviewers=whichviewers;

	int i;
	for (i=0;i<maxwindows;i++) {
		glutSetWindow(1);
		glutCreateSubWindow(1,i*screenx,0,screenx,screeny);
		if (i<numwindows)
			glutShowWindow();
		else
			glutHideWindow();
		GRinitwindow();
	}

	changetitle();
	GRstartidle();
	glutMainLoop();
}

static void GRreshape(int x, int y)
{
	glViewport(0,0,x,y);
	GRblackscreen();

	def_top_window_x=top_window_x=x;
	def_top_window_y=top_window_y=y;
	screenx = top_window_x/numwindows;
	screeny = top_window_y;

	lastviewers=whichviewers;
	int i;
	for (i=0;i<maxwindows;i++) {
		GRsetwin(i+2);
		glutPositionWindow(i*screenx,0);
		glutReshapeWindow(screenx,screeny);
		if (i<numwindows)
			glutShowWindow();
		else
			glutHideWindow();
	}
}

void GRwindowtitle(char *t)
{
	int win=GRwin();
	GRsetwin(1);
	glutSetWindowTitle(t);
	GRsetwin(win);
}

/* The function do_idle will be called whenever we have nothing better to do */
void GRstartidle()
{
        glutIdleFunc(do_idle);
}

/* We will sleep if we have nothing better to do */
void GRstopidle()
{
	glutIdleFunc(NULL);
}

/* Do the drawing. (Flushes and exchanges buffers...) */
void GRcommit()
{
	glutSwapBuffers();
}

/* Return # of current window */
int GRwin()
{
	return glutGetWindow();
}

void GRsetwin(int i)
{
	glutSetWindow(i);
}

BOOL GRIammapped()
{
	return I_am_mapped;
}

/* Sets up the viewing matrix and viewport */
void GRsetview2d(float scale /* = 1.0 */ )
{
	float r;

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	if (screenx > screeny) {
		r=(float)screenx/(float)screeny;
		gluOrtho2D(-r*scale,r*scale,-scale,scale);
	} else {
		r=(float)screeny/(float)screenx;
		gluOrtho2D(-scale,scale,-r*scale,r*scale);
	}
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glViewport(0,0,screenx,screeny);
	glDisable(GL_LIGHTING);
	glDisable(GL_DEPTH_TEST);
}

void GRsetview3d(float theta, float phi, float scale /* = 1.0 */ )
{
	float r;
	const float pos=11;
	scale *= (pos-10)/pos;

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	if (screenx > screeny) {
		r=(float)screenx/(float)screeny;
		//glOrtho(-r*scale,r*scale,-scale,scale,-10,10);
		glFrustum(-r*scale,r*scale,-scale,scale,pos-10.,pos+10.);
	} else {
		r=(float)screeny/(float)screenx;
		//glOrtho(-scale,scale,-r*scale,r*scale,-10,10);
		glFrustum(-scale,scale,-r*scale,r*scale,pos-10.,pos+10.);
	}
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	gluLookAt(0,0,pos,0,0,0,0,1,0);
	glRotatef(rad2deg(theta),-1,0,0);
	glRotatef(rad2deg(phi),0,0,1);
	glViewport(0,0,screenx,screeny);
	glEnable(GL_DEPTH_TEST);
}

/* Clear the viewing area to black */
void GRblackscreen() { glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT); }

void GRwhite() { glColor3f(1.0,1.0,1.0); }
void GRgray() { glColor3f(0.3,0.3,0.3); }
void GRred() { glColor3f(1.0,0.0,0.0); }
void GRgreen() { glColor3f(0.0,1.0,0.0); }
void GRblue() { glColor3f(0.0,0.0,1.0); }
void GRyellow() { glColor3f(1.0,0.8,0.0); }

void GRpoint(float x, float y)
{
	glBegin(GL_POINTS);
	glVertex2f(x,y);
	glEnd();
}

void GRpoint3(float x, float y, float z)
{
	glBegin(GL_POINTS);
	glVertex3f(x,y,z);
	glEnd();
}

void GRline(float x1, float y1, float x2, float y2)
{
	glBegin(GL_LINES);
	glVertex2f(x1,y1);
	glVertex2f(x2,y2);
	glEnd();
}

void GRline3(float x1, float y1, float z1, float x2, float y2, float z2)
{
	glBegin(GL_LINES);
	glVertex3f(x1,y1,z1);
	glVertex3f(x2,y2,z2);
	glEnd();
}

void GRdashedline(float x1, float y1, float x2, float y2)
{
	glLineStipple(1,0xFFC0);
	glEnable(GL_LINE_STIPPLE);
	glBegin(GL_LINES);
	glVertex2f(x1,y1);
	glVertex2f(x2,y2);
	glEnd();
	glDisable(GL_LINE_STIPPLE);
}

void GRdashedline3(float x1, float y1, float z1, float x2, float y2, float z2)
{
	glLineStipple(1,0xFFC0);
	glEnable(GL_LINE_STIPPLE);
	glBegin(GL_LINES);
	glVertex3f(x1,y1,z1);
	glVertex3f(x2,y2,z2);
	glEnd();
	glDisable(GL_LINE_STIPPLE);
}

/* Draw a grayscale pixmap into the region (-1,-1)..(1,1) */
/* For the above coords to be correct, this must be called with the same value
of scale as GRsetview2d */
void GRdrawpixmap(unsigned char *p,int sx, int sy, float scale /* = 1.0 */ )
{
#ifndef USE_TEXMAP
#ifndef USE_POLYGONS
#define USE_DRAWPIXELS
#endif
#endif
#ifdef USE_TEXMAP
	glPixelStorei(GL_UNPACK_ALIGNMENT,1);
	glTexImage2D(GL_TEXTURE_2D,0,4,sx,sy,0,GL_LUMINANCE,GL_UNSIGNED_BYTE,p);
	glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
	glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
	glTexParameterf(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);
	glHint(GL_PERSPECTIVE_CORRECTION_HINT,GL_FASTEST);
	glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_DECAL);
	glEnable(GL_TEXTURE_2D);
	glBegin(GL_QUADS);
	glTexCoord2f(0,0); glVertex2f(-1,-1);
	glTexCoord2f(1,0); glVertex2f(1,-1);
	glTexCoord2f(1,1); glVertex2f(1,1);
	glTexCoord2f(0,1); glVertex2f(-1,1);
	glEnd();
	glDisable(GL_TEXTURE_2D);
#endif
#ifdef USE_DRAWPIXELS
	glPixelStorei(GL_UNPACK_ALIGNMENT,1);
	glRasterPos2f(-1,-1);
	glPixelZoom(((float)MIN(screenx,screeny)/sx/scale),
		    ((float)MIN(screenx,screeny)/sy/scale));
	glDrawPixels(sx,sy,GL_LUMINANCE,GL_UNSIGNED_BYTE,p);
#endif
#ifdef USE_POLYGONS
	float xpos = -1, ypos = -1;
	float dx = 2.0/sx, dy = 2.0/sy;
	unsigned char *next=p;
	for (int j=0; j<sy; j++) {
		xpos = -1;
		glBegin(GL_QUAD_STRIP);
		glVertex2f(xpos, ypos);
		glVertex2f(xpos, ypos+dy);
		for (int i=0; i<sx; i++) {
			xpos += dx;
			unsigned char thiscolor = *(next++);
			glColor3ub(thiscolor,thiscolor,thiscolor);
			glVertex2f(xpos, ypos);
			glVertex2f(xpos, ypos+dy);
		}
		ypos += dy;
		glEnd();
	}
#endif
}

typedef float pointlist[3];
/* Plots a surface patch defined as points on an x-y grid */
void GRsurfaceplot(pointlist *p, int sx, int sy, float theta, float phi)
{
	int i;
	//float n[sx*sy][3];
	pointlist *n = new pointlist[sx*sy];

	/* Compute normals */
	for (i=0;i<sx*sy;i++) {
		if ((i%sx == 0) || (i%sx == sx-1) ||
		    (i/sx == 0) || (i/sx == sy-1) ||
		    ((p[i][0] == 0) && (p[i][1] == 0) && (p[i][2] == 0))) {

			n[i][0]=n[i][1]=0.0; n[i][2] = 1.0;

		} else {
			FindCrossProd(	p[i+1][0]-p[i-1][0],
					p[i+1][1]-p[i-1][1],
					p[i+1][2]-p[i-1][2],
					p[i+sx][0]-p[i-sx][0],
					p[i+sx][1]-p[i-sx][1],
					p[i+sx][2]-p[i-sx][2],
					n[i][0],
					n[i][1],
					n[i][2] );
				// We assume that GL_NORMALIZE is enabled
		}
	}

	/* First draw the base plane */
	GRgray();
	glDisable(GL_DEPTH_TEST);
	glBegin(GL_QUADS);
	glVertex2f(-1,-1);
	glVertex2f(1,-1);
	glVertex2f(1,1);
	glVertex2f(-1,1);
	glEnd();
	glEnable(GL_DEPTH_TEST);

	/* Set up materials for the 3-d */
	{
	//GLfloat mat_diffuse[] = {1.0,0.5,0.5,1};
	GLfloat mat_diffuse[] = {0.67,0.73,1};
	GLfloat mat_specular[] = {1,1,1,1};
	GLfloat mat_shininess[] = {20};
	GLfloat light_ambient[] = {0.3,0.3,0.3,1.0};
	GLfloat light_diffuse[] = {0.5,0.5,0.5,1.0};
	GLfloat light_specular[] = {0.2,0.2,0.2,1.0};
	//GLfloat light_position[] = {0,0,1,0};
	GLfloat light_position[] = { -sin(theta)*sin(phi),
				     -sin(theta)*cos(phi),
				     cos(theta),
				     0 };
	glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE,mat_diffuse);
	glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,mat_specular);
	glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,mat_shininess);
	glLightfv(GL_LIGHT0,GL_AMBIENT,light_ambient);
	glLightfv(GL_LIGHT0,GL_DIFFUSE,light_diffuse);
	glLightfv(GL_LIGHT0,GL_SPECULAR,light_specular);
	glLightfv(GL_LIGHT0,GL_POSITION,light_position);
	}

	/* Draw it */
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glShadeModel(GL_SMOOTH);
	glBegin(GL_QUADS);
	for (i=0;i<sx*(sy-1);i++) {
		if (i % sx == sx-1) {
			continue;
		}
		glNormal3fv(n[i]);
		glVertex3fv(p[i]);
		glNormal3fv(n[i+1]);
		glVertex3fv(p[i+1]);
		glNormal3fv(n[i+sx+1]);
		glVertex3fv(p[i+sx+1]);
		glNormal3fv(n[i+sx]);
		glVertex3fv(p[i+sx]);
	}
	glEnd();
	glShadeModel(GL_FLAT);
	glDisable(GL_LIGHT0);
	glDisable(GL_LIGHTING);
	delete n;
}

/* Draws a nifty indicator of light source direction */
void GRlightsourcedir(float theta, float phi)
{
	int s=(int)((float)MIN(screenx,screeny)/3.0);
	int segments = (MIN(screenx,screeny) > 500)?32:16;

	glPushAttrib(GL_TRANSFORM_BIT|GL_VIEWPORT_BIT);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glOrtho(-2.6,3.6,-3.6,2.6,-10,10);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	glViewport(0,screeny-s,s,s);

	{
	GLfloat mat_diffuse[] = {0.5,0.5,0.5,1};
	GLfloat mat_specular[] = {0.5,0.5,0.5,1};
	GLfloat mat_shininess[] = {15};
	GLfloat light_ambient[] = {0.2,0.2,0.2,1.0};
	GLfloat light_diffuse[] = {0.5,0.5,0.5,1.0};
	GLfloat light_specular[] = {0.5,0.5,0.5,1.0};
	GLfloat light_position[] = {sin(theta)*cos(phi),sin(theta)*sin(phi),cos(theta),0};
	glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE,mat_diffuse);
	glMaterialfv(GL_FRONT_AND_BACK,GL_SPECULAR,mat_specular);
	glMaterialfv(GL_FRONT_AND_BACK,GL_SHININESS,mat_shininess);
	glLightfv(GL_LIGHT0,GL_AMBIENT,light_ambient);
	glLightfv(GL_LIGHT0,GL_DIFFUSE,light_diffuse);
	glLightfv(GL_LIGHT0,GL_SPECULAR,light_specular);
	glLightfv(GL_LIGHT0,GL_POSITION,light_position);
	}

	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glShadeModel(GL_SMOOTH);
	glEnable(GL_DEPTH_TEST);

	glutSolidSphere(1,segments,segments);
	glRotatef(rad2deg(phi),0,0,1);
	glRotatef(rad2deg(theta),0,1,0);
	glTranslatef(0,0,1.9);

	{
	GLfloat mat_diffuse[] = {0.5,1.0,0.5,1};
	GLfloat light_position[] = {0,0,1,0};
	glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE,mat_diffuse);
	glLightfv(GL_LIGHT0,GL_POSITION,light_position);
	}

	glRotatef(180,1,0,0);
	glutSolidCone(0.3,0.8,segments,1);
	GLUquadricObj *quadObj = gluNewQuadric();
	gluQuadricDrawStyle(quadObj,GLU_FILL);
	gluDisk(quadObj,0,0.3,segments,1);
	gluDeleteQuadric(quadObj);
	glRotatef(180,1,0,0);
	glTranslatef(0,0,1.7);

	{
	GLfloat mat_diffuse[] = {0.5,0.5,1.0,1};
	glMaterialfv(GL_FRONT_AND_BACK,GL_AMBIENT_AND_DIFFUSE,mat_diffuse);
	}

	glRotatef(180,1,0,0);
	glutSolidCone(0.1,2.5,segments,1);
	glRotatef(180,1,0,0);

	glDisable(GL_LIGHTING);
	glShadeModel(GL_FLAT);
	glDisable(GL_DEPTH_TEST);
	glPopAttrib();
}
