/*
Szymon Rusinkiewicz

bv_3d.cc
Code for a 3-d view of one slice of a BRDF.
*/

#include <sys/time.h>

#pragma implementation
#include "bv_3d.h"

#include "graphics.h"
#include "transform.h"

static float viewer_theta = PI/3.0, viewer_phi = PI/6.0;
static struct timeval LastRotateTime={0,0};
static BOOL mouse2pressed=FALSE, justspinning=FALSE;

void Viewer_3d::initangles()
{
	int i,j;
	float x,y;

	for (j=0;j<resolution;j++)
	  for (i=0;i<resolution;i++) {
		x=((float)i+0.5)/resolution*2.0-1.0;
		y=((float)j+0.5)/resolution*2.0-1.0;
		if (SQR(x)+SQR(y) < 1.0-EPS) {
			//angles[j*resolution+i][0]=compute_theta(x,y);
			angles[j*resolution+i][0]=sqrt(SQR(x)+SQR(y))*PI/2;
			angles[j*resolution+i][1]=compute_phi(x,y);
			angles[j*resolution+i][2]=sin(angles[j*resolution+i][0])*
						  cos(angles[j*resolution+i][1]);
			angles[j*resolution+i][3]=sin(angles[j*resolution+i][0])*
						  sin(angles[j*resolution+i][1]);
			angles[j*resolution+i][4]=cos(angles[j*resolution+i][0]);
		} else {
			angles[j*resolution+i][0]=-999;
		}
	}
}

void Viewer_3d::compute_brdfs(float theta, float phi)
{
	int i;
	float b;

	for (i=0;i<SQR(resolution);i++) {
		if (angles[i][0] == -999)
			brdfs[i][0] = brdfs[i][1] = brdfs[i][2] = 0.0;
		else {
			b=thebrdf->Eval(
				theta, phi,
				angles[i][0], angles[i][1]);
			b = MODIFY_COS(b,theta,angles[i][0]);
			b = CLAMP(MODIFY_LOG(b*brightness),0.0,10.0);
			brdfs[i][0] = angles[i][2]*b;
			brdfs[i][1] = angles[i][3]*b;
			brdfs[i][2] = angles[i][4]*b;
		}
	}
}

BOOL Viewer_3d::update_mouse(float mx, float my,
			     int whichbutton, int button_press)
{
	static float oldmx=0, oldmy=0;

	if (button_press == MOUSE_RELEASE) {
		mouse2pressed=FALSE;
		return FALSE;
	}
	spin3Dview = FALSE;
	if (whichbutton == 1) {
		clamp_to_circle(mx,my);
		float dummy;
		rotateZ(mx,my,dummy,-viewer_phi);
		brdf_in_x = mx; brdf_in_y = my;
		return TRUE;
	} else if (whichbutton == 2) {
		if (button_press == MOUSE_PRESS) {
			mouse2pressed=TRUE;
			oldmx = mx ; oldmy = my;
			return FALSE;
		}
		viewer_phi += (mx-oldmx);
		if (viewer_phi > PI)
			viewer_phi -= 2.0*PI;
		if (viewer_phi < -PI)
			viewer_phi += 2.0*PI;
		viewer_theta += (my-oldmy);
		if (viewer_theta > PI/2.0-EPS)
			viewer_theta = PI/2.0-EPS;
		if (viewer_theta < 0)
			viewer_theta = 0;
		oldmx = mx ; oldmy = my;
		return FALSE; /* Just changed viewing angle - no need to
				redraw everyone else. */
	}
	return FALSE;
}

void Viewer_3d::spinviewer(int recompute)
{
/* recompute:  0 - The viewer is spinning and nothing else is happening.
                   Don't bother to recompute the BRDF
	       1 - The viewer is spinning, but the BRDF has changed in some way
	      -1 - The viewer is just starting to spin. Just update
	           LastRotateTime.
*/
	struct timeval NewRotateTime;
	gettimeofday(&NewRotateTime,NULL);

	float rotate = (float)(NewRotateTime.tv_sec - LastRotateTime.tv_sec);
	rotate *= 1.0e6;
	rotate += (float)(NewRotateTime.tv_usec - LastRotateTime.tv_usec);
	rotate *= BV_3D_ROTATE_SCALE;
	if (rotate > BV_3D_MAX_ROTATE)
		rotate = BV_3D_MAX_ROTATE;

	if (recompute != -1)
		viewer_phi += rotate;
	if (viewer_phi > PI)
		viewer_phi -= 2.0*PI;
	LastRotateTime = NewRotateTime;
	justspinning = (recompute != 1);
}

void Viewer_3d::redraw()
{
	float ix=brdf_in_x, iy=brdf_in_y;
	float iz=sqrt(1.0-SQR(ix)-SQR(iy));
	GRsetview3d(viewer_theta,viewer_phi);
	GRblackscreen();
	if ((!mouse2pressed) && (!justspinning)) {
		compute_brdfs(compute_theta(ix,iy),
				compute_phi(ix,iy));
	}
	justspinning=FALSE;
	GRsurfaceplot(brdfs,resolution,resolution,viewer_theta,viewer_phi);

	/* Now draw the axes and incoming and outgoing angles */
	GRwhite();
	GRline3(0,0,0,0,0,1);
	GRline3(0,0,0,1,0,0);
	GRgreen();
	GRline3(0,0,0,ix,iy,iz);
	GRwhite();
	GRline3(-ix,-iy,iz,0,0,0);
	GRgreen();
	GRdashedline3(-ix,-iy,iz,0,0,0);
}

