#ifndef _BV_H
#define _BV_H

/*
Szymon Rusinkiewicz

bv.h
This contains defines and prototypes relating to the viewer
*/

/*
Basic theory of operation:

The previous version of this program had exactly one BRDF/viewer per process.
In the interests of user friendliness (sanity?), this was changed to one BRDF
per process, but with possibly many viewers (subwindows). So, for each process
there is exactly one BRDF (thebrdf), and glut is used to create exactly one
top-level window. The viewers array (bv.cc) then stores a bunch of viewers
that are all displaying some view of this BRDF. The viewers process their own
mouse input, and notify all other viewers of changes in viewing angle.

Several variables are stored in a shared memory segment that we
allocate/connect to on startup. This is so that other processes, possibly with
different BRDFs, can snoop on these values, and redraw themselves to keep up.

A final trio of nifty features are Xforms-based BRDF parameter/info panels,
BRDF picker (if no BRDF was specified on the command line), and GUI (to
control which viewers are shown and a few parameters).

The organization is as follows:
main.cc has the initialization routines - it calls one or more of bv_main(),
 bv_gui_main(), or pick_brdf_main(). It is also responsible for initializing
 the BRDF on startup
shm.cc initializes the shared memory segment. Note: it is vital to keep
 bv_active up to date if we do any fancy fork()ing or exec()ing, otherwise
 the shm segment won't get freed.
bv.cc and graphics.cc contain the infrastructure for making the viewers work.
 Basically, if it calls glut or gl directly it's in graphics.cc, and the
 callbacks and event handlers are in bv.cc (exception - the idle loop is in
 graphics.cc)
bv_2d.cc bv_3d.cc and bv_lit.cc are the viewers. For efficiency they
 might precompute a list of angles at which to evaluate the BRDF.
pick_brdf_main.cc bv_gui_main.cc bv_brdf_panels/whatever_panel1.cc do the
 Xforms stuff. The corresponding .c files are generated by fdesign and
 should not be changed.

*/

#include "brdf.h"
#include "shm.h"

#pragma interface

#define STARTING_RESOLUTION 64
#define STARTING_SCREEN_X 400
#define STARTING_SCREEN_Y 300

#define NUM_VIEWERS 4
#define STARTING_VIEWERS (1<<1)

#define BV_3D_ROTATE_SCALE 0.0000004
#define BV_3D_MAX_ROTATE 0.1

#define LOGSCALE1 0.1
#define LOGSCALE2 100.0
#define MODIFY_LOG(x) (uselog ? (LOGSCALE1*log(LOGSCALE2*(x)+1.0)) : (x))

typedef enum {	USECOS_NONE,
		USECOS_IN,
		USECOS_OUT,
		USECOS_PROD,
		USECOS_SUM,
		USECOS_INVALID } usecos_options;
#define MODIFY_COS(x,thi,tho) ( \
		(usecos == USECOS_IN) ? \
			(x)*cos(thi) : \
		(usecos == USECOS_OUT) ? \
			(x)*cos(tho) : \
		(usecos == USECOS_PROD) ? \
			(x)*cos(thi)*cos(tho) : \
		(usecos == USECOS_SUM) ? \
			(x)*(cos(thi)+cos(tho)) : \
			(x))

class Viewer {
protected:
	int resolution;
public:
	Viewer(const int res) : resolution(res) {}
	virtual void redraw() =0;
	virtual BOOL update_mouse(float,float,int,int) =0;
	virtual void changeres(int) =0;
};


/* From main.cc */
extern char myname[];
extern BRDF *thebrdf;

/* From pick_brdf_main.cc */
extern void pick_brdf_main();

/* From bv_gui_main.cc */
extern void bv_gui_main();

/* From bv.cc */
extern int top_window_x,top_window_y;
extern int screenx,screeny;
extern int numwindows;
extern const int maxwindows;
extern float brightness;
extern BOOL spin3Dview;

extern void bv_main();
extern void changetitle();
extern void updateviewers();
extern void refresh();
extern void dospin(int);
extern BOOL updatemouse(int,int,int,BOOL);
extern void changeres(int);
extern void brighter();
extern void dimmer();
extern void togglecos();
extern void togglelog();
extern void togglespin3D();
extern void resetview();
extern void newviewer();
extern void cleanup();

/* From version.cc */
extern char *BV_VERSION;
extern char *BV_COMPILED;

#endif
