#ifndef _MYCOMPLEX_H
#define _MYCOMPLEX_H

/*
Szymon Rusinkiewicz

mycomplex.h
Unfortunately, <complex.h>, <complex>, and friends are not portable, since much
of the world still has not adopted the C++ standard library.  Therefore, we
roll our own...  Note that this is largely based on the code that comes with
G++, and hence is covered under the GPL.

Not implemented: all trig functions, all exponential/log/pow functions, IO
*/

#include <math.h>

#undef TRUE
#undef FALSE

#ifdef _BOOL
#define BOOL bool
#define TRUE true
#define FALSE false
#else
#define BOOL int
#define TRUE 1
#define FALSE 0
#endif

#undef ABS
template <class T> inline T ABS(const T &x) { return x<0 ? -x : x; }


template <class FLOAT_TYPE>
class complex {
private:
	FLOAT_TYPE re, im;
public:
	complex(FLOAT_TYPE _re=0, FLOAT_TYPE _im=0) : re(_re), im(_im) {}
	complex& operator += (const complex&);
	complex& operator -= (const complex&);
	complex& operator *= (const complex&);
	complex& operator /= (const complex&);
	FLOAT_TYPE real() const { return re; }
	FLOAT_TYPE imag() const { return im; }

	friend FLOAT_TYPE real (const complex&);
	friend FLOAT_TYPE imag (const complex&);
	friend complex operator + (const complex&, const complex&);
	friend complex operator - (const complex&, const complex&);
	friend complex operator * (const complex&, const complex&);
	friend complex operator / (const complex&, const complex&);
	friend complex operator + (const complex&);
	friend complex operator - (const complex&);
	friend BOOL operator == (const complex&, const complex&);
	friend BOOL operator != (const complex&, const complex&);
	friend complex polar (FLOAT_TYPE, FLOAT_TYPE);
};

template <class FLOAT_TYPE>
inline FLOAT_TYPE real(const complex<FLOAT_TYPE>& c)
{
	return c.real();
}

template <class FLOAT_TYPE>
inline FLOAT_TYPE imag(const complex<FLOAT_TYPE>& c)
{
	return c.imag();
}

template <class FLOAT_TYPE>
inline FLOAT_TYPE abs(complex<FLOAT_TYPE>& x)
{
	return hypot(real(x), imag(x));
}

template <class FLOAT_TYPE>
inline FLOAT_TYPE arg(complex<FLOAT_TYPE>& x)
{
	return atan2(imag(x), real(x));
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE> polar(FLOAT_TYPE r, FLOAT_TYPE th)
{
	return complex<FLOAT_TYPE> (r*cos(th), r*sin(th));
}

template <class FLOAT_TYPE>
inline complex <FLOAT_TYPE> conj(const complex<FLOAT_TYPE>& x)
{
	return complex<FLOAT_TYPE>(real(x), -imag(x));
}

template <class FLOAT_TYPE>
inline FLOAT_TYPE norm(const complex<FLOAT_TYPE>& x)
{
	return real(x)*real(x) + imag(x)*imag(x);
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>&
complex<FLOAT_TYPE>::operator += (const complex<FLOAT_TYPE>& c)
{
	re += c.re;
	im += c.im;
	return *this;
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>&
complex<FLOAT_TYPE>::operator -= (const complex<FLOAT_TYPE>& c)
{
	re -= c.re;
	im -= c.im;
	return *this;
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>&
complex<FLOAT_TYPE>::operator *= (const complex<FLOAT_TYPE>& c)
{
	FLOAT_TYPE tmp = re*c.re - im*c.im;
	im = re*c.im + im*c.re;
	re = tmp;
	return *this;
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>&
complex<FLOAT_TYPE>::operator /= (const complex<FLOAT_TYPE>& c)
{
	FLOAT_TYPE ar = ABS(c.re);
	FLOAT_TYPE ai = ABS(c.im);
	FLOAT_TYPE nr, ni, t, d;
	if (ar <= ai) {
		t = c.re / c.im;
		d = c.im * (1 + t*t);
		nr = (re * t + im) / d;
		ni = (im * t - re) / d;
	} else {
		t = c.im / c.re;
		d = c.re * (1 + t*t);
		nr = (re + im * t) / d;
		ni = (im - re * t) / d;
	}
	re = nr;
	im = ni;
	return *this;
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>
operator + (const complex<FLOAT_TYPE>& x, const complex<FLOAT_TYPE>& y)
{
	return complex<FLOAT_TYPE>(real(x)+real(y), imag(x)+imag(y));
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>
operator - (const complex<FLOAT_TYPE>& x, const complex<FLOAT_TYPE>& y)
{
	return complex<FLOAT_TYPE>(real(x)-real(y), imag(x)-imag(y));
}


template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>
operator * (const complex<FLOAT_TYPE>& x, const complex<FLOAT_TYPE>& y)
{
	return complex<FLOAT_TYPE>(real(x)*real(y) - imag(x)*imag(y),
				   real(x)*imag(y) + imag(x)*real(y));
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>
operator / (const complex<FLOAT_TYPE>& x, const complex<FLOAT_TYPE>& y)
{
	FLOAT_TYPE ar = ABS(real(y));
	FLOAT_TYPE ai = ABS(imag(y));
	FLOAT_TYPE nr, ni, t, d;
	if (ar <= ai) {
		t = real(y) / imag(y);
		d = imag(y) * (1+t*t);
		nr = (real(x)*t + imag(x))/d;
		ni = (imag(x)*t - real(x))/d;
	} else {
		t = imag(y) / real(y);
		d = real(y) * (1+t*t);
		nr = (real(x) + imag(x)*t)/d;
		ni = (imag(x) - real(x)*t)/d;
	}
	return complex<FLOAT_TYPE>(nr, ni);
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>
operator + (const complex<FLOAT_TYPE>& x)
{
	return x;
}

template <class FLOAT_TYPE>
inline complex<FLOAT_TYPE>
operator - (const complex<FLOAT_TYPE>& x)
{
	return complex<FLOAT_TYPE>(-real(x), -imag(x));
}

template <class FLOAT_TYPE>
inline BOOL
operator == (const complex<FLOAT_TYPE>& x, const complex<FLOAT_TYPE>& y)
{
	return ((real(x) == real(y)) && (imag(x) == imag(y)));
}

template <class FLOAT_TYPE>
inline BOOL
operator != (const complex<FLOAT_TYPE>& x, const complex<FLOAT_TYPE>& y)
{
	return ((real(x) != real(y)) || (imag(x) != imag(y)));
}

template <class FLOAT_TYPE>
complex<FLOAT_TYPE> sqrt(const complex<FLOAT_TYPE>& x)
{
	FLOAT_TYPE r = ABS(x);
	FLOAT_TYPE nr, ni;
	if (r == 0.0)
		nr = ni = r;
	else if (real (x) > 0) {
		nr = sqrt(0.5 * (r + real(x)));
		ni = imag(x) / nr / 2;
	} else {
		ni = sqrt(0.5 * (r - real(x)));
		if (imag(x) < 0)
        		ni = - ni;
		nr = imag(x) / ni / 2;
	}
	return complex<FLOAT_TYPE> (nr, ni);
}


typedef complex<float> float_complex;
typedef complex<double> double_complex;
typedef complex<long double> long_double_complex;

#endif
