/*
Szymon Rusinkiewicz

brdf_ward.cc
Ward's Anisotropic (Elliptical) Gaussian BRDF

Note that the original "full" version is implemented, not the approximation
suggested in the paper.

Ward, Gregory. "Measuring and Modeling Anisotropic Reflection", _SIGGRAPH_
Proceedings, 1992.
*/

#pragma implementation
#include "brdf_ward.h"

void Ward_BRDF::Usage()
{
	printf(	"ward        Ward's Anisotropic (Elliptical) Gaussian BRDF\n"
		"            Parameters are: sx [0.05]   Roughness in X\n"
		"                            sy [0.3]    Roughness in Y\n"
		"                            rs [0.05]   Specular reflectivity\n"
		"                            rd [0.4]    Diffuse reflectivity\n"
	      );
}

BRDF *Ward_BRDF::Create(const char *params)
{
	float sx=0.05,sy=0.3,rs=0.05,rd=0.4;
	
	if (params)
		sscanf(params,"%f %f %f %f",&sx,&sy,&rs,&rd);

	return new Ward_BRDF(sx,sy,rs,rd);
}

void Ward_BRDF::Getname(char *nm)
{
	sprintf(nm,"Ward  sx=%.2f, sy=%.2f, rs=%.2f, rd=%.2f",
			     sx,      sy,      rs,      rd);
}

brdf_return_t Ward_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			      BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	/* Beta is the angle btw the normal and H (the halfway vector) */
	BRDF_FLOAT ix=sin(theta_in)*cos(phi_in);
	BRDF_FLOAT iy=sin(theta_in)*sin(phi_in);
	BRDF_FLOAT iz=CLAMP(cos(theta_in),EPS,1.0);
	BRDF_FLOAT ox=sin(theta_out)*cos(phi_out);
	BRDF_FLOAT oy=sin(theta_out)*sin(phi_out);
	BRDF_FLOAT oz=CLAMP(cos(theta_out),EPS,1.0);
	BRDF_FLOAT lh=sqrt(SQR(ix+ox)+SQR(iy+oy)+SQR(iz+oz));
	BRDF_FLOAT hx=(ix+ox)/lh;
	BRDF_FLOAT hy=(iy+oy)/lh;
	BRDF_FLOAT hz=(iz+oz)/lh;
	BRDF_FLOAT cosbeta=CLAMP(hz,EPS,1.0-EPS);
	BRDF_FLOAT sqsinbeta=1.0-SQR(cosbeta);
	BRDF_FLOAT sqtanbeta=sqsinbeta/SQR(cosbeta);
	BRDF_FLOAT phi_h=compute_phi(hx,hy);

	return rd/PI + rs/sqrt(iz*oz)/(4.0*PI*sx*sy)*
		exp(-sqtanbeta*(SQR(cos(phi_h)/sx)+SQR(sin(phi_h)/sy)));
		
}

