/*
Szymon Rusinkiewicz

brdf_ts.cc
The Cook-Torrance-Sparrow model, with Beckmann distribution function, Blinn
geometric attenuation, and Fresnel reflection term
*/

#pragma implementation
#include "brdf_ts.h"

#include "fresnel.h"

void TS_BRDF::Usage()
{
	printf(	"ts          The Torrance-Sparrow-Blinn-Cook-Beckmann BRDF\n"
		"            Parameters are: m [0.08]    Surface roughness\n"
		"                            nreal [1.6] Index of refraction (real part)\n"
		"                            nimag [-0.2]                    (imaginary part)\n"
		"                            rs [0.6]    Specular reflectivity\n"
		"                            rd [0.4]    Diffuse reflectivity\n"
	      );
}

BRDF *TS_BRDF::Create(const char *params)
{
	float m=0.08,nreal=1.6,nimag=-0.2,rs=0.6,rd=0.4;
	
	if (params)
		sscanf(params,"%f %f %f %f %f",&m,&nreal,&nimag,&rs,&rd);

	return new TS_BRDF(m,nreal,nimag,rs,rd);
}

void TS_BRDF::Getname(char *nm)
{
	sprintf(nm,"Torrance-Sparrow  m=%.2f, n=%.2f%+.2fi, rs=%.2f, rd=%.2f",
					m,   real(n),imag(n),  rs,      rd);
}

brdf_return_t TS_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			    BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	/* Beta is the angle between the normal and the halfway vector */
	BRDF_FLOAT ix=sin(theta_in)*cos(phi_in);
	BRDF_FLOAT iy=sin(theta_in)*sin(phi_in);
	BRDF_FLOAT iz=cos(theta_in);
	BRDF_FLOAT ox=sin(theta_out)*cos(phi_out);
	BRDF_FLOAT oy=sin(theta_out)*sin(phi_out);
	BRDF_FLOAT oz=cos(theta_out);
	BRDF_FLOAT lh=sqrt(SQR(ix+ox)+SQR(iy+oy)+SQR(iz+oz));
	BRDF_FLOAT hx=(ix+ox)/lh;
	BRDF_FLOAT hy=(iy+oy)/lh;
	BRDF_FLOAT hz=(iz+oz)/lh;
	BRDF_FLOAT cosbeta=CLAMP(hz,EPS,1.0-EPS);
	BRDF_FLOAT sinbeta=sqrt(1.0-SQR(cosbeta));
	BRDF_FLOAT tanbeta=sinbeta/cosbeta;
	BRDF_FLOAT c=CLAMP(ix*hx+iy*hy+iz*hz,EPS,1.0);

	BRDF_FLOAT D=exp(-SQR(tanbeta/m))/SQR(2.0*m*SQR(cosbeta));
	BRDF_FLOAT G=MIN(1.0,MIN(2.0*cosbeta*iz/c,2.0*cosbeta*oz/c));
	BRDF_FLOAT F=Fresnel_term(c,n);

	return (rd+rs*F*D*G/iz/oz)/PI;
}

