/*
Szymon Rusinkiewicz

brdf_strauss.cc
Strauss's empirical BRDF
*/

#pragma implementation
#include "brdf_strauss.h"

void Strauss_BRDF::Usage()
{
	printf(	"strauss     Strauss's empirical BRDF\n"
		"            Parameters are: s [0.75]    The \"smoothness\" of the surface\n"
		"                            m [0.5]     The \"metalness\" of the surface\n"
		"                            ks [0.5]    Specular reflectivity\n"
		"                            kd [0.5]    Diffuse reflectivity\n"
	      );
}

BRDF *Strauss_BRDF::Create(const char *params)
{
	float s=0.75,m=0.5,ks=0.5,kd=0.5;
	
	if (params)
		sscanf(params,"%f %f %f %f",&s,&m,&ks,&kd);

	return new Strauss_BRDF(s,m,ks,kd);
}

void Strauss_BRDF::Getname(char *nm)
{
	sprintf(nm,"Strauss  s=%.2f, m=%.2f, ks=%.2f, kd=%.2f",
			       s,      m,    ks,      kd);
}

brdf_return_t Strauss_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			      BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	const BRDF_FLOAT kb=0.1, kf=1.12, kg=1.01; // Strauss's magic constants

	BRDF_FLOAT theta_r = theta_in;
	BRDF_FLOAT phi_r = phi_in+PI;
	BRDF_FLOAT rdotv = sin(theta_r)*cos(phi_r)*sin(theta_out)*cos(phi_out) +
			   sin(theta_r)*sin(phi_r)*sin(theta_out)*sin(phi_out) +
			   cos(theta_r)*cos(theta_out);
	BRDF_FLOAT frd=kd*(1.0-s*s*s)*(1.0-m*s);
	BRDF_FLOAT t_in=theta_in*2.0/PI;
	BRDF_FLOAT t_out=theta_out*2.0/PI;
	BRDF_FLOAT F=(1.0/SQR(t_in-kf)-1.0/SQR(kf))/(1.0/SQR(1.0-kf)-1.0/SQR(kf));
	BRDF_FLOAT G=(1.0/SQR(1.0-kg)-1.0/SQR(t_out-kg))/(1.0/SQR(1.0-kg)-1.0/SQR(kg));
	BRDF_FLOAT sa=MIN(1.0,ks+(ks+kb)*F*G);
	BRDF_FLOAT sc=1.0+m*(1.0-F);
	BRDF_FLOAT frs=sa*sc*pow(CLAMP(rdotv,0.0,1.0),3.0/(1.0-s))/cos(theta_out);

	return frs+frd;
}

