/*
Szymon Rusinkiewicz

brdf_phong.cc
The Phong BRDF

Note that the Phong model is usually defined as an *illumination* model, not
a BRDF. This means that we have to divide by cos(theta_in) to get the
corresponding BRDF. The physically plausible Phong BRDF (as defined by Lewis
_Computer Graphics Forum_, June 1994) is kd/PI + ks*(n+2)/2/PI*cos^n(r dot v)

Also, we implement the Schlick BRDF as a simple hack of this.
*/

#pragma implementation
#include "brdf_phong.h"

void Phong_BRDF::Usage()
{
	printf(	"phong       The Phong BRDF\n"
		"            Parameters are: n [50]      Phong exponent\n"
		"                            ks [0.4]    Specular coefficient\n"
		"                            kd [0.4]    Diffuse coefficient\n"
		"                            [rv|nh]     Use R dot V (default) or N dot H\n"
		"                            [orig|new]  Use the original (physically incorrect)\n"
		"                                         Phong illumination model (default), or\n"
		"                                         Lewis's physically plausible Phong-like\n"
		"                                         BRDF\n"
	      );
}

void Schlick_BRDF::Usage()
{
	printf(	"schlick     The Schlick BRDF\n"
		"            Parameters are the same as for phong\n");
}

BRDF *Phong_BRDF::Create_Phonglike(const char *params,
						int Phong_Flags=PHONG_PHONG)
{
	float n=50,ks=0.4,kd=0.4;
	char flag1[255]="",flag2[255]="";
	
	if (params)
		sscanf(params,"%f %f %f %s %s",&n,&ks,&kd,flag1,flag2);

	if (*flag1)
		if (strcasecmp(flag1,"rv") == 0)
			Phong_Flags |= PHONG_RV;
		else if (strcasecmp(flag1,"nh") == 0)
			Phong_Flags |= PHONG_NH;
		else if (strcasecmp(flag1,"orig") == 0)
			Phong_Flags |= PHONG_ORIG;
		else if (strcasecmp(flag1,"new") == 0)
			Phong_Flags |= PHONG_NEW;
		else
			return NULL;

	if (*flag2)
		if (strcasecmp(flag2,"rv") == 0)
			Phong_Flags |= PHONG_RV;
		else if (strcasecmp(flag2,"nh") == 0)
			Phong_Flags |= PHONG_NH;
		else if (strcasecmp(flag2,"orig") == 0)
			Phong_Flags |= PHONG_ORIG;
		else if (strcasecmp(flag2,"new") == 0)
			Phong_Flags |= PHONG_NEW;
		else
			return NULL;

	return new Phong_BRDF(n,ks,kd,Phong_Flags);
}

void Phong_BRDF::Getname(char *nm)
{
	char *nm1,*nm2,*nm3;

	nm1=(Phong_Flags&PHONG_NEW)?"Modified ":"";
	nm2=(Phong_Flags&PHONG_SCHLICK)?"Schlick":"Phong";
	nm3=(Phong_Flags&PHONG_NH)?"N dot H":"R dot V";

	sprintf(nm,"%s%s (%s)  n=%.2f, ks=%.2f, kd=%.2f",
		    nm1,nm2,nm3, n,	  ks,	   kd);
}

brdf_return_t Phong_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			       BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	BRDF_FLOAT val;

	if (Phong_Flags & PHONG_NH) {
		BRDF_FLOAT ix=sin(theta_in)*cos(phi_in);
		BRDF_FLOAT iy=sin(theta_in)*sin(phi_in);
		BRDF_FLOAT iz=cos(theta_in);
		BRDF_FLOAT ox=sin(theta_out)*cos(phi_out);
		BRDF_FLOAT oy=sin(theta_out)*sin(phi_out);
		BRDF_FLOAT oz=cos(theta_out);
		BRDF_FLOAT lh=sqrt(SQR(ix+ox)+SQR(iy+oy)+SQR(iz+oz));
		BRDF_FLOAT hz=(iz+oz)/lh;
		BRDF_FLOAT ndoth=hz;
		val=ndoth;
	} else {
		BRDF_FLOAT theta_r = theta_in;
		BRDF_FLOAT phi_r = phi_in+PI;
		BRDF_FLOAT rdotv = sin(theta_r)*cos(phi_r)*sin(theta_out)*cos(phi_out) +
				   sin(theta_r)*sin(phi_r)*sin(theta_out)*sin(phi_out) +
				   cos(theta_r)*cos(theta_out);
		val=rdotv;
	}

	if (Phong_Flags & PHONG_SCHLICK)
		val=val/(n-n*val+val);
	else
		val=pow(CLAMP(val,0.0,1.0),n);

	if (Phong_Flags & PHONG_NEW)
		return kd/PI+ks*val*(n+2.0)/(2.0*PI);
	else
		return kd+ks*val/MAX(EPS,cos(theta_in));
}
