/*
Szymon Rusinkiewicz

brdf_on.cc
The rough diffuse surface model by Oren and Nayar. This is similar to the
Torrance-Sparrow model, but assumes that the underlying surface is diffuse,
not specular.

Note that what is implemented is the first simplification given in the paper,
not the "qualitative model".

Oren, Michael and Nayar, Shree K. "Generalization of Lambert's Reflectance
Model", _SIGGRAPH_ Proceedings, 1994.

*/

#pragma implementation
#include "brdf_on.h"

void ON_BRDF::Usage()
{
	printf(	"on          The Oren-Nayar rough diffuse surface BRDF\n"
		"            Parameters are: s [0.3]     Surface roughness\n"
		"                            r [1]       Reflectivity\n"
	      );
}

BRDF *ON_BRDF::Create(const char *params)
{
	float s=0.3, r=1;
	
	if (params)
		sscanf(params,"%f %f",&s,&r);

	return new ON_BRDF(s,r);
}

void ON_BRDF::Getname(char *nm)
{
	sprintf(nm,"Oren-Nayar  s=%.2f, r=%.2f",
				  s,	  r);
}

brdf_return_t ON_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			    BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	BRDF_FLOAT alpha=MAX(theta_in,theta_out);
	BRDF_FLOAT beta=MIN(theta_in,theta_out);
	BRDF_FLOAT c=cos(phi_out-phi_in);
	BRDF_FLOAT C1=1.0-0.5*SQR(s)/(SQR(s)+.33);
	BRDF_FLOAT C2=(c > 0) ?
		.45*SQR(s)/(SQR(s)+.09)*sin(alpha) :
		.45*SQR(s)/(SQR(s)+.09)*(sin(alpha)-CUBE(2.*beta/PI));
	BRDF_FLOAT C3=0.125*SQR(s)/(SQR(s)+.09)*SQR(4.*alpha*beta/SQR(PI));
	BRDF_FLOAT L1=r/PI*(
		C1 +
		c*C2*tan(beta) +
		(1.-ABS(c))*C3*tan((alpha+beta)/2.) );
	BRDF_FLOAT L2=.17*SQR(r)/PI*SQR(s)/(SQR(s)+.13)*(1.-c*SQR(2.*beta/PI));
	return L1+L2;
}

