/*
Szymon Rusinkiewicz

brdf_htsg.cc
The He-Torrance-Sillon-Greenberg BRDF
*/

#pragma implementation
#include "brdf_htsg.h"

#include "fresnel.h"

void HTSG_BRDF::Usage()
{
	printf(	"htsg        The He-Torrance-Sillon-Greenberg BRDF\n"
		"            Parameters are: t [20]      tau/lambda in He's model\n"
		"                            s [1]       sigma/lambda in He's model\n"
		"                            nreal [1.6] Index of refraction (real part)\n"
		"                            nimag [-0.2]                    (imaginary part)\n"
		"                            rs [0.8]    Specular reflectivity\n"
		"                            rd [0.2]    Diffuse reflectivity\n"
	      );
}

BRDF *HTSG_BRDF::Create(const char *params)
{
	float t=20,s=1,nreal=1.6,nimag=-0.2,rs=0.6,rd=0.4;
	
	if (params)
		sscanf(params,"%f %f %f %f %f %f",&t,&s,&nreal,&nimag,&rs,&rd);

	return new HTSG_BRDF(t,s,nreal,nimag,rs,rd);
}

void HTSG_BRDF::Getname(char *nm)
{
	sprintf(nm,"He-Torrance-Sillon-Greenberg  t=%.2f s=%.2f, n=%.2f%+.2fi, rs=%.2f, rd=%.2f",
						    t,     s,   real(n),imag(n),  rs,      rd);
}

/* Solves y=x*exp(x^2) for x given y */
static inline BRDF_FLOAT solve_xexpx2(BRDF_FLOAT y)
{
	BRDF_FLOAT x=(y>1)?sqrt(log(y)):y;
	int i;
	if (y > 3.0)
		for (i=0;i<4;i++)
			x=sqrt(log(y/x));
	else
		for (i=0;i<4;i++)
			x=0.5*(x+y*exp(-SQR(x)));
	return x;
}

static inline BRDF_FLOAT Htsg_sum(BRDF_FLOAT g, BRDF_FLOAT T)
{
	if (g < 15) {
		BRDF_FLOAT sum=0;
		int m=1;
		BRDF_FLOAT term1=exp(-g);
		for (;m<40;m++) {
			register BRDF_FLOAT recm=(1.0/(BRDF_FLOAT)m);
			term1*=g*recm;
			sum += term1*recm*exp(T*recm);
		}
		return sum;
	}
	BRDF_FLOAT mx=g;
	int i;
	for (i=0;i<4;i++)
		mx=g*exp(-1.5/mx-T/SQR(mx));
	return sqrt(g)*exp(mx*log(g)-g-(mx+1.5)*log(mx)+mx+T/mx);
}

brdf_return_t HTSG_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			    BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	if (theta_in < EPS) {
		theta_in = EPS;
		phi_in=0;
	}
	if (theta_out < EPS) {
		theta_out = EPS;
		phi_out=PI/2.0;
	}

	BRDF_FLOAT ix=sin(theta_in)*cos(phi_in);
	BRDF_FLOAT iy=sin(theta_in)*sin(phi_in);
	BRDF_FLOAT iz=cos(theta_in);
	BRDF_FLOAT ox=sin(theta_out)*cos(phi_out);
	BRDF_FLOAT oy=sin(theta_out)*sin(phi_out);
	BRDF_FLOAT oz=cos(theta_out);
	BRDF_FLOAT lh=sqrt(SQR(ix+ox)+SQR(iy+oy)+SQR(iz+oz));
	BRDF_FLOAT hx=(ix+ox)/lh;
	BRDF_FLOAT hy=(iy+oy)/lh;
	BRDF_FLOAT hz=(iz+oz)/lh;

	BRDF_FLOAT c=CLAMP(ix*hx+iy*hy+iz*hz,EPS,1.0);

	BRDF_FLOAT vx=ox+ix, vy=oy+iy, vz=oz+iz;
	BRDF_FLOAT vxy2=SQR(vx)+SQR(vy);
	BRDF_FLOAT six= -iy, siy=ix, siz=0;
	BRDF_FLOAT lsi=sqrt(SQR(six)+SQR(siy)+SQR(siz));
	six/=lsi; siy/=lsi; siz/=lsi;
	BRDF_FLOAT sox=oy, soy= -ox, soz=0;
	BRDF_FLOAT lso=sqrt(SQR(sox)+SQR(soy)+SQR(soz));
	sox/=lso; soy/=lso; soz/=lso;
	BRDF_FLOAT pix,piy,piz, pox,poy,poz;
	FindCrossProd(six,siy,siz,-ix,-iy,-iz,pix,piy,piz);
	FindCrossProd(sox,soy,soz,ox,oy,oz,pox,poy,poz);

	BRDF_FLOAT Ci=(t/s/2.0/tan(theta_in));
	BRDF_FLOAT Co=(t/s/2.0/tan(theta_out));
	BRDF_FLOAT Ei=erfc(Ci);
	BRDF_FLOAT Eo=erfc(Co);
	BRDF_FLOAT Ki=tan(theta_in)*Ei;
	BRDF_FLOAT Ko=tan(theta_out)*Eo;

	BRDF_FLOAT zo_over_so_over_sqrt2=solve_xexpx2((Ki+Ko)/sqrt(PI)/4.0);
	BRDF_FLOAT sigma=s/sqrt(1.0+2.0*SQR(zo_over_so_over_sqrt2));
	BRDF_FLOAT g=SQR(2.0*PI*sigma*(iz+oz));
	BRDF_FLOAT T = -vxy2*SQR(PI*t);
	BRDF_FLOAT D=SQR(PI*t/2.0)*Htsg_sum(g,T);

	BRDF_FLOAT S=(1.0-0.5*Ei)/(1.0+0.5*(1.0/sqrt(PI)/Ci-Ei))*
		     (1.0-0.5*Eo)/(1.0+0.5*(1.0/sqrt(PI)/Co-Eo));

	BRDF_FLOAT kkx, kky, kkz;
	FindCrossProd(ox,oy,oz,-ix,-iy,-iz,kkx,kky,kkz);
	BRDF_FLOAT kk4=SQR(SQR(kkx)+SQR(kky)+SQR(kkz));
	BRDF_FLOAT G=SQR((SQR(vx)+SQR(vy)+SQR(vz))/vz)/kk4*
		(SQR(sox*ix+soy*iy+soz*iz)+SQR(pox*ix+poy*iy+poz*iz))*
		(SQR(six*ox+siy*oy+siz*oz)+SQR(pix*ox+piy*oy+piz*oz));

	BRDF_FLOAT F=Fresnel_term(c,n);

	return (rd+F*G*S*D*rs/iz/oz)/PI;
}

