/*
Szymon Rusinkiewicz

brdf_hls.cc
The Hapke/Lommel-Seeliger Lunar-surface BRDF with Sch\:onberg backscattering
and the Hapke retrodirective function.

Hapke, Bruce. "A Theoretical Photometric Function for the Lunar Surface", in
_Journal of Geophysical Research_, Vol. 68, No. 15, 1963.
*/

#pragma implementation
#include "brdf_hls.h"

void HLS_BRDF::Usage()
{
	printf(	"hls         The Hapke/Lommel-Seeliger Lunar surface BRDF\n"
		"            Parameters are: g [0.6]     Surface density parameter\n"
		"                            f [0.1]     Forward scattering coefficient\n"
		"                            r [1.0]     Reflectivity\n"
	      );
}

BRDF *HLS_BRDF::Create(const char *params)
{
	float g=0.6,f=0.1,r=1.0;
	
	if (params)
		sscanf(params,"%f %f %f",&g,&f,&r);

	return new HLS_BRDF(g,f,r);
}

void HLS_BRDF::Getname(char *nm)
{
	sprintf(nm,"Hapke/Lommel-Seeliger  g=%.2f, f=%.2f, r=%.2f", g, f, r);
}

brdf_return_t HLS_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			     BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	/* Alpha is the angle btw the incoming and outgoing rays */
	BRDF_FLOAT ix=sin(theta_in)*cos(phi_in);
	BRDF_FLOAT iy=sin(theta_in)*sin(phi_in);
	BRDF_FLOAT iz=cos(theta_in);
	BRDF_FLOAT ox=sin(theta_out)*cos(phi_out);
	BRDF_FLOAT oy=sin(theta_out)*sin(phi_out);
	BRDF_FLOAT oz=cos(theta_out);
	BRDF_FLOAT cosalpha=CLAMP(ix*ox+iy*oy+iz*oz,EPS,1.0-EPS);
	BRDF_FLOAT sinalpha=sqrt(1.0-SQR(cosalpha));
	BRDF_FLOAT tanalpha=sinalpha/cosalpha;
	BRDF_FLOAT alpha=acos(cosalpha);

	BRDF_FLOAT R = (alpha < PI/2.0-EPS) ?
			2.0-tanalpha/2.0/g*(1.0-exp(-g/tanalpha))*(3.0-exp(-g/tanalpha)) :
			1;

	BRDF_FLOAT S=2.0/3.0/PI*((sinalpha+(PI-alpha)*cosalpha)/PI
				 + f*SQR(1.0 - cosalpha));

	return r*R*S/(iz+oz);
}


