/*
Szymon Rusinkiewicz

brdf_cyl.cc
Surface consisting of parallel cylinders, as calculated by Poulin and
Fournier.
*/

#pragma implementation
#include "brdf_cyl.h"

void Cyl_BRDF::Usage()
{
	printf(	"cyl         Surface with parallel-cylinder microgeometry\n"
		"            Parameters are: d [2]       Distance between cylinders\n"
		"                            h [0]       Height of floor between cylinders\n"
		"                            n [10]      Shininess of surface\n"
		"                            rs [0.8]    Specular reflectivity of surface\n"
		"                            rd [0.2]    Diffuse reflectivity of surface\n"
	      );
}

BRDF *Cyl_BRDF::Create(const char *params)
{
	float d=2, h=0, n=10, rs=0.8, rd=0.2;
	
	if (params)
		sscanf(params,"%f %f %f %f %f",&d,&h,&n,&rs,&rd);

	return new Cyl_BRDF(d,h,n,rs,rd);
}

void Cyl_BRDF::Getname(char *nm)
{
	sprintf(nm,"Cylinder  d=%.2f, h=%.2f, n=%.2f, rs=%.2f, rd=%.2f",
				d,	h,	n,	 rs,	  rd);
}

brdf_return_t Cyl_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			     BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	if (theta_in < EPS)
		theta_in = EPS;
	if (theta_out < EPS)
		theta_out = EPS;
	if (d < EPS)
		d = EPS;
	if (h > 1.0-EPS)
		h = 1.0-EPS;

	BRDF_FLOAT ix=sin(theta_in)*cos(phi_in);
	BRDF_FLOAT iy=sin(theta_in)*sin(phi_in);
	BRDF_FLOAT iz=cos(theta_in);
	BRDF_FLOAT ox=sin(theta_out)*cos(phi_out);
	BRDF_FLOAT oy=sin(theta_out)*sin(phi_out);
	BRDF_FLOAT oz=cos(theta_out);
	BRDF_FLOAT lh=sqrt(SQR(ix+ox)+SQR(iy+oy)+SQR(iz+oz));
	BRDF_FLOAT hx=(ix+ox)/lh;
	BRDF_FLOAT hy=(iy+oy)/lh;
	BRDF_FLOAT hz=(iz+oz)/lh;

	BRDF_FLOAT thM=MIN(acos(h),(d<2.0?asin(d/2.0):PI/2.0));
	BRDF_FLOAT f=MAX(d-2.0*sqrt(1.0-SQR(h)),0.0);

	BRDF_FLOAT thL=acos(sqrt(1.0/(SQR(iy)/SQR(iz)+1.0)));
	//BRDF_FLOAT thL=acos(iz/sqrt(SQR(iy)+SQR(iz)));
	BRDF_FLOAT thSS=PI/2.0-thL;
	BRDF_FLOAT thS,fi,fs;
	if (thSS > thM) {
		thS = thSS = thM;
		fi=f;
		fs=0;
	} else {
		if ((1.0/cos(thL)) < (d-1.0))
			thS=PI/2.0;
		else
			thS=thL+asin(d*cos(thL)-1.0);
		thS=MIN(thS,thM);
		fs=MIN(1.0/cos(thL)-sqrt(1.0-SQR(h))-h*tan(thL),f);
		fs=MAX(fs,0.0);
		fi=f-fs;
	}

	BRDF_FLOAT thE=acos(sqrt(1.0/(SQR(oy)/SQR(oz)+1.0)));
	//BRDF_FLOAT thE=acos(oz/sqrt(SQR(oy)+SQR(oz)));
	BRDF_FLOAT thSH=PI/2.0-thE;
	BRDF_FLOAT thH,fv,fh;
	if (thSH > thM) {
		thH = thSH = thM;
		fv=f;
		fh=0;
	} else {
		if ((1.0/cos(thE)) < (d-1.0))
			thH=PI/2.0;
		else
			thH=thE+asin(d*cos(thE)-1.0);
		thH=MIN(thH,thM);
		fh=MIN(1.0/cos(thE)-sqrt(1.0-SQR(h))-h*tan(thE),f);
		fh=MAX(fh,0.0);
		fv=f-fh;
	}

	BRDF_FLOAT thl,the,lv,lvi,fvi,phE,phs,phi;
	lv=(sin(thH-thE)+sin(thSH+thE))/cos(thE);
	if ((iy > 0) == (oy > 0)) {
		// L and E on the same side of N
		thl=MIN(thS,thH);
		the=MIN(thSS,thSH);
		lvi=(sin(thl-thE)+sin(the+thE))/cos(thE);
		fvi=MIN(fv,fi);
		phE=PI/2.0-thE;
	} else {
		thl=MIN(thS,thSH);
		the=MIN(thH,thSS);
		lvi=(sin(the-thE)+sin(thl+thE))/cos(thE);
		fvi=MAX(f-fh-fs,0.0);
		phE=PI/2.0+thE;
	}
	phi=PI/2.0-thl;
	phs=PI/2.0+the;
	
	BRDF_FLOAT Ln=iz, Lb=ABS(iy);
	/*
	BRDF_FLOAT diffterm1=cos(phE)*(SQR(sin(phs))-SQR(sin(phi)))+
			     sin(phE)*(phs-phi-sin(phs)*cos(phs)+sin(phi)*cos(phi));
	BRDF_FLOAT diffterm2=sin(phE)*(SQR(sin(phs))-SQR(sin(phi)))+
			     cos(phE)*(phs-phi+sin(phs)*cos(phs)+sin(phi)*cos(phi));
	BRDF_FLOAT diffnorm=cos(phE)*(sin(phs)-sin(phi))+
			    sin(phE)*(cos(phi)-cos(phs));
	*/
	BRDF_FLOAT diffterm1=sin(phE)*(phs-phi)+0.5*(cos(phE-phi-phi)-cos(phE-phs-phs));
	BRDF_FLOAT diffterm2=cos(phE)*(phs-phi)+0.5*(sin(phE-phi-phi)-sin(phE-phs-phs));
	BRDF_FLOAT correction=sin(phE-phi)-sin(phE-phs);
	BRDF_FLOAT diffcyl=(diffterm1*Ln/2.0+diffterm2*Lb/2.0)/correction;
	BRDF_FLOAT difffloor=Ln;
	BRDF_FLOAT totaldiff=rd*(lvi*diffcyl+fvi*difffloor)/(lv+fv);

	BRDF_FLOAT Hn=hz, Hb=(iy>0)?hy:-hy;
	BRDF_FLOAT sphE=sin(phE), cphE=cos(phE);
	BRDF_FLOAT sd=MAX(2.0,sqrt(n)/1.2);
	sd*=MAX(0.2,phs-phi);
	int subdiv=MAX(2,(int)sd);
	subdiv*=2;
	BRDF_FLOAT dph=(phs-phi)/subdiv;
	BRDF_FLOAT sdph=sin(dph), cdph=cos(dph);
	BRDF_FLOAT sph=sin(phi),cph=cos(phi);
	BRDF_FLOAT s=0,sc=0;
	int i;
	for (i=0;i<=subdiv;i++) {
		BRDF_FLOAT coeff;
		if ((i==0) || (i==subdiv))
			coeff=1;
		else if (i%2)
			coeff=4;
		else
			coeff=2;
		sc+=coeff;
		BRDF_FLOAT NdotH=Hn*sph+Hb*cph;
		BRDF_FLOAT NdotL=Ln*sph+Lb*cph;
		s+=coeff*(sph*sphE+cph*cphE)*NdotL*
			NdotH/(n-n*NdotH+NdotH);
		BRDF_FLOAT nsph=sdph*cph+cdph*sph;
		BRDF_FLOAT ncph=cdph*cph-sdph*sph;
		sph=nsph; cph=ncph;
	}
	BRDF_FLOAT speccyl=s/sc*(phs-phi)/correction;
	BRDF_FLOAT specfloor=Ln*Hn/(n-n*Hn+Hn);
	BRDF_FLOAT totalspec=rs*(lvi*speccyl+fvi*specfloor)/(lv+fv);

	return (totaldiff+totalspec) /iz;
}

