/*
Szymon Rusinkiewicz

brdf_cos.cc
Lafortune's generalized cosine lobe model.

Note that this is only one lobe - to get what's described in the paper you'll
need to take the sum of several of these, plus a diffuse term.
We implement only the Cx=Cy case, since it's not obvious that the anisotropy
introduced by the Cx != Cy case is at all useful for "real" BRDFs.

Lafortune, Eric P.F.,  Foo, Sing-Choong,  Torrance, Kenneth E.,
and Greenberg, Donald P.  "Non-Linear Approximation of Reflectance Functions",
_SIGGRAPH_ Proceedings, 1997.
*/

#pragma implementation
#include "brdf_cos.h"

void Cos_BRDF::Usage()
{
	printf(	"cos         Lafortune's generalized cosine lobes\n"
		"            Parameters are: cxy [-1]    Cx, Cy Parameters (Must be equal)\n"
		"                            cz [0.95]   Cz Parameter\n"
		"                            n [20]      Cosine exponent\n"
	      );
}

BRDF *Cos_BRDF::Create(const char *params)
{
	float cxy=-1,cz=0.95,n=20;
	
	if (params)
		sscanf(params,"%f %f %f",&cxy,&cz,&n);

	return new Cos_BRDF(cxy,cz,n);
}

void Cos_BRDF::Getname(char *nm)
{
	sprintf(nm,"Generalized Cosine Lobe  Cxy=%.2f, Cz=%.2f, n=%.2f",
						 cxy,     cz,     n);
}

brdf_return_t Cos_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			      BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	BRDF_FLOAT ix=sin(theta_in)*cos(phi_in);
	BRDF_FLOAT iy=sin(theta_in)*sin(phi_in);
	BRDF_FLOAT iz=CLAMP(cos(theta_in),EPS,1.0);
	BRDF_FLOAT ox=sin(theta_out)*cos(phi_out);
	BRDF_FLOAT oy=sin(theta_out)*sin(phi_out);
	BRDF_FLOAT oz=CLAMP(cos(theta_out),EPS,1.0);

	return pow(MAX(cxy*ix*ox+cxy*iy*oy+cz*iz*oz,0.0),n);
}

