/*
Szymon Rusinkiewicz

brdf_ts.cc
The Beard-Maxwell model
*/

#pragma implementation
#include "brdf_bm.h"

#include "fresnel.h"

void BM_BRDF::Usage()
{
	printf(	"bm          The Beard-Maxwell BRDF\n"
		"            Parameters are: omega [0.4] Surface shadowing parameter\n"
		"                            tau [0.15]  Surface shadowing parameter\n"
		"                            nreal [3.5] Index of refraction (real part)\n"
		"                            nimag [-0.2]                    (imaginary part)\n"
		"                            rs [0.8]    First surface reflectivity\n"
		"                            rd [0.1]    Diffuse reflectivity\n"
		"                            rv [0.1]    Volumetric reflectivity\n"
	      );
}

BRDF *BM_BRDF::Create(const char *params)
{
	float omega=0.4,tau=0.15,nreal=3.5,nimag=-0.2,rs=0.8,rd=0.1,rv=0.1;
	
	if (params)
		sscanf(params,"%f %f %f %f %f %f %f",&omega,&tau,&nreal,&nimag,&rs,&rd,&rv);

	return new BM_BRDF(omega,tau,nreal,nimag,rs,rd,rv);
}

void BM_BRDF::Getname(char *nm)
{
	sprintf(nm,"Beard-Maxwell  omega=%.2f, tau=%.2f, n=%.2f%+.2fi, rs=%.2f, rd=%.2f, rv=%.2f",
					 omega,    tau,  real(n),imag(n), rs,      rd,      rv);
}

brdf_return_t BM_BRDF::Eval(BRDF_FLOAT theta_in, BRDF_FLOAT phi_in,
			    BRDF_FLOAT theta_out, BRDF_FLOAT phi_out)
{
	/* Beta is the angle between the normal and the halfway vector */
	BRDF_FLOAT ix=sin(theta_in)*cos(phi_in);
	BRDF_FLOAT iy=sin(theta_in)*sin(phi_in);
	BRDF_FLOAT iz=cos(theta_in);
	BRDF_FLOAT ox=sin(theta_out)*cos(phi_out);
	BRDF_FLOAT oy=sin(theta_out)*sin(phi_out);
	BRDF_FLOAT oz=cos(theta_out);
	BRDF_FLOAT lh=sqrt(SQR(ix+ox)+SQR(iy+oy)+SQR(iz+oz));
	BRDF_FLOAT hx=(ix+ox)/lh;
	BRDF_FLOAT hy=(iy+oy)/lh;
	BRDF_FLOAT hz=(iz+oz)/lh;
	BRDF_FLOAT cosbeta=CLAMP(hz,EPS,1.0-EPS);
	BRDF_FLOAT beta=acos(cosbeta);
	BRDF_FLOAT c=CLAMP(ix*hx+iy*hy+iz*hz,EPS,1.0);

	BRDF_FLOAT SO=(1.0+beta/omega*exp(-2.0*acos(c)/tau))/(1.0+beta/omega);
	BRDF_FLOAT F=Fresnel_term(c,n);

	return (rd+2.0*rv/MAX(iz+oz,EPS)+rs*SQR(cosbeta)*SO*F/iz/oz)/PI;
}

