/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include <stdio.h>

#if defined(AIX)
#include <termios.h>
#define VWERASE VWERSE
#define VDISCARD VDISCRD
#endif

#include "__utilsource.h"
#include "wiregl/include/wiregl_util.h"
#include "wiregl/include/wiregl_serial.h"

#ifdef _WIN32

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

WireGLSerialPort
UTIL_DECL wireGLSerialOpen( const char *name, int mode, int baud )
{
	HANDLE hComm;
	DCB	   dcb;
	COMMTIMEOUTS timeouts;
	DWORD  flags;

	flags = 0;
	switch ( mode & ( WIREGL_SERIAL_READ | WIREGL_SERIAL_WRITE ) )
	{
	  case WIREGL_SERIAL_READ | WIREGL_SERIAL_WRITE:
		flags |= GENERIC_READ | GENERIC_WRITE;
		break;
	  case WIREGL_SERIAL_READ:
		flags |= GENERIC_READ;
		break;
	  case WIREGL_SERIAL_WRITE:
		flags |= GENERIC_WRITE;
		break;
	  default:
		wireGLError( "wireGLSerialOpen: mode=%x", mode );
	}

	/* call to CreateFile Opens Port use non-Overlapped I/O */
	hComm = CreateFile( name,  
						flags,
						0, 
						0, 
						OPEN_EXISTING,
						0,
						0 );

	if ( hComm == INVALID_HANDLE_VALUE )
	{
		wireGLSimpleError( "wireGLSerialOpen( \"%s\", mode=%d, baud=%d ) : "
						   "error", name, mode, baud );
	}

	memset( &dcb, 0, sizeof(dcb) );
	
	/* get current DCB structure */
	GetCommState(hComm, &dcb);

	/* setup DCB structure */

	switch ( baud )
	{
	  case 9600:
		dcb.BaudRate = CBR_9600;
		break;

	  case 19200:
		dcb.BaudRate = CBR_19200;
		break;

	  case 115200:
		dcb.BaudRate = CBR_115200;
		break;

	  default:
		wireGLError( "wireGLSerialOpen: baud=%d?", baud );
	}

	dcb.ByteSize = 8;

	/* specifiy if read/write ops are terminated if an error occurs */
	dcb.fAbortOnError = 0;

	/* only binary xfers are supported */
	dcb.fBinary = 0;

	/* sensitive to DSR signal? */
	dcb.fDsrSensitivity = 0;

	/* dtr flow control...disable */
	dcb.fDtrControl = DTR_CONTROL_DISABLE;

	/* XON/XOFF input flow control..disable  */
	dcb.fInX = 0;

	/* XON/XOFF output flow control..disable */
	dcb.fOutX = 0;

	/* parity checking...off */
	dcb.fParity = 0;

	/* transmiit CTR/DSR flow control...off */
	dcb.fOutxCtsFlow = 0;
	dcb.fOutxDsrFlow = 0;

	/* disable "ready_to_send" flow control */
	dcb.fRtsControl = RTS_CONTROL_DISABLE;

	/* set the comm state with the DCB structure */
	SetCommState( hComm, &dcb );
	
	/* purge the buffer */
	PurgeComm( hComm, PURGE_RXCLEAR );

	/* get the current timeout values and modify */
	GetCommTimeouts( hComm, &timeouts );
	
	/* return immediately with characters already received, even if
	 * none have been received */
	timeouts.ReadIntervalTimeout = MAXDWORD;
	timeouts.ReadTotalTimeoutConstant = 0;
	timeouts.ReadTotalTimeoutMultiplier = 0;

	/* set the timeouts */
	SetCommTimeouts(hComm, &timeouts);

	/* return the comm device handle */
	return hComm;
}

UTIL_DECL int
wireGLSerialRead( WireGLSerialPort port, void *buf, int n )
{
	DWORD bytesRead;

	/* attempt to read and check for errors */
	if ( !ReadFile( port, buf, n, &bytesRead, NULL ) )
	{
		wireGLWarning( WIREGL_WARN_CRITICAL, "wireGLSerialRead: error %d",
					   GetLastError( ) );
	}

#if 0
	/* purge the buffer */
	PurgeComm(hComm, PURGE_RXCLEAR);
#endif

	return (int) bytesRead;
}

UTIL_DECL int
wireGLSerialWrite( WireGLSerialPort port, void *buf, int n )
{
	DWORD bytesWritten;

	/* attempt write and check for errors */
	if ( !WriteFile( port, buf, n, &bytesWritten, NULL ) )
	{
		wireGLWarning( WIREGL_WARN_CRITICAL, "wireGLSerialWrite: error %d",
					   GetLastError( ) );
	}
	
	return (int) bytesWritten;
}

UTIL_DECL void 
wireGLSerialShowState( WireGLSerialPort port )
{
	/* create DCB structure for holding comm state information */
	DCB dcb;

	memset( &dcb, 0, sizeof(dcb) );
	GetCommState( port, &dcb );

	/* print contents of DCB structure */
	printf("baud rate: %d\n",dcb.BaudRate);
	printf("binary mode?: %d\n",dcb.fBinary);
	printf("parity on?: %d\n",dcb.fParity);
	printf("CTS output flow control?: %d\n",dcb.fOutxCtsFlow);
	printf("DSR output flow control?: %d\n",dcb.fOutxDsrFlow);
	printf("DTR flow control type: %d\n",dcb.fDtrControl);
	printf("attentive to DSR?: %d\n",dcb.fDsrSensitivity);
	printf("XON/XOFF receive flow control?: %d\n",dcb.fInX);
	printf("XON/XOFF transmit flow control?: %d\n",dcb.fOutX);
	
	if (dcb.Parity == EVENPARITY)
		printf("parity scheme: EVEN\n");
	else if (dcb.Parity == ODDPARITY)
		printf("parity scheme: ODD\n");
	else if (dcb.Parity == MARKPARITY)
		printf("parity scheme: MARK\n");
	else if (dcb.Parity == SPACEPARITY)
		printf("parity scheme: SPACE\n");
	else if (dcb.Parity == NOPARITY)
		printf("parity scheme: NO PARITY\n");

	if (dcb.Parity == ONESTOPBIT)
		printf("# of stop bits: 1\n");
	else if (dcb.Parity == ONE5STOPBITS)
		printf("# of stop bits: 1.5\n");
	else if (dcb.Parity == TWOSTOPBITS)
		printf("# of stop bits: 2\n");
	printf("\n");	
}

#else

#include <unistd.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <errno.h>

WireGLSerialPort
wireGLSerialOpen( const char *name, int mode, int baud )
{
	int fh;
	int flags;
    struct termios tios;

	flags = O_NOCTTY | O_NDELAY;
	switch ( mode & ( WIREGL_SERIAL_READ | WIREGL_SERIAL_WRITE ) )
	{
	  case WIREGL_SERIAL_READ | WIREGL_SERIAL_WRITE:
		flags |= O_RDWR;
		break;
	  case WIREGL_SERIAL_READ:
		flags |= O_RDONLY;
		break;
	  case WIREGL_SERIAL_WRITE:
		flags |= O_WRONLY;
		break;
	  default:
		wireGLError( "wireGLSerialOpen: mode=%x", mode );
	}

	/* open the device */
	/* O_NDELAY -- ignore the DCD signal */
	fh = open( name, flags );
	if ( fh < 0 )
	{
		wireGLSimpleError( "wireGLSerialOpen( \"%s\", mode=%d, baud=%d ) : %s",
						   name, mode, baud, strerror(errno) );
	}

	if ( tcgetattr( fh, &tios ) )
	{
		wireGLSimpleError( "wireGLSerialOpen: tcgetattr failed (%s)",
						   strerror( errno ) );
	}

	/* set the baud, flow control, etc */
    tios.c_iflag        = IGNPAR|IGNBRK|IXON|IXOFF;
    tios.c_oflag        = 0;
    tios.c_lflag        = NOFLSH;
    tios.c_cflag        = CS8|CREAD|CLOCAL;
	tios.c_cc[VINTR]    = 0;     /* Ctrl-c */ 
    tios.c_cc[VQUIT]    = 0;     /* Ctrl-\ */
    tios.c_cc[VERASE]   = 0;     /* del */
    tios.c_cc[VKILL]    = 0;     /* @ */
    tios.c_cc[VEOF]     = 4;     /* Ctrl-d */
    tios.c_cc[VTIME]    = 0;     /* inter-character timer unused */
    tios.c_cc[VMIN]     = 0;     /* non-blocking reads */
#ifdef VSWTC
    tios.c_cc[VSWTC]    = 0;     /* '\0' */
#endif
    tios.c_cc[VSTART]   = 0;     /* Ctrl-q */ 
    tios.c_cc[VSTOP]    = 0;     /* Ctrl-s */
    tios.c_cc[VSUSP]    = 0;     /* Ctrl-z */
    tios.c_cc[VEOL]     = 0;     /* '\0' */
    tios.c_cc[VREPRINT] = 0;     /* Ctrl-r */
    tios.c_cc[VDISCARD] = 0;     /* Ctrl-u */
    tios.c_cc[VWERASE]  = 0;     /* Ctrl-w */
    tios.c_cc[VLNEXT]   = 0;     /* Ctrl-v */
    tios.c_cc[VEOL2]    = 0;     /* '\0' */

	switch ( baud )
	{
	  case 9600:
		tios.c_cflag |= B9600;
		break;

	  case 19200:
		tios.c_cflag |= B19200;
		break;

#ifndef AIX
	  case 115200:
		tios.c_cflag |= B115200;
		break;
#endif

	  default:
		wireGLError( "wireGLSerialOpen: bad baud rate %d", baud );
	}

	/* clean the line and apply the settings */
	if ( tcflush( fh, TCIFLUSH ) != 0 )
	{
		wireGLSimpleError( "wireGLSerialOpen: tcflush failed (%s)",
						   strerror( errno ) );
	}
		 
    if ( tcsetattr( fh, TCSANOW, &tios ) )
	{
		wireGLSimpleError( "wireGLSerialOpen: tcsetattr failed (%s)",
						   strerror( errno ) );
	}

#if defined(Linux) || defined(AIX)
	if ( mode & WIREGL_SERIAL_WRITE )
	{
		/* this is documented behavior -- if you turn off ICANON on
		 * the serial port, you need to turn it off on stdout as well
		 * */
		tcgetattr( 1, &tios );
		tios.c_lflag &= ~ICANON;
		tcsetattr( 1, TCSANOW, &tios );
	}
#endif

	return fh;
}

int
wireGLSerialRead( WireGLSerialPort port, void *buf, int n )
{
	ssize_t retval;

	retval = read( port, buf, n );
	if ( retval < 0 )
	{
		if ( errno != EINTR )
		{
			wireGLWarning( WIREGL_WARN_NOTICE, "wireGLSerialRead: %s",
						   strerror( errno ) );
		}
		retval = 0;
	}

	return (int) retval;
}

int
wireGLSerialWrite( WireGLSerialPort port, void *buf, int n )
{
	ssize_t retval;

	retval = write( port, buf, n );
	if ( retval < 0 )
	{
		wireGLWarning( WIREGL_WARN_NOTICE, "wireGLSerialRead: %s",
					   strerror( errno ) );
		retval = 0;
	}

	return (int) retval;
}

void 
wireGLSerialShowState( WireGLSerialPort port )
{
	WIREGL_UNUSED(port);
}

#endif /* _WIN32 */
