/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "__utilsource.h"

#include "wiregl/include/wiregl_util.h"
#include <errno.h>

/*** start of header ***/

typedef struct {
#if defined(WINDOWS)
        HANDLE sema;
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
        pthread_cond_t cond;
        pthread_mutex_t mutex;
        int counter;
#else
#error WIRE GL SEMAPHORE
#endif
} WireGLSemaphore;

#if defined(WINDOWS)
typedef volatile LONG WireGLSpinLock;
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
typedef pthread_mutex_t WireGLSpinLock;
#else
#error SPINLOCKS
#endif

#if defined(WINDOWS)
typedef HANDLE WireGLLock;
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
typedef pthread_mutex_t WireGLLock;
#else
#error LOCKS
#endif

typedef struct {
#if defined(WINDOWS)
        HANDLE handle;
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
        pthread_cond_t cond;
        pthread_mutex_t mutex;
        volatile int flag;
#else
#error WIRE GL EVENT
#endif
} WireGLEvent;

UTIL_DECL void wireGLCheckForBrokenPthreadMutexes( void );

extern WireGLSpinLock __wireGLOutputSema;
UTIL_DECL void wireGLInitSemaphore( WireGLSemaphore *m, int value );
UTIL_DECL void wireGLAcquireSemaphore( WireGLSemaphore *m );
UTIL_DECL void wireGLReleaseSemaphore( WireGLSemaphore *m );
UTIL_DECL void wireGLDestroySemaphore( WireGLSemaphore *m );

UTIL_DECL void wireGLInitSpinLock( WireGLSpinLock *sl );
UTIL_DECL void wireGLAcquireSpinLock( WireGLSpinLock *sl );
UTIL_DECL void wireGLReleaseSpinLock( WireGLSpinLock *sl );

UTIL_DECL void wireGLInitLock( WireGLLock *l );
UTIL_DECL void wireGLAcquireLock( WireGLLock *l );
UTIL_DECL void wireGLReleaseLock( WireGLLock *l );
UTIL_DECL void wireGLDestroyLock( WireGLLock *l );

UTIL_DECL void wireGLInitEvent( WireGLEvent *ev );
UTIL_DECL void wireGLWaitEvent( WireGLEvent *ev );
UTIL_DECL void wireGLSignalEvent( WireGLEvent *ev );

/*** end of header ***/

void
wireGLCheckForBrokenPthreadMutexes( void )
{
#if defined(LINUX)
	/* Some versions of the pthreads library on Linux have bad lock
       behavior when transitively linked (that is, the executable
       doesn't link against pthreads, but it is dynamically linked at
       runtime because a shared object uses pthreads).  This test will
       try to detect some of that lossage. */
	pthread_mutex_t mutex;

	memset( &mutex, 0, sizeof(mutex) );
	pthread_mutex_init( &mutex, NULL );

	pthread_mutex_lock( &mutex );
	if ( mutex.__m_lock.__status == 0 )
	{
		wireGLSimpleError( "pthread_mutex_lock is busted, sorry." );
	}
	pthread_mutex_unlock( &mutex );
	if ( mutex.__m_lock.__status != 0 )
	{
		wireGLSimpleError( "pthread_mutex_unlock is busted, sorry." );
	}

	while ( pthread_mutex_trylock( &mutex ) == EBUSY) 
		; /* EMPTY */
	if ( mutex.__m_lock.__status == 0 )
	{
		wireGLSimpleError( "pthread_mutex_trylock is busted, sorry." );
	}

	pthread_mutex_unlock( &mutex );
	if ( mutex.__m_lock.__status != 0 )
	{
		wireGLSimpleError( "pthread_mutex_unlock is busted, sorry." );
	}
#endif
}

UTIL_DECL void wireGLInitSemaphore( WireGLSemaphore *s, int value )
{
#if defined(WINDOWS)
	s->sema = CreateSemaphore( NULL, value, 1000000, NULL );

	if (!s->sema)
		wireGLError( "Couldn't allocate a semaphore" );
#elif defined(Linux) || defined(IRIX) || defined(IRIX64)
	s->counter = value;
	pthread_mutex_init( &(s->mutex), NULL );
	pthread_cond_init( &(s->cond), NULL );
#else
#error wireGLInitSemaphore
#endif
}

UTIL_DECL void wireGLAcquireSemaphore( WireGLSemaphore *s )
{
#if defined (WINDOWS)
	DWORD status = WaitForSingleObject(s->sema,INFINITE);

	/* Do I need to check for any other status's for failure? */
	if (status == WAIT_FAILED) {
		wireGLError("Error acquiring a semaphore");
	}
#elif defined(Linux) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_lock( &(s->mutex) );
	while (!(s->counter))
		pthread_cond_wait( &(s->cond), &(s->mutex) );
	s->counter--;
	pthread_mutex_unlock( &(s->mutex) );
#else
#error wireGLAcquireSemaphore
#endif
}

UTIL_DECL void wireGLReleaseSemaphore( WireGLSemaphore *s )
{
#if defined (WINDOWS)
	if ( !ReleaseSemaphore( s->sema, 1, NULL ) )
	{
		wireGLError("Error releasing a semaphore:%d", GetLastError( ) );
	}
#elif defined (Linux) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_lock( &(s->mutex) );
	s->counter++;
	pthread_mutex_unlock( &(s->mutex) );
	pthread_cond_signal( &(s->cond) );
#else
#error wireGLReleaseSemaphore
#endif
}

UTIL_DECL void wireGLDestroySemaphore( WireGLSemaphore *s )
{
#if defined (WINDOWS)
	CloseHandle(s->sema);
#elif defined (Linux) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_destroy( &(s->mutex) );
	pthread_cond_destroy( &(s->cond) );
#else
#error wireGLDestorySemaphore
#endif
}

UTIL_DECL void wireGLInitLock( WireGLLock *l )
{
#if defined(WINDOWS)
	*l = CreateSemaphore( NULL, 1, 1, NULL );

	if ( *l == NULL )
	{
		wireGLError( "Couldn't allocate a semaphore" );
	}
#elif defined(Linux) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_init( l, NULL );
#else
#error wireGLInitLock
#endif
}

UTIL_DECL void wireGLAcquireLock( WireGLLock *l )
{
#if defined (WINDOWS)
	DWORD status = WaitForSingleObject( *l, INFINITE );

	/* Do I need to check for any other status's for failure? */
	if ( status == WAIT_FAILED ) 
	{
		wireGLError("Error acquiring a lock");
	}
#elif defined(Linux) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_lock( l );
#else
#error wireGLAcquireLock
#endif
}

UTIL_DECL void wireGLReleaseLock( WireGLLock *l )
{
#if defined (WINDOWS)
	if ( !ReleaseSemaphore( *l, 1, NULL) )
	{
		wireGLError("Error releasing a lock:%d", GetLastError( ) );
	}
#elif defined (Linux) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_unlock( l );
#else
#error wireGLReleaseLock
#endif
}

UTIL_DECL void wireGLDestroyLock( WireGLLock *l )
{
#if defined (WINDOWS)
	CloseHandle( *l );
#elif defined (Linux) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_destroy( l );
#else
#error wireGLDestoryLock
#endif
}

UTIL_DECL void wireGLInitSpinLock( WireGLSpinLock *sl )
{
#if defined(WINDOWS)
	*sl = 0;
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_init( sl, NULL );
#else
#error wireGLInitSpinLock
#endif
}

UTIL_DECL void wireGLAcquireSpinLock( WireGLSpinLock *sl )
{
#if defined(WINDOWS)
	LONG result = InterlockedExchange( (long *) sl, 1 );
	while (result)
	{
		while (*sl)
			;
		result = InterlockedExchange( (long *) sl, 1 );
	}
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
	while (pthread_mutex_trylock( sl ) == EBUSY) 
		; /* EMPTY */
#else
#error wireGLAcquireSpinLock
#endif
}

UTIL_DECL void wireGLReleaseSpinLock( WireGLSpinLock *sl )
{
#if defined(WINDOWS)
	*sl = 0;
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_unlock( sl );
#else
#error wireGLReleaseSpinLock
#endif
}

UTIL_DECL void wireGLInitEvent( WireGLEvent *ev )
{
#if defined(WINDOWS)
	ev->handle = CreateEvent( NULL, TRUE, FALSE, NULL );
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_init( &(ev->mutex), NULL );
	pthread_cond_init( &(ev->cond), NULL );
	ev->flag = 0;
#else
#error wireGLInitEvent
#endif
} 

UTIL_DECL void wireGLWaitEvent( WireGLEvent *ev )
{
#if defined(WINDOWS)
	DWORD status = WaitForSingleObject(ev->handle, INFINITE);

	/* Do I need to check for any other status's for failure? */
	if (status == WAIT_FAILED) {
		wireGLError("Error acquiring a condition variable");
	}
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_lock(&(ev->mutex));
	if (!(ev->flag))
	{
		pthread_cond_wait(&(ev->cond), &(ev->mutex));
	}
	ev->flag = 0;
	pthread_mutex_unlock(&(ev->mutex));
#else
#error wireGLWaitEvent
#endif
} 

UTIL_DECL void wireGLSignalEvent( WireGLEvent *ev )
{
#if defined(WINDOWS)
	BOOL status = SetEvent( ev->handle );
	if (!status)
	{
		wireGLError("Error Signaling a condition variable");
	}
#elif defined(LINUX) || defined(IRIX) || defined(IRIX64)
	pthread_mutex_lock(&(ev->mutex));
	ev->flag = 1;
	pthread_mutex_unlock(&(ev->mutex));
	pthread_cond_signal(&(ev->cond));
#else
#error wireGLSignalEvent
#endif
} 

UTIL_DECL void 
wireGLCreateThread( WireGLThreadFunc entry, void *arg )
{
#if defined(WINDOWS)
	u_long id;

	if (CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE) entry, (LPVOID) arg, 0, &id) == NULL)
		wireGLError ("CreateThread failure on WINDOWS");
#elif defined(Linux) || defined(IRIX) || defined(IRIX64)
	pthread_t newThreadID;
	if(pthread_create(&newThreadID, NULL, (void *(*)(void *))entry,
				arg) !=0)
	{
		wireGLError ("pthread_create failure on Linux");
	}
#else
#error wireGLCreateThread
#endif
}
