/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/
/* util/gm.c */

#include "__utilsource.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <gm.h>
#ifndef WINDOWS
#include <ctype.h> // for tolower
#endif

#include "wiregl/include/wiregl_util.h"
#include "wiregl/include/wiregl_common.h"
#include "wiregl/include/wiregl_protocol.h"

#define WIREGL_GM_USE_CREDITS 1
#define WIREGL_GM_SEND_CREDITS_THRESHOLD ( 1 << 18 )

/* Versions of GM prior to 1.4pre37 have a bug involving short sends
 * that straddle a page boundary.  Set WIREGL_GM_BROKEN_SHORT_SENDS to
 * 1 to enable a work-around. */

#define WIREGL_GM_BROKEN_SHORT_SENDS 0

#if WIREGL_GM_BROKEN_SHORT_SENDS
#define WIREGL_GM_PAGE_SIZE     4096
#define WIREGL_GM_PAGE_ALIGN(x) ((void *) (((unsigned long) (x) + WIREGL_GM_PAGE_SIZE - 1) & ~(WIREGL_GM_PAGE_SIZE-1)) )
#define WIREGL_GM_BUFFER_SPANS_PAGE_BOUNDARY(x,l)	(((unsigned long) (x) & (WIREGL_GM_PAGE_SIZE-1)) + (l) > WIREGL_GM_PAGE_SIZE)
#endif

#ifndef _GM_NTOH
#define gm_ntoh_u32(x)	ntohl(x)
#define gm_ntoh_u16(x)	ntohs(x)
#define gm_ntoh_u8(x)	(x)
#endif

#define WIREGL_GM_BUFFER_SEND_MAGIC 0x95718def
#define WIREGL_GM_BUFFER_RECV_MAGIC 0x97518abc
#define WIREGL_GM_SECONDS_PER_TICK  ( 5e-7 )


typedef	enum { 
	WireGLGmMemoryPinned,
	WireGLGmMemoryUnpinned,
	WireGLGmMemoryBig 
} WireGLGmBufferKind;

typedef struct WireGLGmBuffer {
	unsigned int       magic;
	WireGLGmBufferKind kind;
	unsigned int       len;
	unsigned int       pad;
} WireGLGmBuffer;

typedef struct WireGLMultiBuffer {
	unsigned int  len;
	unsigned int  max;
	void         *buf;
} WireGLMultiBuffer;

typedef struct WireGLGmConnection {
	unsigned int       		   node_id;
	unsigned int       		   port_num;
	WireGLConnection   		  *conn;
	WireGLMultiBuffer  		   multi;
	struct WireGLGmConnection *hash_next;
	struct WireGLGmConnection *credit_prev;
	struct WireGLGmConnection *credit_next;
} WireGLGmConnection;

/* Use a power of 2 to the lookup is just a mask -- this works fine
 * because we expect the gm node numbering to be dense */
#define WIREGL_GM_CONN_HASH_SIZE	64

static struct {
	unsigned int          initialized;
	struct gm_port       *port;
	unsigned int          node_id;
	unsigned int          port_num;
	unsigned int          num_nodes;
	WireGLGmConnection   *gm_conn_hash[WIREGL_GM_CONN_HASH_SIZE];
	WireGLGmConnection   *credit_head;
	WireGLGmConnection   *credit_tail;
	unsigned int          message_size;
	WireGLNetReceiveFunc  recv;
	WireGLNetCloseFunc    close;
	WireGLBufferPool      read_pool;
	WireGLBufferPool      write_pool;
	WireGLBufferPool      unpinned_read_pool;
	unsigned int          num_outstanding_sends;
	unsigned int          num_send_tokens;
	unsigned int          inside_recv;
} wiregl_gm;

#define WIREGL_GM_PINNED_READ_MEMORY_SIZE   ( 8 << 20 )
#define WIREGL_GM_PINNED_WRITE_MEMORY_SIZE  ( 8 << 20 )

#define WIREGL_GM_API_VERSION          GM_API_VERSION_1_1
#define WIREGL_GM_PRIORITY             GM_LOW_PRIORITY

#define WIREGL_GM_ROUND_UP             1

#define WIREGL_GM_CREDITS_DEBUG 	   0

#define WIREGL_GM_DEBUG                0
#define WIREGL_GM_SYNCHRONOUS          0
#define WIREGL_GM_FENCE_BUFFERS        0
#define WIREGL_GM_FENCE_VALUE          0xdeadbeef
#define WIREGL_GM_PARANOIA             0

UTIL_DECL void 
wireGLGmSend( WireGLConnection *conn, void **bufp, 
			  void *start, unsigned int len );

#if WIREGL_GM_DEBUG || WIREGL_GM_CREDITS_DEBUG

static void
wiregl_gm_debug( const char *format, ... )
{
	char txt[8192];
	va_list args;
	va_start( args, format );
	vsprintf( txt, format, args );
	va_end( args );

	wireGLWarning( WIREGL_WARN_DEBUG, "%d%*cGM: %s", wiregl_gm.inside_recv,
				   wiregl_gm.inside_recv * 2, ' ', txt );
}

#endif

static void 
wiregl_gm_info( void )
{
    char name[ GM_MAX_HOST_NAME_LEN ];
    unsigned int node_id, max_node_id, min_size;

    gm_get_host_name( wiregl_gm.port, name );
    gm_get_node_id( wiregl_gm.port, &node_id );
    wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: name=\"%s\" id=%u port=%u",
                   name, node_id, wiregl_gm.port_num );

    gm_max_node_id_inuse( wiregl_gm.port, &max_node_id );
    wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: gm_max_node_id_inuse=%u",
                   max_node_id );

    min_size = gm_min_size_for_length( __wiregl_max_send );
    wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: gm_min_size_for_length( "
                   "__wiregl_max_send=%u ) = %u", __wiregl_max_send,
                   min_size );

    wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: gm_max_length_for_size( "
                   "size=%u ) = %u", min_size,
                   gm_max_length_for_size( min_size ) );

	wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: gm_min_message_size() = %u",
				   gm_min_message_size( wiregl_gm.port ) );

    wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: num_send_tokens=%u "
                   "num_receive_tokens=%u",
				   gm_num_send_tokens( wiregl_gm.port ),
                   gm_num_receive_tokens( wiregl_gm.port ) );
}

#if WIREGL_GM_DEBUG

static const char *
wiregl_gm_str_event_type( gm_recv_event_t *event )
{
	const char *description;

#define CASE(error) case error : description = # error; break

	switch ( GM_RECV_EVENT_TYPE(event) )
	{
	  CASE(GM_NO_RECV_EVENT);
	  CASE(GM_ALARM_EVENT);
	  CASE(GM_SENT_EVENT);
	  CASE(_GM_SLEEP_EVENT);
	  CASE(GM_RAW_RECV_EVENT);
	  CASE(GM_RECV_EVENT);
	  CASE(GM_HIGH_RECV_EVENT);
	  CASE(GM_PEER_RECV_EVENT);
	  CASE(GM_HIGH_PEER_RECV_EVENT);
	  CASE(_GM_DIRECTED_SEND_NOTIFICATION_EVENT);
	  CASE(_GM_FLUSHED_ALARM_EVENT);
	  CASE(GM_SENT_TOKENS_EVENT);
	  CASE(GM_IGNORE_RECV_EVENT);
	  CASE(GM_FAST_RECV_EVENT);
	  CASE(GM_FAST_HIGH_RECV_EVENT);
	  CASE(GM_FAST_PEER_RECV_EVENT);
	  CASE(GM_FAST_HIGH_PEER_RECV_EVENT);
	  CASE(GM_SENDS_FAILED_EVENT);
	  CASE(GM_BAD_SEND_DETECTED_EVENT);
	  CASE(GM_SEND_TOKEN_VIOLATION_EVENT);
	  CASE(GM_RECV_TOKEN_VIOLATION_EVENT);
	  CASE(GM_BAD_RECV_TOKEN_EVENT);
	  CASE(GM_ALARM_VIOLATION_EVENT);
	  CASE(GM_REJECTED_SEND_EVENT);
	  CASE(GM_ORPHANED_SEND_EVENT);
	  CASE(GM_BAD_RESEND_DETECTED_EVENT);
	  CASE(GM_DROPPED_SEND_EVENT);
	  CASE(GM_BAD_SEND_VMA_EVENT);
	  CASE(GM_BAD_RECV_VMA_EVENT);
	  default:
		description = "unknown type";
	}

#undef CASE

	return description;
}

static void 
wiregl_gm_check_recv_event( gm_recv_event_t *event )
{
#if 0
	wireGLWarning( WIREGL_WARN_DEBUG, "GM: gm_receive( type=%u )",
				   GM_RECV_EVENT_TYPE(event) );
#endif
	
	switch ( GM_RECV_EVENT_TYPE(event) )
	{
	  case GM_NO_RECV_EVENT:
	  case GM_ALARM_EVENT:
	  case GM_SENT_EVENT:
	  case _GM_SLEEP_EVENT:
	  case GM_RAW_RECV_EVENT:
	  case GM_RECV_EVENT:
	  case GM_HIGH_RECV_EVENT:
	  case GM_PEER_RECV_EVENT:
	  case GM_HIGH_PEER_RECV_EVENT:
	  case _GM_DIRECTED_SEND_NOTIFICATION_EVENT:
	  case _GM_FLUSHED_ALARM_EVENT:
	  case GM_SENT_TOKENS_EVENT:
	  case GM_IGNORE_RECV_EVENT:
		/* ignore */
		break;

	  case GM_FAST_RECV_EVENT:
	  case GM_FAST_HIGH_RECV_EVENT:
	  case GM_FAST_PEER_RECV_EVENT:
	  case GM_FAST_HIGH_PEER_RECV_EVENT:
#if 1
		wireGLWarning( WIREGL_WARN_DEBUG, "GM fast event (type=%u)",
					   GM_RECV_EVENT_TYPE(event) );
#endif
		break;

	  case GM_SENDS_FAILED_EVENT:
	  case GM_BAD_SEND_DETECTED_EVENT:
	  case GM_SEND_TOKEN_VIOLATION_EVENT:
	  case GM_RECV_TOKEN_VIOLATION_EVENT:
	  case GM_BAD_RECV_TOKEN_EVENT:
	  case GM_ALARM_VIOLATION_EVENT:
	  case GM_REJECTED_SEND_EVENT:
	  case GM_ORPHANED_SEND_EVENT:
	  case GM_BAD_RESEND_DETECTED_EVENT:
	  case GM_DROPPED_SEND_EVENT:
	  case GM_BAD_SEND_VMA_EVENT:
	  case GM_BAD_RECV_VMA_EVENT:
		wireGLWarning( WIREGL_WARN_NOTICE, "GM error event (type=%u)",
					   event->recv.type );
		break;

	  default:
		wireGLWarning( WIREGL_WARN_CRITICAL, "GM unknown event (type=%u)",
					   event->recv.type );
		break;
	}
}

#endif

static const char *
wiregl_gm_str_error( gm_status_t status )
{
    const char *description;

#define CASE(error, desc) case error : description = desc; break

	switch ( status )
	{
		CASE (GM_SUCCESS, "success");
		CASE (GM_FAILURE, "unqualified error");
		CASE (GM_INPUT_BUFFER_TOO_SMALL, "input buffer is too small");
		CASE (GM_OUTPUT_BUFFER_TOO_SMALL, "output buffer is too small");
		CASE (GM_TRY_AGAIN, "try again");
		CASE (GM_BUSY, "busy");
		CASE (GM_MEMORY_FAULT, "memory fault");
		CASE (GM_INTERRUPTED, "interrupted");
		CASE (GM_INVALID_PARAMETER, "invalid parameter");
		CASE (GM_OUT_OF_MEMORY, "out of memory");
		CASE (GM_INVALID_COMMAND, "invalid command");
		CASE (GM_PERMISSION_DENIED, "permission denied");
		CASE (GM_INTERNAL_ERROR, "internal error");
		CASE (GM_UNATTACHED, "unattached");
		CASE (GM_UNSUPPORTED_DEVICE, "unsupported device");
		CASE (GM_SEND_TIMED_OUT, "send timed out");
		CASE (GM_SEND_REJECTED, "send was rejected by receiver");
		CASE (GM_SEND_TARGET_PORT_CLOSED, "target port was closed");
		CASE (GM_SEND_TARGET_NODE_UNREACHABLE, "target node was unreachable");
		CASE (GM_SEND_DROPPED, "send dropped as requested by the client");
		CASE (GM_SEND_PORT_CLOSED, "send port was closed");
		CASE (GM_NODE_ID_NOT_YET_SET,
			  "node ID not set (node has not been configured by the mapper)");
		CASE (GM_STILL_SHUTTING_DOWN,
			  "port is still shutting down from previous open");
		CASE (GM_CLONE_BUSY, "clone device is busy");
	  default:
		description = "unknown error";
	}

#undef CASE

	return description;
}

static double
wiregl_gm_clock( void )
{
#if defined(WINDOWS)
	/* Windows can't convert u64 -> double, only s64 */
	gm_s64_t ticks = (gm_s64_t) gm_ticks( wiregl_gm.port );
#else
	gm_u64_t ticks = gm_ticks( wiregl_gm.port );
#endif
	return WIREGL_GM_SECONDS_PER_TICK * (double) ticks;
}

static void *
wiregl_gm_dma_malloc( unsigned long length )
{
	void *buf;

	buf = gm_dma_malloc( wiregl_gm.port, length );
	if ( buf == NULL )
	{
		wireGLError( "gm_dma_malloc( port=%p, len=%d ) : failed",
					 wiregl_gm.port, length );
	}

#if WIREGL_GM_DEBUG
	wireGLWarning( WIREGL_WARN_DEBUG, "GM: dma_malloc, length=%d "
				   "range=%p:%p", length, buf,
				   (unsigned char*) buf + length );
#endif

	return buf;
}

#if WIREGL_GM_FENCE_BUFFERS

static void
wireGLGmPlaceFence( void *buf )
{
	unsigned int fence = WIREGL_GM_FENCE_VALUE;
	unsigned int len   = __wiregl_max_send;
	memcpy( (char *) buf - sizeof(fence), &fence, sizeof(fence) );
	memcpy( (char *) buf + len, &fence, sizeof(fence) );
}

static void
wireGLGmCheckFence( void *buf )
{
	unsigned int fence = WIREGL_GM_FENCE_VALUE;
	unsigned int len   = __wiregl_max_send;
	if ( memcmp( (char *) buf - sizeof(fence), &fence, sizeof(fence) ) ||
		 memcmp( (char *) buf + len, &fence, sizeof(fence) ) )
	{
		wireGLError( "GM: bad fence value!" );
	}
}

#endif

static void
wireGLGmCreditIncrease( WireGLGmConnection *gm_conn )
{
	WireGLGmConnection *parent;
	int my_credits;

	/* move this node up the doubly-linked, if it isn't already in the
	 * right position */

	parent = gm_conn->credit_prev;
	my_credits = gm_conn->conn->recv_credits;
	if ( parent && parent->conn->recv_credits < my_credits )
	{
		/* we can't be at the head of the list, because our parent is
         * non-NULL */

		/* are we the tail of the list? */
		if ( wiregl_gm.credit_tail == gm_conn )
		{
			/* yes, pull up the tail pointer, and fix our parent to be tail */
			wiregl_gm.credit_tail = parent;
			parent->credit_next = NULL;
		}
		else
		{
			/* no, just link around us */
			parent->credit_next = gm_conn->credit_next;
			gm_conn->credit_next->credit_prev = parent;
		}

		/* walk up the list until we find a guy with more credits than
		 * us or we reach the head */
		while ( parent && parent->conn->recv_credits < my_credits )
		{
			parent = parent->credit_prev;
		}

		/* reinsert us -- we know there is somebody else in the list
         * (and thus we won't be the tail, because we moved up),
         * otherwise we would have been the head, and never bothered
         * trying to move ourself */
		if ( parent )
		{
			WireGLGmConnection *child = parent->credit_next;
			gm_conn->credit_next = child;
			child->credit_prev = gm_conn;

			parent->credit_next = gm_conn;
			gm_conn->credit_prev = parent;
		}
		else
		{
			WireGLGmConnection *child = wiregl_gm.credit_head;
			gm_conn->credit_next = child;
			child->credit_prev = gm_conn;

			wiregl_gm.credit_head = gm_conn;
			gm_conn->credit_prev = NULL;
		}
	}
}

static void
wireGLGmCreditZero( WireGLGmConnection *gm_conn )
{
	wireGLAssert( wiregl_gm.credit_head == gm_conn );

	/* if we aren't already at the tail of the list, */
	if ( wiregl_gm.credit_tail != gm_conn )
	{
		/* then pull us off the head, */
		wiregl_gm.credit_head = gm_conn->credit_next;
		wiregl_gm.credit_head->credit_prev = NULL;

		/* and put us on the tail */
		wiregl_gm.credit_tail->credit_next = gm_conn;
		gm_conn->credit_prev = wiregl_gm.credit_tail;
		gm_conn->credit_next = NULL;

		wiregl_gm.credit_tail = gm_conn;
	}
}

#define WIREGL_GM_HASH(n,p)	wiregl_gm.gm_conn_hash[ (n) & (WIREGL_GM_CONN_HASH_SIZE-1) ]

static __inline WireGLGmConnection *
wireGLGmConnectionLookup( unsigned int node_id, unsigned int port_num )
{
	WireGLGmConnection *gm_conn;

	wireGLAssert( node_id < wiregl_gm.num_nodes );

	gm_conn = WIREGL_GM_HASH( node_id, port_id );
	while ( gm_conn )
	{
		if ( gm_conn->node_id == node_id && gm_conn->port_num == port_num )
		{
			return gm_conn;
		}
		gm_conn = gm_conn->hash_next;
	}
	
	wireGLError( "GM: lookup on unknown source: node=%u port=%u",
				 node_id, port_num );

	/* unreached */
	return NULL;
}

static void
wireGLGmConnectionAdd( WireGLConnection *conn )
{
	WireGLGmConnection *gm_conn, **bucket;

	bucket = &WIREGL_GM_HASH( conn->gm_node_id, conn->gm_port_id );

	for ( gm_conn = *bucket; gm_conn != NULL; gm_conn = gm_conn->hash_next )
	{
		if ( gm_conn->node_id  == conn->gm_node_id && 
			 gm_conn->port_num == conn->gm_port_num )
		{
			wireGLError( "GM: I've already got a connection from node=%u "
						 "port=%u (\"%s\"), so why is it connecting again?",
						 conn->gm_node_id, conn->gm_port_num, conn->hostname );
		}
	}

	gm_conn = wireGLAlloc( sizeof(*gm_conn) );
	gm_conn->node_id   = conn->gm_node_id;
	gm_conn->port_num  = conn->gm_port_num;
	gm_conn->conn      = conn;
	gm_conn->multi.len = 0;
	gm_conn->multi.max = 0;
	gm_conn->multi.buf = NULL;
	gm_conn->hash_next = *bucket;

	*bucket = gm_conn;

	gm_conn->credit_next = NULL;
	if ( wiregl_gm.credit_head )
	{
		gm_conn->credit_prev  = wiregl_gm.credit_tail;
		wiregl_gm.credit_tail->credit_next = gm_conn;
		wiregl_gm.credit_tail = gm_conn;
	}
	else
	{
		gm_conn->credit_prev  = NULL;
		wiregl_gm.credit_head = gm_conn;
		wiregl_gm.credit_tail = gm_conn;
	}

	/* we're on the tail of the list now, so we might need to move
	 * up the list, we certainly can't move down it */
	wireGLGmCreditIncrease( gm_conn );
}

UTIL_DECL void *
wireGLGmAlloc( void )
{
	WireGLGmBuffer *gm_buffer = (WireGLGmBuffer *)
		wireGLBufferPoolPop( &wiregl_gm.write_pool );
	while ( gm_buffer == NULL )
	{
		if ( wiregl_gm.num_outstanding_sends == 0 )
		{
			wireGLError( "wireGLGmAlloc: no available buffers, "
						 "and none outstanding." );
		}
		wireGLGmRecv( );
		gm_buffer = (WireGLGmBuffer *)
			wireGLBufferPoolPop( &wiregl_gm.write_pool );
	}

	return (void *)( gm_buffer + 1 );
}

static void
wiregl_gm_provide_receive_buffer( void *buf )
{
	WireGLMessage *msg = (WireGLMessage *) buf;

	wireGLAssert( ((WireGLGmBuffer *) msg - 1)->magic 
				  == WIREGL_GM_BUFFER_RECV_MAGIC );

#if WIREGL_GM_PARANOIA
	/* HACK, make sure we aren't getting reused memory? */
	memset( buf, 0x33, __wiregl_max_send );
#endif

	msg->type = WIREGL_MESSAGE_ERROR;

	gm_provide_receive_buffer( wiregl_gm.port, buf,
							   wiregl_gm.message_size,
							   WIREGL_GM_PRIORITY );
}

static void
wireGLGmRecvFlowControl( WireGLGmConnection *gm_conn,
						 WireGLMessageFlowControl *msg, unsigned int len )
{
	wireGLAssert( len == sizeof(WireGLMessageFlowControl) );

	gm_conn->conn->send_credits += msg->credits;

#if WIREGL_GM_CREDITS_DEBUG
	wiregl_gm_debug( "received %u credits for host=%s, have %d total",
					 msg->credits, gm_conn->conn->hostname,
					 gm_conn->conn->send_credits );
#endif

	wiregl_gm_provide_receive_buffer( msg );
}

static void
wireGLGmRecvMulti( WireGLGmConnection *gm_conn, WireGLMessageMulti *msg,
				   unsigned int len )
{
	WireGLMultiBuffer *multi = &gm_conn->multi;
	WireGLMessageType type = msg->type;
	unsigned char *src, *dst;

	wireGLAssert( len > sizeof(*msg) );
	len -= sizeof(*msg);

	if ( len + multi->len > multi->max )
	{
		if ( multi->max == 0 )
		{
			multi->len = sizeof(WireGLGmBuffer);
			multi->max = 8192;
		}
		while ( len + multi->len > multi->max )
		{
			multi->max <<= 1;
		}
		wireGLRealloc( &multi->buf, multi->max );
	}

	dst = (unsigned char *) multi->buf + multi->len;
	src = (unsigned char *) msg + sizeof(*msg);
	memcpy( dst, src, len );
	multi->len += len;

	wiregl_gm_provide_receive_buffer( msg );

	if ( type == WIREGL_MESSAGE_MULTI_TAIL )
	{
		WireGLGmBuffer *gm_buffer = (WireGLGmBuffer *) multi->buf;
				  
		/* build a header so we can delete the message later */
		gm_buffer->magic = WIREGL_GM_BUFFER_RECV_MAGIC;
		gm_buffer->kind  = WireGLGmMemoryBig;
		gm_buffer->len   = multi->len - sizeof(*gm_buffer);
		gm_buffer->pad   = 0;

		/* clean this up before calling the user */
		multi->buf = NULL;
		multi->len = 0;
		multi->max = 0;

		wiregl_gm.recv( gm_conn->conn, gm_buffer + 1, gm_buffer->len );
	}
}

#if WIREGL_GM_PARANOIA

typedef struct WireGLGmSavedSend {
	void *ctx;
	void *data;
} WireGLGmSavedSend;

static int                n_saved_send;
static WireGLGmSavedSend *saved_send;
static int last_seqno;


static void
record_send( void *ctx )
{
	int i;

	for ( i = 0; i < n_saved_send; i++ )
	{
		if ( saved_send[i].ctx == NULL )
		{
			saved_send[i].ctx = ctx;
			memcpy( saved_send[i].data, ctx, __wiregl_max_send );
			return;
		}
	}

	wireGLError( "record_send failed! (no slots)" );
}

static void
validate_send( void *ctx )
{
	int i;

	for ( i = 0; i < n_saved_send; i++ )
	{
		if ( saved_send[i].ctx == ctx )
		{
			if ( memcmp( saved_send[i].data, ctx, __wiregl_max_send ) )
			{
				volatile int foo = 1;
				wireGLWarning( WIREGL_WARN_CRITICAL, 
							   "GM: somebody stomped on a send!" );
				while ( foo )
					;
			}
			saved_send[i].ctx = NULL;
			return;
		}
	}

	wireGLError( "validate_send failed! (no such ctx)" );
}

#endif

static void
wireGLGmRecvOther( WireGLGmConnection *gm_conn, WireGLMessage *msg,
				   unsigned int len )
{
	WireGLGmBuffer *temp;

	wireGLAssert( gm_conn->multi.buf == NULL );

#if WIREGL_GM_PARANOIA
	if ( msg->type == WIREGL_MESSAGE_OPCODES )
	{
		WireGLMessageOpcodes *msg2 = (WireGLMessageOpcodes *) msg;
		unsigned long crc = gm_crc( msg2 + 1, len - sizeof(*msg2) );
		if ( crc != msg2->crc )
		{
			int i, j;
			unsigned int num_words, *data;
			volatile int foo = 1;
			wireGLWarning( WIREGL_WARN_CRITICAL, "msg: seqno=%d start=%p "
						   "len=%u "
						   "type=0x%x senderId=%u numOpcodes=%u crc=0x%x",
						   msg2->seqno, msg, len, msg2->type, msg2->senderId,
						   msg2->numOpcodes, msg2->crc );
			wireGLWarning( WIREGL_WARN_CRITICAL, "crc mismatch! (I compute "
						   "crc=0x%x, msg says crc=0x%x)", crc, msg2->crc );

			num_words = len >> 2;
			if ( num_words > 100 )
				num_words = 100;
			data = (unsigned int *) msg;
			for ( i = 0; i < num_words; ) 
			{
				char text[1024];
				int  offset = 0;
				for ( j = 0; i < num_words && j < 8; i++, j++ ) 
				{
					offset += sprintf( text + offset, " 0x%08x", *data++ );
				}
				wireGLWarning( WIREGL_WARN_CRITICAL, text );
			}
			while ( foo )
				;
		}
		last_seqno = msg2->seqno;
	}
#endif

	temp = (WireGLGmBuffer *) wireGLBufferPoolPop( &wiregl_gm.read_pool );

	if ( temp == NULL )
	{
#if WIREGL_GM_DEBUG
		wiregl_gm_debug( "wireGLGmRecv: ran out of pinned memory, "
						 "copying to unpinned" );
#endif

		/* we're out of pinned buffers, copy this message into
		 * an unpinned buffer so we can jam the just received
		 * buffer back in the hardware */
		temp = (WireGLGmBuffer *) 
			wireGLBufferPoolPop( &wiregl_gm.unpinned_read_pool );
		if ( temp == NULL )
		{
			temp = wireGLAlloc( sizeof(WireGLGmBuffer) + __wiregl_max_send );
			temp->magic = WIREGL_GM_BUFFER_RECV_MAGIC;
			temp->kind  = WireGLGmMemoryUnpinned;
			temp->pad   = 0;
		}
		temp->len = len;
		memcpy( temp+1, msg, len );

		wiregl_gm_provide_receive_buffer( msg );

		wiregl_gm.recv( gm_conn->conn, temp+1, len );
	}
	else
	{
		wiregl_gm_provide_receive_buffer( temp + 1 );

		temp = (WireGLGmBuffer *) msg - 1;
		temp->len = len;

		wiregl_gm.recv( gm_conn->conn, msg, len );
	}
}

static void 
wiregl_gm_send_callback( struct gm_port *port, void *ctx, gm_status_t status )
{
	WireGLGmBuffer *buf = (WireGLGmBuffer *) ctx - 1;

    if ( status != GM_SUCCESS )
    {
        wireGLSimpleError( "wiregl_gm_send_callback error: %s (%d)",
                           wiregl_gm_str_error( status ), status );
    }

	wireGLAssert( buf->magic == WIREGL_GM_BUFFER_SEND_MAGIC );

#if WIREGL_GM_DEBUG
	wiregl_gm_debug( "send callback, ctx=%p", ctx );
#endif

	wireGLAssert( wiregl_gm.num_outstanding_sends > 0 );
	wiregl_gm.num_outstanding_sends--;
	wiregl_gm.num_send_tokens++;
	WIREGL_UNUSED(port);

#if WIREGL_GM_PARANOIA
	validate_send( ctx );
#endif

	wireGLBufferPoolPush( &wiregl_gm.write_pool, buf );
}

static void
wiregl_gm_send( WireGLConnection *conn, void *buf, unsigned int len,
				void *ctx )
{
#if WIREGL_GM_DEBUG
	wiregl_gm_debug( "gm_send: dst=%u:%u buf=%p len=%d ctx=%p",
					 conn->gm_node_id, conn->gm_port_num, buf, len, ctx );
#endif

	/* get a send token */
	while ( wiregl_gm.num_send_tokens == 0 )
	{
		wireGLGmRecv( );
	}

#if WIREGL_GM_BROKEN_SHORT_SENDS
	if ( len <= 256 && WIREGL_GM_BUFFER_SPANS_PAGE_BOUNDARY(buf,len) )
	{
		/* This is a short send that straddles a page boundary.  Some
		 * versions of GM (prior to gm-1.4pre37) will not handle the
		 * page crossing correctly in some cases.  Move the data
		 * (within the same buffer) so that it doesn't cross a page
		 * boundary. */
		void *temp = ctx;
		if ( WIREGL_GM_BUFFER_SPANS_PAGE_BOUNDARY(temp,len) )
		{
			temp = WIREGL_GM_PAGE_ALIGN(temp);
		}
#if 0
		wireGLWarning( WIREGL_WARN_CRITICAL, "GM: copying len=%u send from "
					   "start=%p to %p in ctx=%p", len, buf, temp, ctx );
#endif

		/* the source and destination may overlap, but memmove handles
           that cases correctly */
		memmove( temp, buf, len );
		buf = temp;
	}
#endif

#if WIREGL_GM_PARANOIA
	record_send( ctx );
#endif

	wiregl_gm.num_outstanding_sends++;
	wiregl_gm.num_send_tokens--;

	if ( conn->gm_port_num == wiregl_gm.port_num )
	{
		/* only when both are on the same port... */
		gm_send_to_peer_with_callback( wiregl_gm.port, buf,
									   wiregl_gm.message_size, len,
									   WIREGL_GM_PRIORITY, conn->gm_node_id,
									   wiregl_gm_send_callback, ctx );
	}
	else
	{
		gm_send_with_callback( wiregl_gm.port, buf,
							   wiregl_gm.message_size, len,
							   WIREGL_GM_PRIORITY, conn->gm_node_id,
							   conn->gm_port_num,
							   wiregl_gm_send_callback, ctx );
	}

#if WIREGL_GM_SYNCHRONOUS
	/* force synchronous sends */
	while ( wiregl_gm.num_outstanding_sends > 0 )
		wireGLGmRecv( );
#endif

#if WIREGL_GM_DEBUG
	wiregl_gm_debug( "send complete" );
#endif
}

static void
wireGLGmSendCredits( WireGLConnection *conn )
{
	WireGLGmBuffer *gm_buffer;
	WireGLMessageFlowControl *msg;

	wireGLAssert( wiregl_gm.num_send_tokens > 0 );
	wireGLAssert( conn->recv_credits > 0 );

	gm_buffer = (WireGLGmBuffer *)
		wireGLBufferPoolPop( &wiregl_gm.write_pool );

	wireGLAssert( gm_buffer );

	msg = (WireGLMessageFlowControl *) ( gm_buffer + 1 );

	msg->type    = WIREGL_MESSAGE_FLOW_CONTROL;
	msg->credits = conn->recv_credits;

#if WIREGL_GM_CREDITS_DEBUG
	wiregl_gm_debug( "sending %d credits to host %s",
					 conn->recv_credits, conn->hostname );
#endif

	conn->recv_credits = 0;

	wiregl_gm_send( conn, msg, sizeof(*msg), msg );
}

static void
wireGLGmMaybeSendCredits( void )
{
	if ( wiregl_gm.num_send_tokens == 0 || wiregl_gm.write_pool.num == 0 )
		return;

	if ( wiregl_gm.credit_head && 
		 wiregl_gm.credit_head->conn->recv_credits >= 
		 WIREGL_GM_SEND_CREDITS_THRESHOLD )
	{
		wireGLGmSendCredits( wiregl_gm.credit_head->conn );
		wireGLGmCreditZero( wiregl_gm.credit_head );
	}
}

UTIL_DECL int
wireGLGmRecv( void )
{
	gm_recv_event_t *event;
	
	/* MWE: Note that we are inside a recv so we can catch the error
       of calling wireGLGmSend or wireGLGmFlowControl within a recv
       callback.  This prevents the observed problem of the send
       function checking for any pending recvs and having those
       processed as well, and a resulting recv->send->recv recursion
       that eventually explodes. */
	wiregl_gm.inside_recv++;

#if WIREGL_GM_USE_CREDITS
#if WIREGL_GM_SYNCHRONOUS_XXX
	/* HACK - don't do this for now, it changes the bug we are trying
       to find... */
	/* if we are doing synchronous sends, only ever send credits if we
       don't already have an outstanding send */
	if ( wiregl_gm.num_outstanding_sends == 0 )
	{
		wireGLGmMaybeSendCredits( );
	}
#else
	wireGLGmMaybeSendCredits( );
#endif
#endif

	event = gm_receive( wiregl_gm.port );

	if ( GM_RECV_EVENT_TYPE(event) == GM_NO_RECV_EVENT )
	{
		wiregl_gm.inside_recv--;
		return 0;
	}

#if WIREGL_GM_DEBUG
	// wiregl_gm_check_recv_event( event );
#endif

	switch ( GM_RECV_EVENT_TYPE(event) ) 
	{
	  case GM_RECV_EVENT:
	  case GM_HIGH_RECV_EVENT:
	  case GM_PEER_RECV_EVENT:
	  case GM_HIGH_PEER_RECV_EVENT:
		{
			gm_u32_t len = gm_ntoh_u32( event->recv.length );
			WireGLMessage *msg = (WireGLMessage *)
				gm_ntohp( event->recv.buffer );
			gm_u16_t src_node = gm_ntoh_u16( event->recv.sender_node_id );
			gm_u8_t  src_port = gm_ntoh_u8( event->recv.sender_port_id );
			WireGLGmConnection *gm_conn = 
				wireGLGmConnectionLookup( src_node, src_port );

#if WIREGL_GM_DEBUG
			wiregl_gm_debug( "gm_receive: src=%d:%d buf=%p len=%u (%s)",
							 src_node, src_port, msg, len, 
							 wiregl_gm_str_event_type( event ) );
#endif

#if WIREGL_GM_FENCE_BUFFERS
			wireGLGmCheckFence( msg );
#endif
			switch ( msg->type )
			{
			  case WIREGL_MESSAGE_MULTI_BODY:
			  case WIREGL_MESSAGE_MULTI_TAIL:
				wireGLGmRecvMulti( gm_conn, &msg->multi, len );
				break;

			  case WIREGL_MESSAGE_FLOW_CONTROL:
				wireGLGmRecvFlowControl( gm_conn, &msg->flowControl, len );
				break;

			  case WIREGL_MESSAGE_OPCODES:
			  case WIREGL_MESSAGE_READ_PIXELS:
			  case WIREGL_MESSAGE_WRITEBACK:
				wireGLGmRecvOther( gm_conn, msg, len );
				break;

			  default:
				/* We can end up here if anything strange happens in
				 * the GM layer.  In particular, if the user tries to
				 * send unpinned memory over GM it gets sent as all
				 * 0xAA instead.  This can happen when a program exits
				 * ungracefully, so the GM is still DMAing memory as
				 * it is disappearing out from under it.  We can also
				 * end up here if somebody adds a message type, and
				 * doesn't put it in the above case block.  That has
				 * an obvious fix. */
				{
#if WIREGL_GM_PARANOIA
					int i, num_bytes;

					wireGLWarning( WIREGL_WARN_CRITICAL, "GM: received a bad "
								   "message: src=%u len=%u", src_node, len );
					wireGLWarning( WIREGL_WARN_CRITICAL, "GM: last good "
								   "seqno=%d", last_seqno );
					num_bytes = len;
					if ( num_bytes > 256 )
						num_bytes = 256;
					for ( i = 0; i < num_bytes; i += 32 )
					{
						char string[128];
						int  x = num_bytes - i;
						if ( x > 32 ) x = 32;
						wireGLNetWordsToString( string, sizeof(string),
												(unsigned char *) msg + i, x );
						wireGLWarning( WIREGL_WARN_CRITICAL, "  %s", string );
					}
					{
						volatile int foo = 1;
						while ( foo )
							;
					}
#else
					char string[128];
					wireGLNetBytesToString( string, sizeof(string), msg, len );
					wireGLError( "GM: received a bad message: src=%u len=%u "
								 "buf=[%s]", src_node, len, string );
#endif					
				}
			}
		}
		break;

	  case GM_FAST_RECV_EVENT:
	  case GM_FAST_HIGH_RECV_EVENT:
	  case GM_FAST_PEER_RECV_EVENT:
	  case GM_FAST_HIGH_PEER_RECV_EVENT:
		/* can't handle these yet, I don't know if we need to call
		 * gm_provide_receive_buffer() after handling these or not */

	  default:
#if WIREGL_GM_DEBUG
		wiregl_gm_debug( "gm_unknown: %s", wiregl_gm_str_event_type( event ) );
#endif
		gm_unknown( wiregl_gm.port, event );
	}

	wiregl_gm.inside_recv--;

	return 1;
}

#if 0

static unsigned int
xxx( void *buf, unsigned int len, unsigned int pos, void *pack )
{
	unsigned int n_bytes = len - pos;
	if ( pos == 0 ) 
	{
		WireGLMessageMultiHead *msg = (WireGLMessageMultiHead *) pack;
		wireGLAssert( len > __wiregl_max_send );
		msg->type = WIREGL_MESSAGE_MULTI_HEAD;
		msg->len  = len;
		n_bytes = __wiregl_max_send - sizeof(*msg);
		memcpy( msg + 1, (unsigned char *) buf + pos, n_bytes );
	}
	else if ( sizeof(WireGLMessageTail) + n_bytes <= __wiregl_max_send ) 
	{
		WireGLMessageMultiTail *msg = (WireGLMessageMultiTail *) pack;
		msg->type = WIREGL_MESSAGE_MULTI_TAIL;
		memcpy( msg + 1, (unsigned char *) buf + pos, n_bytes );
	}
	else {
		WireGLMessageMultiBody *msg = (WireGLMessageMultiBody *) pack;
		msg->type = WIREGL_MESSAGE_MULTI_BODY;
		n_bytes = __wiregl_max_send - sizeof(*msg);
		memcpy( msg + 1, (unsigned char *) buf + pos, n_bytes );
	}

	{
		WireGLMessage *msg = (WireGLMessageMultiHead *) wireGLGmAlloc( );
		msg->type = WIREGL_MESSAGE_MULTI_HEAD;
		msg->len  = len;
		n_bytes   = __wiregl_max_send - sizeof(*msg);
		memcpy( msg + 1, buf, n_bytes );
		
		wiregl_gm_send( conn, msg, n_bytes + sizeof(*msg), msg );
	}

	len -= n_bytes;
	src  = (unsigned char *) buf + n_bytes;

	while ( len > __wiregl_max_send - sizeof(WireGLMessage) )
	{
		WireGLMessage *msg = (WireGLMessage *) wireGLGmAlloc( );
	}
}

#endif

static void
wireGLGmWaitForSendCredits( WireGLConnection *conn )
{
	double start, elapsed;

	start = wiregl_gm_clock( );
	do
	{
		wireGLGmRecv( );
		elapsed = wiregl_gm_clock( ) - start;
	}
	while ( conn->send_credits <= 0 && elapsed < 1.0 );

	if ( conn->send_credits <= 0 )
	{
		wireGLWarning( WIREGL_WARN_CRITICAL, "GM: waiting for credits to "
					   "talk to \"%s\"", conn->hostname );

		while ( conn->send_credits <= 0 )
		{
			wireGLGmRecv( );
		}

		elapsed = wiregl_gm_clock( ) - start;

		wireGLWarning( WIREGL_WARN_CRITICAL, "GM: waited %.1f seconds for "
					   "credits to talk to \"%s\"", elapsed, conn->hostname );
	}
}

static void
wireGLGmSendMulti( WireGLConnection *conn, void *buf, unsigned int len )
{
	unsigned char *src;
	wireGLAssert( buf != NULL && len > 0 );

	if ( len <= __wiregl_max_send )
	{
		/* the user is doing a send from memory not allocated by the
		 * network layer, but it does fit within a single message, so
		 * don't bother with fragmentation */
		void *pack = wireGLGmAlloc( );
		memcpy( pack, buf, len );
		wireGLGmSend( conn, &pack, pack, len );
		return;
	}

#if WIREGL_GM_USE_CREDITS
	/* first get some credits */
	if ( conn->send_credits <= 0 )
	{
		wireGLGmWaitForSendCredits( conn );
	}

	conn->send_credits -= len;
#endif

	src = (unsigned char *) buf;
	while ( len > 0 )
	{
		WireGLMessageMulti *msg = (WireGLMessageMulti *) wireGLGmAlloc( );
		unsigned int        n_bytes;

		if ( len + sizeof(*msg) > __wiregl_max_send )
		{
			msg->type = WIREGL_MESSAGE_MULTI_BODY;
			n_bytes   = __wiregl_max_send - sizeof(*msg);
		}
		else
		{
			msg->type = WIREGL_MESSAGE_MULTI_TAIL;
			n_bytes   = len;
		}
		memcpy( msg + 1, src, n_bytes );

		wiregl_gm_send( conn, msg, n_bytes + sizeof(*msg), msg );

		src += n_bytes;
		len -= n_bytes;
	}
}

UTIL_DECL void 
wireGLGmSend( WireGLConnection *conn, void **bufp, 
			  void *start, unsigned int len )
{
	WireGLGmBuffer *gm_buffer;

	if ( wiregl_gm.inside_recv )
	{
		wireGLError( "wireGLGmSend: can not be called within wireGLGmRecv" );
	}

#if WIREGL_GM_PARANOIA
	{
		WireGLMessageOpcodes *msg = (WireGLMessageOpcodes *) start;
		if ( msg->type == WIREGL_MESSAGE_OPCODES )
		{
			static int seqno = 0;
			msg->seqno = seqno++;
			msg->crc   = gm_crc( msg + 1, len - sizeof(*msg) );
			if ( len <= 256 )
			{
				static FILE *foolog = NULL;
				int i, j;
				unsigned int num_words, *data;
				
				if ( foolog == NULL )
				{
					foolog = fopen( "/tmp/foolog.txt", "w" );
					if ( foolog == NULL )
					{
						wireGLError( "can't create foolog!" );
					}
				}
				fprintf( foolog, "msg: seqno=%d dst=%u start=%p len=%u "
						 "type=0x%x senderId=%u numOpcodes=%u crc=0x%x",
						 msg->seqno, conn->gm_node_id,
						 start, len, msg->type, msg->senderId,
						 msg->numOpcodes, msg->crc );
				num_words = len >> 2;
				if ( num_words > 100 )
					num_words = 100;
				data = (unsigned int *) start;
				for ( i = 0; i < num_words; ) {
					fputs( "\n", foolog );
					for ( j = 0; i < num_words && j < 8; i++, j++ ) {
						fprintf( foolog, " 0x%08x", *data++ );
					}
				}
				fputs( "\n", foolog );

			}
		}
	}
#endif

	if ( bufp == NULL )
	{
		wireGLGmSendMulti( conn, start, len );
		return;
	}

#if WIREGL_GM_FENCE_BUFFERS
	wireGLGmCheckFence( *bufp );
#endif

	gm_buffer = (WireGLGmBuffer *) (*bufp) - 1;

	wireGLAssert( gm_buffer->magic == WIREGL_GM_BUFFER_SEND_MAGIC );

#if WIREGL_GM_USE_CREDITS && WIREGL_GM_CREDITS_DEBUG
	if ( conn->send_credits <= 0 )
	{
		wiregl_gm_debug( "need %d credits to talk to host %s, only have %d",
						 len, conn->hostname, conn->send_credits );
	}
#endif

	/* Every time the client calls send we try to be polite and grab
     * any pending GM messages.  We also have to block if we don't
     * have any credits. */
	while ( wireGLGmRecv( ) )
		;

#if WIREGL_GM_USE_CREDITS
	if ( conn->send_credits <= 0 )
	{
		wireGLGmWaitForSendCredits( conn );
	}

	conn->send_credits -= len;
#endif

#if WIREGL_GM_CREDITS_DEBUG
	wiregl_gm_debug( "sending a len=%d message to host=%s, have %d credits "
					 "remaining", len, conn->hostname, conn->send_credits );
#endif

	wiregl_gm_send( conn, start, len, *bufp );

	*bufp = NULL;
}

void
wireGLGmFree( WireGLConnection *conn, void *buf )
{
	WireGLGmBuffer *gm_buffer = (WireGLGmBuffer *) buf - 1;

	wireGLAssert( gm_buffer->magic == WIREGL_GM_BUFFER_RECV_MAGIC );
	conn->recv_credits += gm_buffer->len;

	switch ( gm_buffer->kind )
	{
	  case WireGLGmMemoryPinned:
		wireGLBufferPoolPush( &wiregl_gm.read_pool, gm_buffer );
		break;

	  case WireGLGmMemoryUnpinned:
		wireGLBufferPoolPush( &wiregl_gm.unpinned_read_pool, gm_buffer );
		break;

	  case WireGLGmMemoryBig:
		wireGLFree( gm_buffer );
		break;

	  default:
		wireGLAbort( );
	}

	wireGLGmCreditIncrease( wireGLGmConnectionLookup( conn->gm_node_id, 
													  conn->gm_port_num ) );
}

static int 
wiregl_gm_looks_like_same_host( const char *a, const char *b )
{
	wireGLAssert( a && b );
	while ( *a && *b ) {
		if ( tolower(*a) != tolower(*b) )
			return 0;
		a++;
		b++;
	}

	/* they're the same if the names match, or if one is a DNS prefix
       of the other */
	return ( ( *a == '\0' && *b == '\0' ) ||
			 ( *a == '.'  && *b == '\0' ) ||
			 ( *a == '\0' && *b == '.'  ) );
}

static void
wiregl_gm_set_acceptable_sizes( void )
{
	gm_status_t      status;
	enum gm_priority priority;
	unsigned long    mask;

	for ( priority = 0; priority < GM_NUM_PRIORITIES; priority++ )
	{
		mask = 0;
		if ( priority == WIREGL_GM_PRIORITY )
			mask = 1 << wiregl_gm.message_size;

		status = gm_set_acceptable_sizes( wiregl_gm.port, priority, mask );
		if ( status != GM_SUCCESS )
		{
			wireGLSimpleError( "GM: gm_set_acceptable_sizes( port=%u, pri=%u, "
							   "mask=%u ) failed: %s (%d)", wiregl_gm.port_num,
							   priority, mask, 
							   wiregl_gm_str_error( status ), status );
		}
	}
}

UTIL_DECL void
wireGLGmInit( WireGLNetReceiveFunc recvFunc, WireGLNetCloseFunc closeFunc )
{
	gm_status_t status;
	unsigned int port, min_port, max_port;
	unsigned int node_id, max_node_id, i;
	unsigned int stride, count, n_bytes, num_recv_tokens;
	unsigned char *mem;

    if ( wiregl_gm.initialized )
	{
		wireGLAssert( wiregl_gm.recv == recvFunc );
		wireGLAssert( wiregl_gm.close == closeFunc );
		return;
	}

	wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: initializing" );

	status = gm_init( );
	if ( status != GM_SUCCESS )
	{
		wireGLSimpleError( "GM: gm_init() failed: %s (%d)",
						   wiregl_gm_str_error( status ), status );
	}

	/* open a reasonable port */
	min_port = 4;
	max_port = gm_num_ports( NULL /* known unused */ ) - 1;
	for ( port = min_port; port <= max_port; port++ )
	{
		status = gm_open( &wiregl_gm.port, 0, port,
						  "WireGL GM", WIREGL_GM_API_VERSION );
		if ( status == GM_SUCCESS )
		{
			break;
		}
		if ( status != GM_BUSY )
		{
			wireGLSimpleError( "GM: gm_open( port=%u ) failed: %s (%d)",
							   port, wiregl_gm_str_error( status ), status );
		}
	}
	if ( port > max_port )
	{
		wireGLSimpleError( "GM: all ports busy (tried port %u through %u)",
						   min_port, max_port );
	}
	wiregl_gm.port_num = port;

	wiregl_gm.message_size = gm_min_size_for_length( __wiregl_max_send );

	wiregl_gm_set_acceptable_sizes( );

	wiregl_gm_info( );

	gm_max_node_id_inuse( wiregl_gm.port, &max_node_id );

	wiregl_gm.num_nodes = max_node_id + 1;
	for ( i = 0; i < WIREGL_GM_CONN_HASH_SIZE; i++ )
	{
		wiregl_gm.gm_conn_hash[i] = NULL;
	}
	
	wiregl_gm.credit_head = NULL;
	wiregl_gm.credit_tail = NULL;

    gm_get_node_id( wiregl_gm.port, &node_id );
	wiregl_gm.node_id = node_id;

	stride  = __wiregl_max_send;
#if WIREGL_GM_ROUND_UP
	stride  = gm_max_length_for_size( gm_min_size_for_length( stride ) );
#endif
#if WIREGL_GM_FENCE_BUFFERS
	if ( __wiregl_max_send + sizeof(unsigned int) > stride )
	{
		stride += sizeof(unsigned int);
	}
#endif
	stride += sizeof(WireGLGmBuffer);
	count   = WIREGL_GM_PINNED_READ_MEMORY_SIZE / stride;
	n_bytes = count * stride;
	wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: read pool: bufsize=%u, "
				   "%u buffers, %u bytes", stride, count, n_bytes );
	mem     = wiregl_gm_dma_malloc( n_bytes );

	num_recv_tokens = gm_num_receive_tokens( wiregl_gm.port );

	wireGLBufferPoolInit( &wiregl_gm.read_pool, count );
	for ( i = 0; i < count; i++ )
	{
		WireGLGmBuffer *buf = (WireGLGmBuffer *) ( mem + i * stride );
		buf->magic = WIREGL_GM_BUFFER_RECV_MAGIC;
		buf->kind  = WireGLGmMemoryPinned;
		buf->len   = __wiregl_max_send;
		buf->pad   = 0;

#if WIREGL_GM_FENCE_BUFFERS
		wireGLGmPlaceFence( buf + 1 );
#endif

		if ( i < num_recv_tokens )
		{
			wiregl_gm_provide_receive_buffer( buf + 1 );
		}
		else
		{
			wireGLBufferPoolPush( &wiregl_gm.read_pool, buf );
		}
	}

	wireGLBufferPoolInit( &wiregl_gm.unpinned_read_pool, 16 );

	stride  = __wiregl_max_send;
#if WIREGL_GM_ROUND_UP
	stride  = gm_max_length_for_size( gm_min_size_for_length( stride ) );
#endif
#if WIREGL_GM_FENCE_BUFFERS
	if ( __wiregl_max_send + sizeof(unsigned int) > stride )
	{
		stride += sizeof(unsigned int);
	}
#endif
	stride += sizeof(WireGLGmBuffer);
	count   = WIREGL_GM_PINNED_WRITE_MEMORY_SIZE / stride;
	n_bytes = count * stride;
	wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: write pool: bufsize=%u, "
				   "%u buffers, %u bytes", stride, count, n_bytes );
	mem     = wiregl_gm_dma_malloc( n_bytes );

	wireGLBufferPoolInit( &wiregl_gm.write_pool, count );
	for ( i = 0; i < count; i++ )
	{
		WireGLGmBuffer *buf = (WireGLGmBuffer *) ( mem + i * stride );
		buf->magic = WIREGL_GM_BUFFER_SEND_MAGIC;
		buf->kind  = WireGLGmMemoryPinned;
		buf->len   = __wiregl_max_send;
		buf->pad   = 0;

#if WIREGL_GM_FENCE_BUFFERS
		wireGLGmPlaceFence( buf + 1 );
#endif
		wireGLBufferPoolPush( &wiregl_gm.write_pool, buf );
	}

	wiregl_gm.recv  = recvFunc;
	wiregl_gm.close = closeFunc;

	wiregl_gm.num_outstanding_sends = 0;
	wiregl_gm.num_send_tokens = gm_num_send_tokens( wiregl_gm.port );

#if WIREGL_GM_PARANOIA
	n_saved_send = wiregl_gm.num_send_tokens;
	saved_send   = (WireGLGmSavedSend *) 
		wireGLAlloc( sizeof(saved_send[0]) * n_saved_send );
	for ( i = 0; i < wiregl_gm.num_send_tokens; i++ )
	{
		saved_send[i].ctx  = NULL;
		saved_send[i].data = wireGLAlloc( __wiregl_max_send );
	}
#endif

	wiregl_gm.inside_recv = 0;

	wiregl_gm.initialized = 1;
}

UTIL_DECL unsigned int
wireGLGmNodeId( void )
{
	wireGLAssert( wiregl_gm.initialized );
	return wiregl_gm.node_id;
}

UTIL_DECL unsigned int
wireGLGmPortNum( void )
{
	wireGLAssert( wiregl_gm.initialized );
	return wiregl_gm.port_num;
}

UTIL_DECL void 
wireGLGmConnection( WireGLConnection *conn )
{
	char *actual_name;

	wireGLAssert( wiregl_gm.initialized );

    if ( conn->gm_node_id == GM_NO_SUCH_NODE_ID )
    {
        wireGLSimpleError( "GM: there's no host called \"%s\"?",
						   conn->hostname );
    }

	actual_name = gm_node_id_to_host_name( wiregl_gm.port, conn->gm_node_id );
	if ( !actual_name )
	{
		wireGLSimpleError( "GM: gm_node_id_to_host_name( id=%u ) failed",
						   conn->gm_node_id );
	}

	if ( conn->hostname && 
		 !wiregl_gm_looks_like_same_host( conn->hostname, actual_name ) )
	{
		wireGLSimpleError( "GM: \"%s\" says id=%u, but I think \"%s\" "
						   "is on that ID", conn->hostname,
						   conn->gm_node_id, actual_name );
	}

	wireGLGmConnectionAdd( conn );

#if 0
    /* don't close the socket, it shows we are alive! */
    wireGLCloseSocket( conn->tcp_socket );
#endif

    conn->type  = WIREGL_GM;
	conn->Alloc = wireGLGmAlloc;
	conn->Send  = wireGLGmSend;
	conn->Free  = wireGLGmFree;

    wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "GM: accepted connection from "
				   "host=%s id=%d (gm_name=%s)", conn->hostname,
				   conn->gm_node_id, actual_name );
}
