/* IEEE Floating point representation
 * ----------------------------------
 *
 * +-+--------+-----------------------+
 * |S|Exponent|Mantissa               |
 * +-+--------+-----------------------+
 *   3        2                       0
 *   1        3
 */

static __inline unsigned int __extractMantissa ( GLfloat number )
{
	unsigned int intNumber = *(unsigned int*)&number;
	intNumber &= 0x007FFFFF;
	intNumber += 0x00800000;
	return(intNumber);
}

static __inline unsigned int __extractExponent ( GLfloat number )
{
	unsigned int intNumber = *(unsigned int*)&number;
	intNumber &=  0x7F800000;
	intNumber >>= 23;
	return(intNumber);
}

/* __extractProperExponent
 * -----------------------
 * Same as __extractExponent but properly returns the unbiased exponent
 */

static __inline int __extractProperExponent ( GLfloat number )
{
	unsigned int intNumber = *(unsigned int*)&number;
	if(intNumber == 0)
		return 0;
	intNumber &=  0x7F800000;
	intNumber >>= 23;
	intNumber -= 127;
	return(intNumber);
}

static __inline unsigned int __extractSign ( GLfloat number )
{
	unsigned int intNumber = *(unsigned int*)&number;
	intNumber &= 0x80000000;
	intNumber >>= 31;
	return(intNumber);
}

static __inline int __isNegative ( GLfloat number )
{
	unsigned int intNumber = *(unsigned int*)&number;
	return(intNumber&0x80000000);
}

static __inline void __absoluteValue( GLfloat *number)
{
	unsigned int *intNumber = (unsigned int*)number;
	*intNumber &= 0x7FFFFFFF;
}

/* __extractDenormalMantissa
 * -----------------------
 * Extracts a mantissa and shifts it so that it represents the decimal portion of the
 * floating point number.
 *
 * For example, __extractNormalMantissa(1.0) returns 0x00800000
 *				__extractNormalMantissa(0.5) returns 0x00400000
 */

static __inline unsigned int __extractDenormalMantissa ( GLfloat number )
{
	unsigned int mantissa = __extractMantissa(number);
	unsigned int exponent = 127 - __extractExponent(number);
	mantissa >>= exponent;
	return(mantissa);
}

/* __normalizeMantissa
 * -------------------
 * This function takes the denormalized mantissa that is produced by __extractDenormalMantissa
 * and turns it back into a IEEE floating point value.
 */

static __inline GLfloat __normalizeMantissa ( unsigned int value )
{
	unsigned int exponent = 127 << 23;

	if(value == 0)
		return 0.0;

	/* Keep shifting the mantissa to the left until it has a one in the
	 * 23rd bit, which means it is in IEEE format. At the same time,
	 * keep decreasing the exponent to compensate*/

	while( !(value & 0x00800000) )
	{
		value <<= 1;
		exponent -= 0x00800000;
	}

	value &= 0x007FFFFF;
	value |= exponent;

	return(*(float*)&value);
}
