/* volcrunch.c */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <memory.h>
#include <assert.h>

#include "volume.h"
#include "globals.h"

void
Volume_SubSample( Volume *src, u32 factor, Volume *dst )
{
    u32 i, j, k, ii, jj, kk, ii_max, jj_max, kk_max;
    u32 src_width, src_height, src_depth, src_area;
    u32 dst_width, dst_height, dst_depth;
    u8 *dst_density, *src_ptr, *dst_ptr;
    u32 sum;

    src_width  = src->width;
    src_height = src->height;
    src_depth  = src->depth;
    src_area   = src_width * src_height;

    dst_width  = ( src_width  + factor - 1 ) / factor;
    dst_height = ( src_height + factor - 1 ) / factor;
    dst_depth  = ( src_depth  + factor - 1 ) / factor;

    dst_density = (u8 *) xmalloc( dst_width * dst_height * dst_depth );
    dst_ptr = dst_density;

    for ( k = 0; k < src_depth; k += factor ) {
        for ( j = 0; j < src_height; j += factor ) {
            for ( i = 0; i < src_width; i += factor ) {

                ii_max = MIN( src_width,  i + factor );
                jj_max = MIN( src_height, j + factor );
                kk_max = MIN( src_depth,  k + factor );

                sum = 0;
                for ( kk = k; kk < kk_max; kk++ ) {
                    src_ptr = src->density + src_area * kk + src_width * j + i;
                    for ( jj = j; jj < jj_max; jj++ ) {
                        for ( ii = i; ii < ii_max; ii++, src_ptr++ )
                            sum += *src_ptr;
                        src_ptr += src_width - ( ii_max - i );
                    }
                }

                *dst_ptr = sum / (( kk_max - k ) * ( jj_max - j ) * ( ii_max - i ));
                dst_ptr++;
            }
        }
    }

    assert( dst_ptr - dst_density == (int) ( dst_width * dst_height * dst_depth ) );

    if ( src == dst )
        xfree( src->density );

    dst->width    = dst_width;
    dst->height   = dst_height;
    dst->depth    = dst_depth;
    dst->density  = dst_density;
    dst->i_chunks = 0;
    dst->j_chunks = 0;
    dst->k_chunks = 0;
    dst->chunk    = NULL;
}

void
Volume_Pad( Volume *volume, int dst_width, int dst_height, int dst_depth )
{
    int src_width, src_height, src_depth;
    int x_pad, y_pad, z_pad, x_off, y_off, z_off;
    int dst_area, n;
    u8 *new_density, *src, *dst;
    int i, j, k;

    src_width  = volume->width;
    src_height = volume->height;
    src_depth  = volume->depth;

    assert( dst_width  >= src_width  &&
            dst_height >= src_height &&
            dst_depth  >= src_depth );

    if ( src_width  == dst_width &&
         src_height == dst_height &&
         src_depth  == dst_depth ) return;

    x_pad = dst_width  - src_width;
    y_pad = dst_height - src_height;
    z_pad = dst_depth  - src_depth;

    /* we'll split the pad we add evenly around all the sides of the
       volume */
    x_off = x_pad >> 1;
    y_off = y_pad >> 1;
    z_off = z_pad >> 1;

    dst_area = dst_width * dst_height;

    /* lucky dog, no padding whatsoever required */
    if ( !x_pad && !y_pad && !z_pad ) return;

    n = dst_area * dst_depth;
    new_density = (u8 *) xmalloc( n );
    assert( new_density );

    /* probably don't need to do this.  If we haven't free'd any
       memory the OS will be giving us new pages, which means they
       should be zero'd */
    memset( new_density, 0, n );

    dst = new_density + z_off * dst_area + y_off * dst_width + x_off;
    src = volume->density;
    for ( k = 0; k != src_depth; k++ ) {
        for ( j = 0; j != src_height; j++ ) {
            for ( i = 0; i != src_width; i++ )
                *dst++ = *src++;
            dst += x_pad;
        }
        dst += y_pad * dst_width;
    }

    xfree( volume->density );

    volume->width   = dst_width;
    volume->height  = dst_height;
    volume->depth   = dst_depth;
    volume->density = new_density;
}

static void
CubeMinMax( int width, int height, int depth, int row_skip, int slice_skip,
            u8 *vals, u8 *min_val, u8 *max_val )
{
    u8 min, max;
    int i, j, k;

    min = max = *vals;

    for ( k = 0; k < depth; k++ ) {
        for ( j = 0; j < height; j++ ) {
            for ( i = 0; i < width; i++ ) {
                if ( *vals > max ) max = *vals;
                if ( *vals < min ) min = *vals;
                vals++;
            }
            vals += row_skip - width;
        }
        vals += slice_skip - ( row_skip * height );
    }

    *min_val = min;
    *max_val = max;
}

static void
ChunkMinMax( VolumeChunk *chunk )
{
    int chunk_w, chunk_h, chunk_d, row_skip, slice_skip;
    int i, j, k, n;
    VolumeMinMax *temp;

    chunk_w    = chunk->w;
    chunk_h    = chunk->h;
    chunk_d    = chunk->d;
    row_skip   = chunk->row_skip;
    slice_skip = chunk->slice_skip;

    /* chunk dimensions must be a power of 2 */
    assert( !(chunk_w & 0x1) && !(chunk_h & 0x1) && !(chunk_d & 0x1) );

    n = ( chunk_w >> 1 ) * ( chunk_h >> 1 ) * ( chunk_d >> 1 );

    chunk->minmax  = (VolumeMinMax *) xmalloc( sizeof(*temp) * n );
    assert( chunk->minmax );

    temp = chunk->minmax;
    for ( k = 0; k < chunk_d; k += 2 ) {
        for ( j = 0; j < chunk_h; j +=2 ) {
            for ( i = 0; i < chunk_w; i += 2 ) {
                CubeMinMax( 3, 3, 3, row_skip, slice_skip,
                            chunk->density + k * slice_skip + j * row_skip + i,
                            &temp->min, &temp->max );
                temp++;
            }
        }
    }
}

static VolumeTraversalOrder *
Volume_ComputeTraversalOrder( int width, int height, int depth, int order )
{
    int i, j, k, ii, jj, kk, i_max, j_min, j_max, step, step_max, area, n;
    VolumeTraversalOrder *list, *temp;

    assert( width < 256 && height < 256 && depth < 256 );

    area = width * height;
    n = area * depth;
    list = (VolumeTraversalOrder *) xmalloc( sizeof(*list) * n );
    assert( list );

    step_max = width + height + depth - 3;

    temp = list;
    for ( step = 0; step <= step_max; step++ ) {

        i_max = MIN( width - 1, step );
        for ( i = 0; i <= i_max; i++ ) {

            j_min = MAX( 0, step - i - depth + 1 );
            j_max = MIN( height - 1, step - i );
            for ( j = j_min; j <= j_max; j++ ) {

                k = step - i - j;

                ii = ( order & 1 ) ? ( width  - 1 - i ) : i;
                jj = ( order & 2 ) ? ( height - 1 - j ) : j;
                kk = ( order & 4 ) ? ( depth  - 1 - k ) : k;

                temp->i      = ii;
                temp->j      = jj;
                temp->k      = kk;
                temp->step   = step;
                temp->offset = kk * area + jj * width + ii;

                temp++;
            }
        }
    }

    assert( list + n == temp );

    return list;
}

typedef struct TraversalCache {
    int                    w, h, d;
    VolumeTraversalOrder **set;
    struct TraversalCache *next;
} TraversalCache;

static VolumeTraversalOrder **
Volume_ComputeTraversalOrderSet( int width, int height, int depth )
{
    static TraversalCache *head = NULL;
    TraversalCache *temp;
    VolumeTraversalOrder **set;
    int order;

    for ( temp = head; temp; temp = temp->next ) {
        if ( temp->w == width && temp->h == height && temp->d == depth )
            return temp->set;
    }

    /*
    printf( "Cache Miss: %dx%dx%d\n", width, height, depth );
    */

    set = (VolumeTraversalOrder **) xmalloc( sizeof(*set) * 8 );
    assert( set );
    for ( order = 0; order != 8; order++ )
        set[order] = \
            Volume_ComputeTraversalOrder( width, height, depth, order );

    temp = (TraversalCache *) xmalloc( sizeof( *temp ) );
    assert( temp );

    temp->w    = width;
    temp->h    = height;
    temp->d    = depth;
    temp->set  = set;
    temp->next = head;

    head = temp;

    return set;
}

void
Volume_Chunk( Volume *volume, int chunk_w, int chunk_h, int chunk_d )
{
    int width, height, depth, area, offset;
    int i_chunks, j_chunks, k_chunks, n, i, j, k;
    VolumeChunk *chunk;

    assert( chunk_w > 1 && chunk_h > 1 && chunk_d > 1 );

    width  = volume->width;
    height = volume->height;
    depth  = volume->depth;
    area   = width * height;

    i_chunks = ( width  - 2 - ( width  & 0x1 ) + chunk_w - 1 ) / chunk_w;
    j_chunks = ( height - 2 - ( height & 0x1 ) + chunk_h - 1 ) / chunk_h;
    k_chunks = ( depth  - 2 - ( depth  & 0x1 ) + chunk_d - 1 ) / chunk_d;

	if ( globals.thread_id == 0 )
	{
		printf( "Volume Chunk:\n  chunksize=%dx%dx%d\n",
				chunk_w, chunk_h, chunk_d );
		printf( "  %dx%dx%d chunks\n", i_chunks, j_chunks, k_chunks );
	}

    n = i_chunks * j_chunks * k_chunks;

    chunk = (VolumeChunk *) xmalloc( sizeof(volume->chunk[0]) * n );
    assert( chunk );

    volume->i_chunks = i_chunks;
    volume->j_chunks = j_chunks;
    volume->k_chunks = k_chunks;
	volume->num_chunks = i_chunks * j_chunks * k_chunks;
    volume->chunk    = chunk;

    for ( k = 0; k < k_chunks; k++ ) {
        for ( j = 0; j < j_chunks; j++ ) {
            for ( i = 0; i < i_chunks; i++ ) {

                chunk->i0  = 1 + i * chunk_w;
                chunk->j0  = 1 + j * chunk_h;
                chunk->k0  = 1 + k * chunk_d;

                chunk->i_idx = i;
                chunk->j_idx = j;
                chunk->k_idx = k;

                chunk->w   = MIN( (width  & ~1) - 1 - chunk->i0, chunk_w);
                chunk->h   = MIN( (height & ~1) - 1 - chunk->j0, chunk_h);
                chunk->d   = MIN( (depth  & ~1) - 1 - chunk->k0, chunk_d);

                assert( chunk->w > 0 && chunk->h > 0 && chunk->d > 0 );

                chunk->row_skip   = width;
                chunk->slice_skip = area;

                offset    = area * chunk->k0 + width * chunk->j0 + chunk->i0;
                chunk->density    = volume->density + offset;

                CubeMinMax( chunk->w+1, chunk->h+1, chunk->d+1,
                            width, area, chunk->density,
                            &chunk->min_density, &chunk->max_density );

                ChunkMinMax( chunk );

                chunk->accelOrder = \
                    Volume_ComputeTraversalOrderSet( chunk->w >> 1,
                                                     chunk->h >> 1,
                                                     chunk->d >> 1 );

                chunk++;
            }
        }
    }

    volume->chunkOrder = \
        Volume_ComputeTraversalOrderSet( i_chunks, j_chunks, k_chunks );
}
