/* misc.c */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <memory.h>
#include <stdarg.h>
#include <assert.h>

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#include <GL/gl.h>

#include "misc.h"
#include "globals.h"

#define X(ex)   { ex, # ex }

static struct {
    GLenum      err;
    const char *str;
} errmap[] = {
    X(GL_INVALID_ENUM),
    X(GL_INVALID_VALUE),
    X(GL_INVALID_OPERATION),
    X(GL_STACK_OVERFLOW),
    X(GL_STACK_UNDERFLOW),
    X(GL_OUT_OF_MEMORY)
};

#undef X

void
CheckOglError( const char *file, int line )
{
    int    i;
    GLenum err;

    err = glGetError( );
    while ( err != GL_NO_ERROR ) {

        for ( i = 0; i != NELEMS(errmap); i++ ) {
            if ( errmap[i].err == err )
                break;
        }

        if ( i != NELEMS(errmap) )
            fprintf( stderr, "GL_ERROR: %s (%d:%s)\n",
                     errmap[i].str, line, file );
        else
            fprintf( stderr, "GL_ERROR: bad code (0x%04x) (%d:%s)\n",
                     err, line, file );

        err = glGetError( );
    }
}

void *
xmalloc( size_t size )
{
	void *ptr = malloc( size );
	if ( ptr == NULL ) 
	{
		xfatal( "Couldn't allocate %d bytes\n", (int) size );
	}

	globals.malloc_bytes += (u32) size;

	return ptr;
}

void
xfree( void *ptr )
{
	if ( ptr )
	{
		free( ptr );
	}
}

void
xfatal( const char *format, ... )
{
	va_list args;
	fputs( "march: ", stderr );
	va_start( args, format );
	vfprintf( stderr, format, args );
	va_end( args );
	fputc( '\n', stderr );
	exit( 1 );
}
