/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <stdio.h>

#include "wiregl/include/wiregl_util.h"

#define WIREGLSERVICENAME  "WireGL"

typedef struct WireGLService {
	SERVICE_STATUS_HANDLE    hServiceStatus;
	SERVICE_STATUS           status;
	PROCESS_INFORMATION      pinfo;
	HANDLE                   watchdog;
	HANDLE                   lock;
	HANDLE                   watchdog_quit;
} WireGLService;

WireGLService svc;

void
AquireSvcLock (void) {
	if (WaitForSingleObject (svc.lock, 1000) == WAIT_FAILED)
		wireGLError("svc.lock failed!");
}

void
ReleaseSvcLock (void) {
	if ( !ReleaseSemaphore( svc.lock, 1, NULL) )
		wireGLError("release failed!");
}


/* This thread keeps the pipeserver 
** restarting when it quits.
*/
DWORD WatchDogMain (void *arg) {
	DWORD exitcode = 0;
	DWORD retval;
	STARTUPINFO sinfo;
	char path[2048];

	ZeroMemory(&sinfo, sizeof (sinfo));
	sinfo.cb = sizeof (sinfo);

    if (!GetModuleFileName(NULL, path, 2048))
		wireGLError("Unable to get path to module.");
	
	WIREGL_UNUSED(arg);

	for (;;) {

		/* Check every second */
		retval = WaitForSingleObject (svc.watchdog_quit, 1000);
	
		/* Check to make sure that we
		** haven't been signaled to 
		** quit
		*/
		switch (retval) {
		case WAIT_ABANDONED:
			wireGLError("Lock abandoned");
		case WAIT_FAILED:
			wireGLError("Watchdog_quit Lock Failed");
		case WAIT_OBJECT_0:
			ExitThread(0);
		}

		AquireSvcLock();
		
		/* Quit the watchdog if we're not trying to 
		** keep the pipeserver running.
		*/
		if (svc.status.dwCurrentState != SERVICE_RUNNING) {
			ReleaseSvcLock();
			ExitThread(0);
		}

		GetExitCodeProcess(svc.pinfo.hProcess, &exitcode);
		if (exitcode != STILL_ACTIVE) {
			
			/* Restart the pipeserver */
			retval = CreateProcess(
				path,                    /* Application Name */
				"",                        /* Application Args */
				NULL,                      /* Process Security Descriptor */
				NULL,                      /* Thread Security Descriptor */
				TRUE,				       /* Inherit Handles */
				CREATE_DEFAULT_ERROR_MODE | CREATE_NO_WINDOW, /* Create Flags */
				NULL,                      /* Enviroment Variables */
				NULL,                      /* Current Directory */
				&sinfo,                    /* Startup Info */
				&svc.pinfo                 /* Process Info */
				);

			if (!retval) {
				wireGLError("CreateProcess Failed!");
				ReleaseSvcLock();
				ExitThread(0);
			}
		}

		ReleaseSvcLock();
	}
}

// Service control routine
void WINAPI ServiceCtrl(DWORD ctrlcode)
{
	DWORD exitcode;
    
	switch(ctrlcode)
    {

	case SERVICE_CONTROL_STOP:
		/* STOP : The service must stop */
		
		AquireSvcLock();
		svc.status.dwCurrentState = SERVICE_STOP_PENDING;
		svc.status.dwCheckPoint = 1;

		if (!SetServiceStatus(svc.hServiceStatus, &svc.status)) {
			wireGLError("Failed to set status");
			ReleaseSvcLock();
			return;
		}

		/* Signal to the watchdog to quit */
		ReleaseSemaphore(svc.watchdog_quit, 1, NULL);
		ReleaseSvcLock();

		/* Give the watchdog thread a chance to exit
		** gracefully. 
		*/
		Sleep(100);
		GetExitCodeProcess(svc.watchdog, &exitcode);
		if (exitcode) {
			/* Apply fist of death! */
			TerminateThread(svc.watchdog, 0);
		}

		AquireSvcLock();

		/* Kill the pipeserver.  We should do something 
		** more polite here but since the pipeserver
		** rarely responds to windows events
		** this is the only alternative.
		*/
		TerminateProcess(svc.pinfo.hProcess, 0);

		svc.status.dwCurrentState = SERVICE_STOPPED;
		svc.status.dwCheckPoint = 0;
		if (!SetServiceStatus(svc.hServiceStatus, &svc.status)) {
			wireGLError("Failed to set status");
		}
		ReleaseSvcLock();
		return;

	default:
		AquireSvcLock();
		if (!SetServiceStatus(svc.hServiceStatus, &svc.status)) {
			wireGLError("Failed to set status");
		}
		ReleaseSvcLock();
    }
}

void WINAPI ServiceStart(DWORD argc, char**argv)
{

	STARTUPINFO sinfo;
	BOOL retval;
	DWORD exitcode;
	char path[2048];

	WIREGL_UNUSED(argc);
	WIREGL_UNUSED(argv);

	/* We should be more carefull here.
	** since there may be left over
	** cruft from the last time we 
	** were started 
	*/
	ZeroMemory(&svc, sizeof (svc));

	svc.lock = CreateSemaphore( NULL, 1, 1, NULL );
	if ( svc.lock == NULL )
		wireGLError( "Couldn't allocate a semaphore" );

	AquireSvcLock();

	svc.watchdog_quit = CreateSemaphore (NULL, 0, 1, NULL);
	if ( svc.watchdog_quit == NULL )
		wireGLError( "Couldn't allocate a semaphore" );
	
	/* 
	** Register the service control handler
	*/
	svc.hServiceStatus = RegisterServiceCtrlHandler(WIREGLSERVICENAME, ServiceCtrl);
	
	if (!svc.hServiceStatus)
		wireGLError("Unable to register control handler");

	/*
	** Initalize the service status
	*/
    svc.status.dwServiceType = SERVICE_WIN32 | SERVICE_INTERACTIVE_PROCESS;
    svc.status.dwServiceSpecificExitCode = 0;
	svc.status.dwControlsAccepted = 0;
	svc.status.dwWin32ExitCode = NO_ERROR;
	svc.status.dwWaitHint = 15000;
	svc.status.dwCheckPoint = 0;

	/*
	** Start the service
	*/
	svc.status.dwCurrentState = SERVICE_START_PENDING;
	svc.status.dwCheckPoint = 1;

    if (!SetServiceStatus(svc.hServiceStatus, &svc.status))
		wireGLError("Failed to issue start pending status");

	/* The default settings are used if 
	** all members of start_info are zero
	*/
	ZeroMemory(&sinfo, sizeof (sinfo));
	sinfo.cb = sizeof (sinfo);

    if (!GetModuleFileName(NULL, path, 2048))
		wireGLError("Unable to get path to module.");

	/* Create the process */
	retval = CreateProcess(
		path,                      /* Application Name */
		"",                        /* Application Args */
		NULL,                      /* Process Security Descriptor */
		NULL,                      /* Thread Security Descriptor */
		TRUE,				       /* Inherit Handles */
		CREATE_DEFAULT_ERROR_MODE | CREATE_NO_WINDOW, /* Create Flags */
		NULL,                      /* Enviroment Variables */
		NULL,                      /* Current Directory */
		&sinfo,                    /* Startup Info */
		&svc.pinfo                 /* Process Info */
		);


	if (!retval) {
		int err = GetLastError();
		char msg[256];
		sprintf(msg, "Create Process Failed: %d\n", err);
		wireGLError ("Failed to create process");
		svc.status.dwCurrentState = SERVICE_STOPPED;
		svc.status.dwCheckPoint = 0;
		if (!SetServiceStatus(svc.hServiceStatus, &svc.status))
			wireGLError ("Failed to issue start pending status");
		ReleaseSvcLock();
		return;
	}

	/* Wait for the process to get started, max 1 second */
	WaitForInputIdle (svc.pinfo.hProcess, 1000);

	/* Check to make sure that everything is 
	** still running...
	*/
	GetExitCodeProcess (svc.pinfo.hProcess, &exitcode);

	/* The process quit before we got to start, bummer */
	if (exitcode != STILL_ACTIVE) {
		wireGLError ("Process failed to start");
		svc.status.dwCurrentState = SERVICE_STOPPED;
		svc.status.dwCheckPoint = 0;
		if (!SetServiceStatus(svc.hServiceStatus, &svc.status))
			wireGLError("Failed to startup the pipeserver");
		ReleaseSvcLock();
		return;
	}    

	/* We're good, report our status to the SVM */
	svc.status.dwCurrentState = SERVICE_RUNNING;
	svc.status.dwControlsAccepted = SERVICE_ACCEPT_STOP;
	svc.status.dwCheckPoint = 0;
	if (!SetServiceStatus(svc.hServiceStatus, &svc.status)) {
		wireGLError("Failed to issue start pending status");
		ReleaseSvcLock();
		return;
	}

	/* Start the WatchDog thread */
	svc.watchdog = CreateThread (NULL, 0, (LPTHREAD_START_ROUTINE) WatchDogMain, NULL, 0, NULL);
	
	ReleaseSvcLock();

	return;
}

void wireGLServiceMain ( void )
{ 
    SERVICE_TABLE_ENTRY   DispatchTable[] = 
    { 
        { WIREGLSERVICENAME, ServiceStart  }, 
        { NULL,              NULL          } 
    };

    if (!StartServiceCtrlDispatcher( DispatchTable)) 
    { 
		wireGLError("Unable to start service dispacther");
    } 

} 
 
#define SERVICEARG "-service"

int
wireGLInstallService(void)
{
	const unsigned int pathlength = 2048;
	char path[2048];
	char servicecmd[2048];
	SC_HANDLE   hservice;
	SC_HANDLE   hsrvmanager;

    if (!GetModuleFileName(NULL, path, pathlength-(strlen(SERVICEARG)+2)))
		wireGLError("Unable to get path to module.");

	if (strlen(path) + 4 + strlen(SERVICEARG) < pathlength)
		sprintf(servicecmd, "\"%s\" %s", path, SERVICEARG);
	else
		wireGLError("Path too long to install.");

    hsrvmanager = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS);
	if (hsrvmanager == NULL)
		wireGLError("Unable to open service manager.");

	hservice = CreateService(
		hsrvmanager,				// SCManager database
		WIREGLSERVICENAME,			// name of service
		"WireGL Server",	        // name to display
		SERVICE_ALL_ACCESS,			// desired access
		SERVICE_WIN32_OWN_PROCESS | SERVICE_INTERACTIVE_PROCESS,
									// service type
		SERVICE_AUTO_START,			// start type
		SERVICE_ERROR_NORMAL,		// error control type
		servicecmd,					// service's binary
		NULL,						// no load ordering group
		NULL,						// no tag identifier
		NULL,						// no dependencies
		NULL,						// LocalSystem account
		NULL);						// no password
	CloseServiceHandle(hsrvmanager);
	
	if (hservice == NULL)
		wireGLError("Unable to create service.");

	CloseServiceHandle(hservice);

	return 1;
}

int
wireGLRemoveService()
{
	SC_HANDLE   hservice;
	SC_HANDLE   hsrvmanager;

	// Open the SCM
	hsrvmanager = OpenSCManager(
                NULL,                   // machine (NULL == local)
                NULL,                   // database (NULL == default)
                SC_MANAGER_ALL_ACCESS   // access required
                );
	if (hsrvmanager)
	{
		hservice = OpenService(hsrvmanager, WIREGLSERVICENAME, SERVICE_ALL_ACCESS);

		if (hservice != NULL)
		{
			SERVICE_STATUS status;

			/* Stop the service */
			if (ControlService(hservice, SERVICE_CONTROL_STOP, &status))
			{
				while(QueryServiceStatus(hservice, &status))
				{
					if (status.dwCurrentState == SERVICE_STOP_PENDING)
						Sleep(1000);
					else
						break;
				}
			}

			/*Delete the service */
			if(!DeleteService(hservice))
				return 0;
			CloseServiceHandle(hservice);
		}
		else
			wireGLError("Unable to find the WireGL service.");

		CloseServiceHandle(hsrvmanager);
	}
	else
		wireGLError("Unable to open the service manager.");

	return 1;
}


#endif /*WINDOWS*/
