/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_pipe.h"
#include "wiregl/include/wiregl_util.h"

typedef struct _wqnode {
	WireGLWorkQueue *q;
	struct _wqnode *next;
} wqnode;

typedef struct {
	int count;
	int num_waiting;
	WireGLWorkQueue **waiting;
} WireGLPAPIBarrier;

typedef struct {
	int count;
	wqnode *waiting, *tail;
} WireGLPAPISemaphore;

WireGLHashTable *PAPIbarriers, *PAPIsemaphores;

void __decodeBarrierCreate( void )
{
	GLuint name, count;
	WireGLPAPIBarrier *barrier;

	name = READ_DATA( 0, GLuint );
	count = READ_DATA( 4, GLuint );
	INCR_DATA_PTR( 8 );

	barrier = (WireGLPAPIBarrier *) 
		wireGLHashtableSearch( PAPIbarriers, name );

	if ( barrier == NULL )
	{
		barrier = (WireGLPAPIBarrier *) wireGLAlloc( sizeof(*barrier) );
		barrier->count = (GLint) count;
		barrier->num_waiting = 0;
		barrier->waiting = (WireGLWorkQueue **) 
			wireGLAlloc( count * sizeof(*(barrier->waiting)) );

		wireGLHashtableAdd( PAPIbarriers, name, barrier );
	}
	else
	{
		/* HACK -- this allows everybody to create a barrier, and all
           but the first creation are ignored, assuming the count
           match. */
		if ( barrier->count != (GLint) count )
		{
			wireGLError( "Barrier name=%u created with count=%u, but already "
						 "exists with count=%u", name, count, barrier->count );
		}
	}
}

void __decodeBarrierDestroy( void )
{
	GLuint name;

	name = READ_DATA( 0, GLuint );
	INCR_DATA_PTR( 4 );

	wireGLSimpleError( "NO DESTROY FOR YOU!  (name=%u)", name );
}

void __decodeBarrierExec( void )
{
	GLuint name;
	WireGLPAPIBarrier *barrier;
	WireGLWorkQueue *q;

	name = READ_DATA( 0, GLuint );
	INCR_DATA_PTR( 4 );

	//	printf ("Barrier exec: %d\n", name );
	barrier = (WireGLPAPIBarrier *) 
		wireGLHashtableSearch( PAPIbarriers, name );
	if ( barrier == NULL )
	{
		wireGLSimpleError( "No such barrier: %d", name );
	}
	q = __currentWorkQueue();
	barrier->waiting[barrier->num_waiting++] = q;
	q->blocked = 1;
	if ( barrier->num_waiting == barrier->count )
	{
		int i;
		for ( i = 0; i < barrier->count; i++ )
		{
			barrier->waiting[i]->blocked = 0;
		}
		barrier->num_waiting = 0;
	}
}

void __decodeSemaphoreCreate( void )
{
	GLuint name, count;
	WireGLPAPISemaphore *sema;

	name = READ_DATA( 0, GLuint );
	count = READ_DATA( 4, GLuint );
	INCR_DATA_PTR( 8 );

	sema = (WireGLPAPISemaphore *) wireGLAlloc( sizeof( *sema ) );
	wireGLHashtableAdd( PAPIsemaphores, name, sema );
	sema->count = count;
	sema->waiting = sema->tail = NULL;
}

void __decodeSemaphoreDestroy( void )
{
	GLuint name;

	name = READ_DATA( 0, GLuint );
	INCR_DATA_PTR( 4 );

	wireGLSimpleError( "NO DESTROY FOR YOU! (name=%u)", name );
}

void __decodeSemaphoreP( void )
{
	GLuint name;
	WireGLPAPISemaphore *sema;

	name = READ_DATA( 0, GLuint );
	INCR_DATA_PTR( 4 );
	sema = (WireGLPAPISemaphore *) wireGLHashtableSearch( PAPIsemaphores, name );
	if (!sema)
		wireGLSimpleError( "No such semaphore: %d", name );
	if (sema->count)
		sema->count--;
	else
	{
		wqnode *node;
		WireGLWorkQueue *q = __currentWorkQueue();
		q->blocked = 1;
		node = (wqnode *) wireGLAlloc( sizeof( *node ) );
		node->q = q;
		node->next = NULL;
		if (sema->tail)
		{
			sema->tail->next = node;
		}
		else
		{
			sema->waiting = node;
		}
		sema->tail = node;
	}
}

void __decodeSemaphoreV( void )
{
	GLuint name;
	WireGLPAPISemaphore *sema;

	name = READ_DATA( 0, GLuint );
	INCR_DATA_PTR( 4 );

	sema = (WireGLPAPISemaphore *) wireGLHashtableSearch( PAPIsemaphores, name );
	if (!sema)
		wireGLSimpleError( "No such semaphore: %d", name );
	if (sema->waiting)
	{
		wqnode *temp = sema->waiting;
		temp->q->blocked = 0;
		sema->waiting = temp->next;
		wireGLFree( temp );
		if (!sema->waiting)
			sema->tail = NULL;
	}
	else
	{
		sema->count++;
	}
}
