/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif
#include <GL/gl.h>

#include "wiregl/include/wiregl_pipe.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_util.h"
#include "wiregl/include/wiregl_common.h"

#include "wiregl/glcontext/glcontext.h"

WireGLPipe wiregl_pipe;

/* XXX HACK: This should go into a header file, but which one? */
extern GLcurrent current;

static const char *progname = "pipeserver";

#if !defined(WINDOWS)

static void
parseDisplay( char **display )
{
	wiregl_pipe.display_string = strdup( display[0] );
}

static void
parseDirect( char **args )
{
	WIREGL_UNUSED(args);

	if ( !wiregl_pipe.try_direct )
	{
		wireGLSimpleError( "Cannot force both direct and indirect "
						   "rendering." );
	}

	wiregl_pipe.force_direct = 1;
}

static void
parseIndirect( char **args )
{
	WIREGL_UNUSED(args);

	if ( wiregl_pipe.force_direct )
	{
		wireGLSimpleError( "Cannot force both direct and indirect "
						   "rendering." );
	}

	wiregl_pipe.try_direct = 0;
}

static void
parseSync( char **args )
{
	WIREGL_UNUSED(args);

	wiregl_pipe.sync = 1;
}

#else

static void
parseInstall( char **args )
{
	WIREGL_UNUSED(args);
	if (wireGLInstallService())
		fprintf (stderr, "WireGL service installed sucessfully.\n");
	exit (0);
}

static void
parseRemove( char **args )
{
	WIREGL_UNUSED(args);
	if (wireGLRemoveService());
		fprintf (stderr, "WireGL service removed sucessfully.\n");
	exit (0);
}

static void
parseService( char **args )
{
	WIREGL_UNUSED(args);
	wireGLServiceMain();
	exit (0);
}

#endif /*!WINDOWS*/

static void
parseNoGFX( char **args )
{
	WIREGL_UNUSED(args);

	wiregl_pipe.no_graphics = 1;
}

static void
parseDebug( char **args )
{
	WIREGL_UNUSED(args);

	wiregl_pipe.debug = 1;
}

static void 
parsePort( char **port )
{
	int x;

	x = atoi( *port );
	if ( x <= 0 || x > 0xffff )
	{
		wireGLSimpleError( "bad port number: %s", *port );
	}
	wiregl_pipe.port = (unsigned short) x;
}

static void 
parseFullscreen( char **args )
{
	WIREGL_UNUSED(args);

	wiregl_pipe.fullscreen = 1;
}

static void
parseLog( char **fname ) {
	wireGLAssert (*fname);
	wiregl_pipe.log = 1;
	wiregl_pipe.timer = wireGLTimerNewTimer();
	wiregl_pipe.world_time = wireGLTimerNewTimer();
	wireGLStartTimer(wiregl_pipe.world_time);
	wireGLInstrumentToFile (*fname);
	
}

static void
parseDTex( char **args )
{
	WIREGL_UNUSED(args);

	wiregl_pipe.distrib_textures = 1;
}

static void
parseHelp( char **args )
{
	WIREGL_UNUSED(args);

	fprintf( stderr, "usage: %s <options>\n", progname );
	fputs( "options:\n"
#ifndef WINDOWS
		   "  -display <display>    : X display\n"
		   "  -direct               : force direct rendering\n"
		   "  -indirect             : force indirect rendering\n"
		   "  -sync                 : make the X connection synchronous\n"
#else
		   "  -install              : install as service\n"
		   "  -remove               : remove service\n"
		   "  -service              : launch as service (not for command line use)\n"
#endif
		   "  -gldebug, -debug      : check for OpenGL errors\n"
		   "  -p <port>             : port to listen on\n"
		   "  -f, -fullscreen       : fullscreen mode\n"
		   "  -nogfx                : don't create a window for make any OpenGL calls\n"
		   "  -log filename         : log network and gfx activity\n"
		   "  -dtex		    : allow writing distributed-texture files\n"
		   "  -h, -help             : this message\n", stderr );

	exit( 1 );
}

typedef struct {
	char *flag;
	void (*fn)(char **args);
	int numParameters;
} ParseParameterT;

ParseParameterT parse_structs[] = {
#if !defined(WINDOWS)
	{ "-display",    parseDisplay,             1 },
	{ "-direct",     parseDirect,              0 },
	{ "-indirect",   parseIndirect,            0 },
	{ "-sync",       parseSync,                0 },
#else
	{ "-install",    parseInstall,             0 },
	{ "-remove",     parseRemove,              0 },
	{ "-service",    parseService,             0 },
#endif
	{ "-gldebug",    parseDebug,               0 },
	{ "-debug",      parseDebug,               0 },
	{ "-p",          parsePort,                1 },
	{ "-f",          parseFullscreen,          0 },
	{ "-fullscreen", parseFullscreen,          0 },
	{ "-nogfx",      parseNoGFX,               0 },
	{ "-h",          parseHelp,                0 },
	{ "-help",       parseHelp,                0 },
	{ "-log",        parseLog,                 1 },
	{ "-dtex",	 parseDTex,		   1 },
	{ NULL,          NULL,                     0 }
};

static void 
setDefaults( void )
{
	wiregl_pipe.port                = WIREGL_TCPIP_DEFAULT_PORT;
	wiregl_pipe.pipe_num            = -1;
	wiregl_pipe.window_x            = 0;
	wiregl_pipe.window_y            = 0;
	wiregl_pipe.window_width        = 1024;
	wiregl_pipe.window_height       = 768;
	wiregl_pipe.swap_barrier        = 0;
	wiregl_pipe.barrier_conn        = NULL;
	wiregl_pipe.apply_viewtransform = 0;
	wiregl_pipe.extent[0].x1 		= 0;
	wiregl_pipe.extent[0].y1		= 0;
	wiregl_pipe.extent[0].x2		= 0;
	wiregl_pipe.extent[0].y2		= 0;
	wiregl_pipe.num_extents			= 0;
	wiregl_pipe.mural_w  			= 0;
	wiregl_pipe.mural_h  			= 0;
	wiregl_pipe.fullscreen          = 0;
	wiregl_pipe.debug               = 0;
	wiregl_pipe.use_L2              = 0;
	wiregl_pipe.no_graphics         = 0;
	wiregl_pipe.rotate_color        = 0;
	wiregl_pipe.rotate_color_flag   = 0;
	wiregl_pipe.L2.single_buffer    = 0; /* HACK */
	wiregl_pipe.L2.frame_num        = 0;
	wiregl_pipe.vis_send_swaps      = 0;
	wiregl_pipe.vis_sync            = 1;
	wiregl_pipe.distrib_textures    = 0;
#if defined(WINDOWS)
	wiregl_pipe.L2.back_channel.name = "COM2";
#else
	wiregl_pipe.L2.back_channel.name = "/dev/ttyS1";
#endif
	wiregl_pipe.L2.back_channel.baud = 115200;
	wiregl_pipe.L2.back_channel.port = WIREGL_SERIAL_INVALID_PORT;
#if !defined(WINDOWS)
	wiregl_pipe.dpy                 = NULL;
	wiregl_pipe.display_string      = NULL;
	wiregl_pipe.try_direct          = 1;
	wiregl_pipe.force_direct        = 0;
	wiregl_pipe.sync                = 0;
	wiregl_pipe.optimize_bucket     = 0;
#endif
}

static void
parseArgs( int argc, char **argv )
{
	int i, j;

	setDefaults( );

	for ( i = 1; i < argc; i++ )
	{
		for ( j = 0; parse_structs[j].flag; j++ )
		{
			if ( !strcmp( argv[i], parse_structs[j].flag ) )
			{
				if ( i + parse_structs[j].numParameters >= argc )
				{
					wireGLSimpleError( "%s expects %d arguments", 
									   parse_structs[j].flag,
									   parse_structs[j].numParameters );
				}
				
				parse_structs[j].fn( &argv[i+1] );
				i += parse_structs[j].numParameters;
				break;
			}
		}

		if( !parse_structs[j].flag )
		{
			wireGLSimpleError( "bad argument \"%s\"", argv[i] );
		}
	}
}

int 
main( int argc, char *argv[] )
{
	GLconfig *c;

	progname = argv[0];

	wireGLInitInstrumentation( TO_SCREEN );
	wireGLInitOS( );
	wireGLNetInit( wireGLPipeserverReceive, NULL );
	wireGLInitWarnings( getenv( "WIREGL_WARN_LEVEL" ),
						( getenv( "WIREGL_GUI_ERROR" ) != NULL ) );

	parseArgs( argc, argv );

	c = __glconfig_CreateDefaultConfig( );
	c->mode = (GLenum) GLCONFIG_TRACK;
	c->apply_viewtransform = 0;
	c->current = &current;
	__glInit( c );

	wireGLInitNormalLookupTable( );

	PAPIbarriers = wireGLAllocHashtable( );
	PAPIsemaphores = wireGLAllocHashtable( );

	wireGLBecomeServer( wiregl_pipe.port, wireGLPipeserverConnect );

	//wireGLChangeThreadPriority( WIREGL_HIGH_PRIORITY );
	wireGLRenderRemoteStreams( );

	return 0;
}
