/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif
#include <GL/gl.h>
#include "wiregl/glcontext/glcontext.h"
#include "wiregl/include/wiregl_pipe.h"

typedef struct {
	GLuint window_width;
	GLuint window_height;
	GLuint left;
	GLuint right;
	GLuint top;
	GLuint bottom;
	GLuint texture_size_left;
	GLuint texture_size_right;
	GLuint texture_size_top;
	GLuint texture_size_bottom;
	GLuint texture_id_left;
	GLuint texture_id_right;
	GLuint texture_id_top;
	GLuint texture_id_bottom;
	GLuint texture_id_overlay;
	GLfloat cp_left[4];
	GLfloat cp_right[4];
	GLfloat cp_top[4];
	GLfloat cp_bottom[4];
	GLboolean dofeather;
	GLboolean dooverlay;
} featherstate;

featherstate fs;

/*
 * muralFeatherBezierEval - Evaluates a Bezier curve at a given t using control
 * points specified in cp.
 */
static void __featherBezierEval( GLfloat t, GLfloat* cp, GLfloat* out ) 
{
  float tSquared = t*t;
  float oneMinusT = 1 - t;
  float oneMinusTSquared = oneMinusT*oneMinusT;
  *out = (oneMinusTSquared*oneMinusT*cp[0] +
	  3*t*oneMinusTSquared*cp[1] +
	  3*tSquared*oneMinusT*cp[2] +
	  tSquared*t*cp[3]);
}

static void __featherRestoreState( void )
{
	GLcontext *g = __glGetCurrentContext();

	glMatrixMode( GL_PROJECTION );
	glPopMatrix();
	
	glMatrixMode( GL_MODELVIEW );
	glPopMatrix();
	
	if (g->polygon.cullface)
		glEnable(GL_CULL_FACE);
	if (g->texture.enabled1d)
		glEnable(GL_TEXTURE_1D);
	if (g->texture.enabled2d)
		glEnable(GL_TEXTURE_2D);
	if (g->lighting.lighting)
		glEnable(GL_LIGHTING);
	if (g->buffer.depthtest)
		glEnable(GL_DEPTH_TEST);
	if (g->viewport.scissortest)
		glEnable(GL_SCISSOR_TEST);
	if (!g->buffer.blend)
		glDisable(GL_BLEND);
/*
	if (g->?????)
		COLOR_MATERIAL???
*/
	if (g->texture.envmode != GL_REPLACE)
		glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, g->texture.envmode);
	if (g->buffer.blendsrc != GL_SRC_ALPHA ||
		g->buffer.blenddst != GL_ONE_MINUS_SRC_ALPHA)
		glBlendFunc( g->buffer.blendsrc, g->buffer.blenddst );
	if (g->trans.mode != GL_MODELVIEW)
		glMatrixMode(g->trans.mode);
	
	glBindTexture(GL_TEXTURE_1D, g->texture.currenttexture1dname);
	glViewport(g->viewport.v_x, g->viewport.v_y, g->viewport.v_w, g->viewport.v_h);
}

/* __featherInit - Initializes the OpenGL state in preparation for doing
 * the feathering. First, any existing OpenGL state is disabled to prevent
 * interference with the feathering code. Next, an orthographic projection
 * is loaded into the projection matrix. For the top and right sides, this
 * matrix is backwards so the same code can draw on two different sides.
 * Then, the shading model is set to flat, and the blending function is set
 * to a standard function. Finally, the viewport is set.
 */

static void __featherInit( void )
{
	glDisable( GL_SCISSOR_TEST );
	glDisable( GL_COLOR_MATERIAL );
	glDisable( GL_CULL_FACE );
	glDisable( GL_TEXTURE_1D );
	glDisable( GL_TEXTURE_2D );
	glDisable( GL_LIGHTING );
	glDisable( GL_DEPTH_TEST );

	glMatrixMode( GL_PROJECTION );
	glPushMatrix();
	glLoadIdentity();

	glMatrixMode( GL_MODELVIEW );
	glPushMatrix();
	glLoadIdentity();

	glEnable( GL_BLEND );

	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE );
	glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
}

/* __drawHorizontal/VerticalStrips - Draws the actual feathering strips. Does so by generating
 * an 1D texture of alpha values that is placed over a single quad.
 */


static void __drawHorizontalStrip( void )
{
	glBegin( GL_QUADS );
	glTexCoord1f(1.0);
	glVertex2f(-1.0, -1.0);
	glTexCoord1f(1.0);
	glVertex2f(1.0, -1.0);
	glTexCoord1f(0.0);
	glVertex2f(1.0, 1.0);
	glTexCoord1f(0.0);
	glVertex2f(-1.0, 1.0);
	glEnd();
}

static void __drawVerticalStrip( void )
{
	glBegin( GL_QUADS );
	glTexCoord1f(1.0);
	glVertex2f(-1.0, -1.0);
	glTexCoord1f(0.0);
	glVertex2f(1.0, -1.0);
	glTexCoord1f(0.0);
	glVertex2f(1.0, 1.0);
	glTexCoord1f(1.0);
	glVertex2f(-1.0, 1.0);
	glEnd();
}

/* __drawFeatheringStrips( ... )
 * -----------------------------
 *
 * This function feathers the edges of a window for use in a multi-projector display. It does
 * so by fading the image to black according to the Bezier curve passed into the function.
 *
 * An example way of calling this function would be:
 *
 *	GLfloat size[2] = {window->width, window->height};
 *	GLfloat cp[4] = {1.0, 0.9, 0.7, 0.0};
 *
 *	__drawFeatheringStrips(size, 30, 1.0, 0.0, 500, LEFT, cp);
 *	__drawFeatheringStrips(size, 30, 1.0, 0.0, 500, TOP, cp);
 *	__drawFeatheringStrips(size, 30, 1.0, 0.0, 500, RIGHT, cp);
 *	__drawFeatheringStrips(size, 30, 1.0, 0.0, 500, BOTTOM, cp);
 *
 * windowSize - Size of the window in pixels. First entry is width, and second entry is height.
 *
 * featherSize - Number of pixels to feather.
 *
 * start/endAlpha - Starting and ending alpha values for the Bezier curve.
 *
 * dice - Number of slices to draw.
 *
 * orient - Orientation of the feathering. One of: LEFT (0), BOTTOM (1), RIGHT (2), or TOP (3).
 *
 * cp - Control points for the Bezier curve. Four points are expected for a cubic curve.
 *
 */

void wireGLDrawFeatheringStrips( void )
{

	/* Is feathering turned on? */
	if (!fs.dofeather && !fs.dooverlay)
		return;

	__featherInit();

	if (fs.dooverlay)
	{
		glEnable(GL_TEXTURE_2D);
		glViewport (0, 0, fs.window_width, fs.window_height);
		glBindTexture(GL_TEXTURE_2D, fs.texture_id_overlay);

		glBegin(GL_QUADS);
			glTexCoord2d(1.0, 1.0);
			glVertex2f(1.0f, 1.0f);
			glTexCoord2d(0.0, 1.0);
			glVertex2f(-1.0f, 1.0f);
			glTexCoord2d(0.0, 0.0);
			glVertex2f(-1.0f, -1.0f);
			glTexCoord2d(1.0, 0.0);
			glVertex2f(1.0f, -1.0f);
		glEnd();

		glDisable(GL_TEXTURE_2D);
	}

	if (fs.dofeather)
	{
		glEnable(GL_TEXTURE_1D);

		if (fs.left) {
			glViewport (0, 0, fs.left, fs.window_height);
			glBindTexture(GL_TEXTURE_1D, fs.texture_id_left);
			glScalef(-1.0f, -1.0f, 1.0f);
			__drawVerticalStrip( );
			glLoadIdentity();
		}
		if (fs.right) {
			glViewport (fs.window_width - fs.right, 0, fs.right, fs.window_height);
			glBindTexture(GL_TEXTURE_1D, fs.texture_id_right);
			__drawVerticalStrip( );
		}
		if (fs.top) {
			glViewport (0, fs.window_height - fs.top, fs.window_width, fs.top);
			glBindTexture(GL_TEXTURE_1D, fs.texture_id_top);
			__drawHorizontalStrip( );
		}
		if (fs.bottom) {
			glViewport (0, 0, fs.window_width, fs.bottom);
			glBindTexture(GL_TEXTURE_1D, fs.texture_id_bottom);
			glScalef(-1.0f, -1.0f, 1.0f);
			__drawHorizontalStrip( );
			glLoadIdentity();
		}

		glDisable(GL_TEXTURE_1D);
	}

	__featherRestoreState();
}

GLuint __generateOverlay( char *fname )
{
	GLuint featherid;
	GLint currentid;

	glGenTextures(1, &featherid);
	glGetIntegerv(GL_TEXTURE_BINDING_2D, &currentid);
	glBindTexture(GL_TEXTURE_2D, featherid);

	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );

	glPushAttrib(GL_PIXEL_MODE_BIT);
	glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );

	wireGLLoadTARGATexture(fname);

	glPopAttrib();
	glBindTexture(GL_TEXTURE_2D, currentid);

	return featherid;
}

GLuint __generateTexture( int size, float *cp )
{
	GLfloat *texture, alpha;
	int i;
	GLuint featherid;
	GLint currentid;

	texture = (GLfloat*)malloc(size * sizeof(GLfloat));

	for( i = 0; i < size; i++ )
	{
		__featherBezierEval( ( GLfloat ) i / ( size - 1 ), cp, &alpha);
		texture[i] = alpha;
	}

	glGenTextures(1, &featherid);
	glGetIntegerv(GL_TEXTURE_BINDING_1D, &currentid);
	glBindTexture(GL_TEXTURE_1D, featherid);

	glTexParameteri( GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP );
	glTexParameteri( GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
	glTexParameteri( GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );

	glPushAttrib(GL_PIXEL_MODE_BIT);
	glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );

	glTexImage1D( GL_TEXTURE_1D, 0, GL_RGBA, size, 0, GL_ALPHA, GL_FLOAT, texture);

	glPopAttrib();
	glBindTexture(GL_TEXTURE_1D, currentid);
	free(texture);

	return featherid;
}

GLuint __getpow2 (GLuint size) {
	GLuint a, sizepow2=0;

	if (size == 0)
		return 0;

	/* Get the largest power of 2 */
	for (a=1; a; a = a<<1) {
		if (size & a)
			sizepow2 = a;
	}

	/* Set texturesize to be the largest 
	** power of 2 greater than texturesize
	*/
	if (size | ~sizepow2) {
		size = sizepow2<<1;
	}

	return size;
}

void wireGLConfigureFeathering ( int windowSize_x, int windowSize_y,
								 WireGLFeatherParam *param )
{
	fs.window_width = windowSize_x;
	fs.window_height = windowSize_y;
	fs.left   = param->left_size;
	fs.right  = param->right_size;
	fs.top    = param->top_size;
	fs.bottom = param->bottom_size;

	fs.cp_left[0] = param->left[0];
	fs.cp_left[1] = param->left[1];
	fs.cp_left[2] = param->left[2];
	fs.cp_left[3] = param->left[3];

	fs.cp_right[0] = param->right[0];
	fs.cp_right[1] = param->right[1];
	fs.cp_right[2] = param->right[2];
	fs.cp_right[3] = param->right[3];

	fs.cp_top[0] = param->top[0];
	fs.cp_top[1] = param->top[1];
	fs.cp_top[2] = param->top[2];
	fs.cp_top[3] = param->top[3];

	fs.cp_bottom[0] = param->bottom[0];
	fs.cp_bottom[1] = param->bottom[1];
	fs.cp_bottom[2] = param->bottom[2];
	fs.cp_bottom[3] = param->bottom[3];

	fs.dooverlay = (GLboolean) ( param->overlay[0] != 0 );

	if(fs.dooverlay)
		fs.texture_id_overlay = __generateOverlay( param->overlay );

	if (!fs.left && !fs.right && !fs.top && !fs.bottom) {
		fs.dofeather = GL_FALSE;
		return;
	}
	fs.dofeather = GL_TRUE;

	fs.texture_size_left = __getpow2 (fs.left);
	fs.texture_size_right = __getpow2 (fs.right);
	fs.texture_size_top = __getpow2 (fs.top);
	fs.texture_size_bottom = __getpow2 (fs.bottom);

	if (fs.texture_size_left > 256)		fs.texture_size_left = 256;
	if (fs.texture_size_right > 256)	fs.texture_size_right = 256;
	if (fs.texture_size_top > 256)		fs.texture_size_top = 256;
	if (fs.texture_size_bottom > 256)	fs.texture_size_bottom = 256;

	if (fs.texture_size_left)
		fs.texture_id_left	= __generateTexture (fs.texture_size_left, (float *) fs.cp_left);
	if (fs.texture_size_right)
		fs.texture_id_right = __generateTexture (fs.texture_size_right, (float *) fs.cp_right);
	if (fs.texture_size_top)
		fs.texture_id_top	= __generateTexture (fs.texture_size_top, (float *) fs.cp_top);
	if (fs.texture_size_bottom)
		fs.texture_id_bottom = __generateTexture (fs.texture_size_bottom, (float *) fs.cp_bottom);
}
