/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <stdio.h>

#include "wiregl/glcontext/glcontext.h"
#include "wiregl/include/wiregl_util.h"
#include "wiregl/include/wiregl_client.h"

BOOL WINAPI
DllMain( HINSTANCE hinstDLL, DWORD fdwReason, LPVOID lpvReserved )
{
        WIREGL_UNUSED(hinstDLL);
        WIREGL_UNUSED(fdwReason);
        WIREGL_UNUSED(lpvReserved);
        return 1;
}

static int use_system_gl = 0;

static struct {
	int   (WINAPI *wglChoosePixelFormat)( HDC, CONST PIXELFORMATDESCRIPTOR * );
	BOOL  (WINAPI *wglCopyContext)( HGLRC, HGLRC, UINT );
	HGLRC (WINAPI *wglCreateContext)( HDC );
	HGLRC (WINAPI *wglCreateLayerContext)( HDC, int );
	BOOL  (WINAPI *wglDeleteContext)( HGLRC );
	BOOL  (WINAPI *wglDescribeLayerPlane)( HDC, int, int, UINT, LPLAYERPLANEDESCRIPTOR );
	int   (WINAPI *wglDescribePixelFormat)( HDC, int, UINT, LPPIXELFORMATDESCRIPTOR );
	HGLRC (WINAPI *wglGetCurrentContext)( void );
	HDC   (WINAPI *wglGetCurrentDC)( void );
	int   (WINAPI *wglGetLayerPaletteEntries)( HDC, int, int, int, CONST COLORREF * );
	int   (WINAPI *wglGetPixelFormat)( HDC );
	PROC  (WINAPI *wglGetProcAddress)( LPCSTR );
	BOOL  (WINAPI *wglMakeCurrent)( HDC, HGLRC );
    BOOL  (WINAPI *wglRealizeLayerPalette)( HDC, int, BOOL );
	int   (WINAPI *wglSetLayerPaletteEntries)( HDC, int, int, int, CONST COLORREF * );
	int   (WINAPI *wglSetPixelFormat)( HDC hdc, int, const PIXELFORMATDESCRIPTOR * );
	BOOL  (WINAPI *wglShareLists)( HGLRC, HGLRC );
	BOOL  (WINAPI *wglSwapBuffers)( HDC );
	BOOL  (WINAPI *wglSwapLayerBuffers)( HDC, UINT );
	DWORD (WINAPI *wglSwapMultipleBuffers)( UINT, CONST void * );
    BOOL  (WINAPI *wglUseFontBitmapsA)( HDC, DWORD, DWORD, DWORD );
    BOOL  (WINAPI *wglUseFontBitmapsW)( HDC, DWORD, DWORD, DWORD );
	BOOL  (WINAPI *wglUseFontOutlinesA)( HDC, DWORD, DWORD, DWORD, FLOAT, FLOAT, int, LPGLYPHMETRICSFLOAT );
	BOOL  (WINAPI *wglUseFontOutlinesW)( HDC, DWORD, DWORD, DWORD, FLOAT, FLOAT, int, LPGLYPHMETRICSFLOAT );
} fptable;

#define X(a)	{ # a, (GLfunc *) &fptable.##a }

GLjumptable wgl_functions[] = {
    X(wglChoosePixelFormat), 
    X(wglCopyContext),
    X(wglCreateContext),
    X(wglCreateLayerContext),
    X(wglDeleteContext),
    X(wglDescribeLayerPlane),
    X(wglDescribePixelFormat),
    X(wglGetCurrentContext),
    X(wglGetCurrentDC),
    X(wglGetLayerPaletteEntries),
    X(wglGetPixelFormat),
    X(wglGetProcAddress),
    X(wglMakeCurrent),
    X(wglRealizeLayerPalette),
    X(wglSetLayerPaletteEntries),
    X(wglSetPixelFormat),
    X(wglShareLists),
    X(wglSwapBuffers),
    X(wglSwapLayerBuffers),
    X(wglSwapMultipleBuffers),
    X(wglUseFontBitmapsA),
    X(wglUseFontBitmapsW),
    X(wglUseFontOutlinesA),
    X(wglUseFontOutlinesW),
    { NULL, NULL }
};

#undef X

static void WINAPI
MissingFunction( void )
{
	wireGLError( "Called a missing function!" );
}

void WINAPI
wireGLUseSystemGL( void )
{
	char opengl32[256];
	WireGLDSO *dso;
	int i;

	GetSystemDirectory( opengl32, sizeof(opengl32) );
	strcat( opengl32, "\\opengl32.dll" );
	dso = wireGLDSOOpen( opengl32 );

	for ( i = 0; wgl_functions[i].name; i++ )
	{
		WireGLDSOFunc func = wireGLDSOGet( dso, wgl_functions[i].name );
		*wgl_functions[i].func = (GLfunc) func;
	}

	for ( i = 0; wiregl_im_api[i].name; i++ ) 
	{
		WireGLDSOFunc func = wireGLDSOGetNoError( dso, wiregl_im_api[i].name );
		if ( func )
			*wiregl_im_api[i].func = (GLfunc) func;
		else
			*wiregl_im_api[i].func = (GLfunc) MissingFunction;
	}

#if 0
	/* don't close the DSO, since we'll be calling into it... DUH */
	wireGLDSOClose( dso );
#endif

	use_system_gl = 1;
}

int WINAPI
wglChoosePixelFormat_prox( HDC hdc, CONST PIXELFORMATDESCRIPTOR *pfd )
{
    DWORD okayFlags;

	if ( use_system_gl )
	{
		return fptable.wglChoosePixelFormat( hdc, pfd );
	}

    if ( pfd->nSize != sizeof(*pfd) || pfd->nVersion != 1 ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat: bad pfd\n" );
        return 0;
    }

    okayFlags = ( PFD_DRAW_TO_WINDOW        |
                  PFD_SUPPORT_GDI           |
                  PFD_SUPPORT_OPENGL        |
                  PFD_DOUBLEBUFFER          |
                  PFD_DOUBLEBUFFER_DONTCARE |
                  PFD_STEREO_DONTCARE       |
                  PFD_DEPTH_DONTCARE        );
    if ( pfd->dwFlags & ~okayFlags ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat: only "
                                           "support flags=0x%x", okayFlags );
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat: you gave "
                                           "me flags=0x%x", pfd->dwFlags );
        return 0;
    }

    if ( pfd->iPixelType != wiregl_pfd.iPixelType ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat: only "
                                           "support RGBA\n" );
        return 0;
    }

    if ( pfd->cColorBits > wiregl_pfd.cColorBits ||
         pfd->cRedBits   > wiregl_pfd.cRedBits  ||
         pfd->cGreenBits > wiregl_pfd.cGreenBits  ||
         pfd->cBlueBits  > wiregl_pfd.cBlueBits  ||
         pfd->cAlphaBits > wiregl_pfd.cAlphaBits ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat: too much "
                      "color precision requested\n" );
        return 0;
    }

    if ( pfd->cAccumBits      > wiregl_pfd.cAccumBits ||
         pfd->cAccumRedBits   > wiregl_pfd.cAccumRedBits ||
         pfd->cAccumGreenBits > wiregl_pfd.cAccumGreenBits ||
         pfd->cAccumBlueBits  > wiregl_pfd.cAccumBlueBits ||
         pfd->cAccumAlphaBits > wiregl_pfd.cAccumAlphaBits ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat: asked for "
                      "accumulation buffer, ignoring\n" );
    }

    if ( pfd->cDepthBits > wiregl_pfd.cDepthBits ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat; asked for "
                      "too many depth bits\n" );
        return 0;
    }

    if ( pfd->cStencilBits > wiregl_pfd.cStencilBits ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat: asked for "
                      "too many stencil bits\n" );
        return 0;
    }

    if ( pfd->cAuxBuffers > wiregl_pfd.cAuxBuffers ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat: asked for "
                      "aux buffers\n" );
        return 0;
    }

    if ( pfd->iLayerType != wiregl_pfd.iLayerType ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglChoosePixelFormat: asked for "
                      "a strange layer\n" );
        return 0;
    }

    return 1;
}

BOOL WINAPI
wglSetPixelFormat_prox( HDC hdc, int pixelFormat, 
						CONST PIXELFORMATDESCRIPTOR *pdf )
{
	if ( use_system_gl )
	{
		return fptable.wglSetPixelFormat( hdc, pixelFormat, pdf );
	}

    if ( pixelFormat != 1 ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglSetPixelFormat: "
                      "pixelFormat=%d?\n", pixelFormat );
        return 0;
    }

    return 1;
}

BOOL WINAPI
wglDeleteContext_prox( HGLRC hglrc )
{
	if ( use_system_gl )
	{
		return fptable.wglDeleteContext( hglrc );
	}

	wireGLWarning( WIREGL_WARN_DEBUG, "wglDeleteContext: ignoring" );
	return 1;
}

BOOL WINAPI
wglMakeCurrent_prox( HDC hdc, HGLRC hglrc )
{
	if ( use_system_gl )
	{
		return fptable.wglMakeCurrent( hdc, hglrc );
	}

	wireGLMakeCurrent( );

	return 1;
}

HGLRC WINAPI
wglGetCurrentContext_prox( void )
{
	if ( use_system_gl )
	{
		return fptable.wglGetCurrentContext( );
	}

	return (HGLRC) __wiregl_globals.context;
}

HDC WINAPI
wglGetCurrentDC_prox( void )
{
	if ( use_system_gl )
	{
		return fptable.wglGetCurrentDC( );
	}
	
	return __wiregl_globals.client_hdc;
}

int WINAPI
wglGetPixelFormat_prox( HDC hdc )
{
	if ( use_system_gl )
	{
		return fptable.wglGetPixelFormat( hdc );
	}

	/* this is what we call our generic pixelformat, regardless of the HDC */
	return 1;
}

int WINAPI
wglDescribePixelFormat_prox( HDC hdc, int pixelFormat, UINT nBytes,
							 LPPIXELFORMATDESCRIPTOR pfd )
{
	if ( use_system_gl )
	{
		return fptable.wglDescribePixelFormat( hdc, pixelFormat, nBytes, pfd );
	}

    /* if pfd is null, then its a query */
    if ( pfd == NULL )
    {
        /*  Note that we return 2 when we really
        **  only export 1.  This is to fix a 
        **  common glut bug since it assumes that
        **  identifiers start with 0.  LAME!
        */
        return 2;
    }
    
    if ( pixelFormat != 1 ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglDescribePixelFormat: "
                      "pixelFormat=%d?\n", pixelFormat );
        
        if (pfd && nBytes)
            memset (pfd, 0, nBytes);

        return 0;
    }

    if ( nBytes != sizeof(*pfd) ) {
        wireGLWarning( WIREGL_WARN_NOTICE, "wglDescribePixelFormat: "
                      "nBytes=%u?\n", nBytes );
        return 0;
    }

    *pfd = wiregl_pfd;

    /* the max PFD index */
    return 1;
}

BOOL WINAPI
wglShareLists_prox( HGLRC hglrc1, HGLRC hglrc2 )
{
	if ( use_system_gl )
	{
		return fptable.wglShareLists( hglrc1, hglrc2 );
	}

	wireGLSimpleError( "wglShareLists: unsupported" );
	return 0;
}

HGLRC WINAPI
wglCreateContext_prox( HDC hdc )
{
	if ( use_system_gl )
	{
		return fptable.wglCreateContext( hdc );
	}

	__wiregl_globals.client_hdc = hdc;
	__wiregl_globals.client_hwnd = WindowFromDC( hdc );

	wireGLCreateContext( 0 );

	/* hack hack hack */
	return (HGLRC) 0x10000;
}

BOOL WINAPI
wglSwapBuffers_prox( HDC hdc )
{
	if ( use_system_gl )
	{
		return fptable.wglSwapBuffers( hdc );
	}

	wireGLSwapBuffers( );
	return 1;
}

BOOL WINAPI
wglCopyContext_prox( HGLRC src, HGLRC dst, UINT mask )
{
	if ( use_system_gl )
	{
		return fptable.wglCopyContext( src, dst, mask );
	}

	wireGLSimpleError( "wglCopyContext: unsupported" );
	return 0;
}

HGLRC WINAPI
wglCreateLayerContext_prox( HDC hdc, int layerPlane )
{
	if ( use_system_gl )
	{
		return fptable.wglCreateLayerContext( hdc, layerPlane );
	}

	wireGLSimpleError( "wglCreateLayerContext: unsupported" );
	return 0;
}

PROC WINAPI
wglGetProcAddress_prox( LPCSTR name )
{
	if ( use_system_gl )
	{
		return fptable.wglGetProcAddress( name );
	}

	wireGLSimpleError( "wglGetProcAddress: unsupported" );
	return (PROC) 0;
}

BOOL WINAPI
wglUseFontBitmapsA_prox( HDC hdc, DWORD first, DWORD count, DWORD listBase )
{
	if ( use_system_gl )
	{
		return fptable.wglUseFontBitmapsA( hdc, first, count, listBase );
	}

	wireGLSimpleError( "wglUseFontBitmapsA: unsupported" );
	return 0;
}

BOOL WINAPI
wglUseFontBitmapsW_prox( HDC hdc, DWORD first, DWORD count, DWORD listBase )
{
	if ( use_system_gl )
	{
		return fptable.wglUseFontBitmapsW( hdc, first, count, listBase );
	}

	wireGLSimpleError( "wglUseFontBitmapsW: unsupported" );
	return 0;
}

BOOL WINAPI
wglDescribeLayerPlane_prox( HDC hdc, int pixelFormat, int layerPlane,
							UINT nBytes, LPLAYERPLANEDESCRIPTOR lpd )
{
	if ( use_system_gl )
	{
		return fptable.wglDescribeLayerPlane( hdc, pixelFormat, layerPlane,
											  nBytes, lpd );
	}

	wireGLSimpleError( "wglDescribeLayerPlane: unimplemented" );
	return 0;
}

int WINAPI
wglSetLayerPaletteEntries_prox( HDC hdc, int layerPlane, int start,
								int entries, CONST COLORREF *cr )
{
	if ( use_system_gl )
	{
		return fptable.wglSetLayerPaletteEntries( hdc, layerPlane, start,
												  entries, cr );
	}

	wireGLSimpleError( "wglSetLayerPaletteEntries: unsupported" );
	return 0;
}

int WINAPI
wglGetLayerPaletteEntries_prox( HDC hdc, int layerPlane, int start,
								int entries, COLORREF *cr )
{
	if ( use_system_gl )
	{
		return fptable.wglGetLayerPaletteEntries( hdc, layerPlane, start,
												  entries, cr );
	}

	wireGLSimpleError( "wglGetLayerPaletteEntries: unsupported" );
	return 0;
}

BOOL WINAPI
wglRealizeLayerPalette_prox( HDC hdc, int layerPlane, BOOL realize )
{
	if ( use_system_gl )
	{
		return fptable.wglRealizeLayerPalette( hdc, layerPlane, realize );
	}

	wireGLSimpleError( "wglRealizeLayerPalette: unsupported" );
	return 0;
}

DWORD WINAPI
wglSwapMultipleBuffers_prox( UINT a, CONST void *b )
{
	if ( use_system_gl )
	{
		return fptable.wglSwapMultipleBuffers( a, b );
	}

	wireGLSimpleError( "wglSwapMultipleBuffer: unsupported" );
	return 0;
}

BOOL WINAPI
wglUseFontOutlinesA_prox( HDC hdc, DWORD first, DWORD count, DWORD listBase,
						  FLOAT deviation, FLOAT extrusion, int format,
						  LPGLYPHMETRICSFLOAT gmf )
{
	if ( use_system_gl )
	{
		return fptable.wglUseFontOutlinesA( hdc, first, count, listBase,
											deviation, extrusion, format,
											gmf );
	}

	wireGLSimpleError( "wglUseFontOutlinesA: unsupported" );
	return 0;
}

BOOL WINAPI
wglUseFontOutlinesW_prox( HDC hdc, DWORD first, DWORD count, DWORD listBase,
						  FLOAT deviation, FLOAT extrusion, int format,
						  LPGLYPHMETRICSFLOAT gmf )
{
	if ( use_system_gl )
	{
		return fptable.wglUseFontOutlinesW( hdc, first, count, listBase,
											deviation, extrusion, format,
											gmf );
	}

	wireGLSimpleError( "wglUseFontOutlinesW: unsupported" );
	return 0;
}

BOOL WINAPI
wglSwapLayerBuffers_prox( HDC hdc, UINT planes )
{
	if ( use_system_gl )
	{
		return fptable.wglSwapLayerBuffers( hdc, planes );
	}

	wireGLSimpleError( "wglSwapLayerBuffers: unsupported" );
	return 0;
}
