/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifdef WINDOWS

#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_util.h"
#include "wiregl/glcontext/glcontext.h"

#include "geometry.h"

#include <math.h>

static __inline void __wiregl_handleVertexData( const GLfloat *x, const GLfloat *y, const GLfloat *z )
{
	unsigned char *data_ptr;
	unsigned int *data;
	unsigned int *clear;
	int outData[3];

	/* Make sure the compression data is valid. */
	
	if(!__wiregl_compressiondatavalid)
		__wiregl_GuessCompressData(x, y, z);

	/* Now do the compression. It's in assembly so it's fast. The computation is:
	 * outData[i] = (unsigned int)((v[i] + __wiregl_fastcompressdata.add[i]) * __wiregl_fastcompressdata.mult[i]);
	 * if(outData[i] & nMask[i])
	 *     goto CompressionNotSafe;
	 */

	_asm {
		cmp __wiregl_fastcompressdata.compressionFailed, 1
		je  CompressionNotSafe
		mov ebx, x
		mov esi, y
		mov edi, z
		fld [ebx]
		fld [esi]
		fld [edi]
		fld __wiregl_fastcompressdata.add[8]
		fadd
		fld __wiregl_fastcompressdata.mult[8]
		fmul
		fistp dword ptr outData[8]
		fld __wiregl_fastcompressdata.add[4]
		fadd
		fld __wiregl_fastcompressdata.mult[4]
		fmul
		fistp dword ptr outData[4]
		fld __wiregl_fastcompressdata.add[0]
		fadd
		fld __wiregl_fastcompressdata.mult[0]
		fmul
		fistp dword ptr outData[0]
		mov eax, dword ptr outData[8]
		mov ebx, dword ptr outData[4]
		mov ecx, dword ptr outData[0]
		test eax, dword ptr __wiregl_fastcompressdata.nmask[8]
		jnz  CompressionNotSafe
		test ebx, dword ptr __wiregl_fastcompressdata.nmask[4]
		jnz  CompressionNotSafe
		test ecx, dword ptr __wiregl_fastcompressdata.nmask[0]
		jnz  CompressionNotSafe
	}

	GET_BUFFERED_COUNT_POINTER( __wiregl_fastcompressdata.bytes );

	clear = data = (unsigned int *)data_ptr;

	/* Make sure the buffer is filled with 0's so we can use OR. */

	__asm {
		mov ecx, __wiregl_fastcompressdata.bytes
		mov ebx, clear
		shr ecx, 2
clearLoop:
		mov dword ptr [ebx],0
		add ebx, 4
		dec ecx
		jne clearLoop
	}

	/* Write the data into the bit stream. Only the third group could
	 * be split up, so special-case it. */

	*data = outData[0];
	*data |= outData[1] << __wiregl_fastcompressdata.shiftLeft[0];
	*data |= outData[2] << __wiregl_fastcompressdata.shiftLeft[1];
	if(__wiregl_fastcompressdata.bitsLeft)
		*(data + 1) = outData[2] >> __wiregl_fastcompressdata.bitsLeft;

	WRITE_OPCODE( WIREGL_VERTEX3COMPRESSED_OPCODE );
	return;

CompressionNotSafe:

	/* Compression's failed. Just send the vertex. */

	GET_BUFFERED_COUNT_POINTER( 12 );
	WRITE_DATA( 0, GLfloat, *x );
	WRITE_DATA( 4, GLfloat, *y );
	WRITE_DATA( 8, GLfloat, *z );
	WRITE_OPCODE( WIREGL_VERTEX3F_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( VertexSafeCompressBB3d ) ( GLdouble x, GLdouble y, GLdouble z )
{
	CREATE_3D_FLOATS();
	UPDATE_3D_BBOX();
	__wiregl_handleVertexData(&fx, &fy, &fz);
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( VertexSafeCompressBB3dv ) ( const GLdouble *v )
{
	CREATE_3D_VFLOATS();
	UPDATE_3D_BBOX();
	__wiregl_handleVertexData(&fx, &fy, &fz);
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( VertexSafeCompressBB3f ) ( GLfloat fx, GLfloat fy, GLfloat fz )
{
	UPDATE_3D_BBOX();
	__wiregl_handleVertexData(&fx, &fy, &fz);
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( VertexSafeCompressBB3fv ) ( const GLfloat *v )
{
	CREATE_3D_VFLOATS();
	UPDATE_3D_BBOX();
	__wiregl_handleVertexData(&fx, &fy, &fz);
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( VertexSafeCompressBB3i ) ( GLint x, GLint y, GLint z )
{
	CREATE_3D_FLOATS();
	UPDATE_3D_BBOX();
	__wiregl_handleVertexData(&fx, &fy, &fz);
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( VertexSafeCompressBB3iv ) ( const GLint *v )
{
	CREATE_3D_VFLOATS();
	UPDATE_3D_BBOX();
	__wiregl_handleVertexData(&fx, &fy, &fz);
}

GLapi vertexSafeCompressBBapi[] = {
	{"glVertex2d",  (GLfunc) WIREGL_PACK_FUNCTION( VertexBB2d )},
	{"glVertex2dv", (GLfunc) WIREGL_PACK_FUNCTION( VertexBB2dv )},
	{"glVertex2f",  (GLfunc) WIREGL_PACK_FUNCTION( VertexBB2f )},
	{"glVertex2fv", (GLfunc) WIREGL_PACK_FUNCTION( VertexBB2fv )},
	{"glVertex2i",  (GLfunc) WIREGL_PACK_FUNCTION( VertexBB2i )},
	{"glVertex2iv", (GLfunc) WIREGL_PACK_FUNCTION( VertexBB2iv )},
	{"glVertex2s",  (GLfunc) WIREGL_PACK_FUNCTION( VertexBB2s )},
	{"glVertex2sv", (GLfunc) WIREGL_PACK_FUNCTION( VertexBB2sv )},
	{"glVertex3d",  (GLfunc) WIREGL_PACK_FUNCTION( VertexSafeCompressBB3d )},
	{"glVertex3dv", (GLfunc) WIREGL_PACK_FUNCTION( VertexSafeCompressBB3dv )},
	{"glVertex3f",  (GLfunc) WIREGL_PACK_FUNCTION( VertexSafeCompressBB3f )},
	{"glVertex3fv", (GLfunc) WIREGL_PACK_FUNCTION( VertexSafeCompressBB3fv )},
	{"glVertex3i",  (GLfunc) WIREGL_PACK_FUNCTION( VertexSafeCompressBB3i )},
	{"glVertex3iv", (GLfunc) WIREGL_PACK_FUNCTION( VertexSafeCompressBB3iv )},
	{"glVertex3s",  (GLfunc) WIREGL_PACK_FUNCTION( VertexBB3s )},
	{"glVertex3sv", (GLfunc) WIREGL_PACK_FUNCTION( VertexBB3sv )},
	{"glVertex4d",  (GLfunc) WIREGL_PACK_FUNCTION( VertexBB4d )},
	{"glVertex4dv", (GLfunc) WIREGL_PACK_FUNCTION( VertexBB4dv )},
	{"glVertex4f",  (GLfunc) WIREGL_PACK_FUNCTION( VertexBB4f )},
	{"glVertex4fv", (GLfunc) WIREGL_PACK_FUNCTION( VertexBB4fv )},
	{"glVertex4i",  (GLfunc) WIREGL_PACK_FUNCTION( VertexBB4i )},
	{"glVertex4iv", (GLfunc) WIREGL_PACK_FUNCTION( VertexBB4iv )},
	{"glVertex4s",  (GLfunc) WIREGL_PACK_FUNCTION( VertexBB4s )},
	{"glVertex4sv", (GLfunc) WIREGL_PACK_FUNCTION( VertexBB4sv )},
	{NULL, NULL}};

#endif
