/* opengl_stub/glx.c */

#include <GL/glx.h>

#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_util.h"

#if defined(IRIX) || defined(IRIX64)

  #if defined(_ABI64)
    #define SYSTEM_OPENGL_LIBRARY "/usr/lib64/libGL.so"
  #elif defined(_ABIN32)
    #define SYSTEM_OPENGL_LIBRARY "/usr/lib32/libGL.so"
  #else
    #define SYSTEM_OPENGL_LIBRARY "/usr/lib/libGL.so"
  #endif

#elif defined(Linux)

  #define SYSTEM_OPENGL_LIBRARY "/usr/lib/libGL.so"

#elif defined(AIX)

  #define SYSTEM_OPENGL_LIBRARY "/usr/lib/libGL.a"

#else

  #error Where is the system OpenGL library?

#endif

static int use_system_gl = 0;

static struct {
    XVisualInfo* (*glXChooseVisual)(Display *, int, int *);
    void         (*glXCopyContext)(Display *, GLXContext, GLXContext, GLuint);
    GLXContext   (*glXCreateContext)(Display *, XVisualInfo *, GLXContext, Bool);
    GLXPixmap    (*glXCreateGLXPixmap)(Display *, XVisualInfo *, Pixmap);
    void         (*glXDestroyContext)(Display *, GLXContext);
    void         (*glXDestroyGLXPixmap)(Display *, GLXPixmap);
    int          (*glXGetConfig)(Display *, XVisualInfo *vis, int, int *);
    GLXContext   (*glXGetCurrentContext)(void);
    GLXDrawable  (*glXGetCurrentDrawable)(void);
    Bool         (*glXIsDirect)(Display *, GLXContext);
    Bool         (*glXMakeCurrent)(Display *, GLXDrawable, GLXContext);
    Bool         (*glXQueryExtension)(Display *, int *, int *);
    Bool         (*glXQueryVersion)(Display *, int *, int *);
    void         (*glXSwapBuffers)(Display *, GLXDrawable);
    void         (*glXUseXFont)(Font, int, int, int);
    void         (*glXWaitGL)(void);
    void         (*glXWaitX)(void);
    const char * (*glXQueryExtensionsString)(Display *, int);
    const char * (*glXGetClientString)(Display *, int);
    const char * (*glXQueryServerString)(Display *, int, int);
} fptable;

#define X(a)	{ # a, (GLfunc *) &fptable.##a }

GLjumptable glx_functions[] = {
	X(glXChooseVisual),
	X(glXCopyContext),
	X(glXCreateContext),
	X(glXCreateGLXPixmap),
	X(glXDestroyContext),
	X(glXDestroyGLXPixmap),
	X(glXGetConfig),
	X(glXGetCurrentContext),
	X(glXGetCurrentDrawable),
	X(glXIsDirect),
	X(glXMakeCurrent),
	X(glXQueryExtension),
	X(glXQueryVersion),
	X(glXSwapBuffers),
	X(glXUseXFont),
	X(glXWaitGL),
	X(glXWaitX),
	X(glXQueryExtensionsString),
	X(glXGetClientString),
	X(glXQueryServerString),
	{ NULL, NULL }
};

#undef X

static void
MissingFunction( void )
{
	wireGLError( "Called a missing function!" );
}

void wireGLUseSystemGL( void )
{
	WireGLDSO *dso = wireGLDSOOpen( SYSTEM_OPENGL_LIBRARY );
	int i;

	for ( i = 0; glx_functions[i].name; i++ )
	{
		void *func = wireGLDSOGet( dso, glx_functions[i].name );
		*glx_functions[i].func = (GLfunc) func;
	}

	for ( i = 0; wiregl_im_api[i].name; i++ ) 
	{
		void *func = wireGLDSOGetNoError( dso, wiregl_im_api[i].name );
		if ( func )
			*wiregl_im_api[i].func = (GLfunc) func;
		else
			*wiregl_im_api[i].func = (GLfunc) MissingFunction;
	}

#if 0
	/* don't close the DSO, since we'll be calling into it... DUH */
	wireGLDSOClose( dso );
#endif

	use_system_gl = 1;
}

static XVisualInfo *
ReasonableVisual( Display *dpy, int screen )
{
	int i, n;
	XVisualInfo vis_template, *visual, *best;

	/* find the set of all visuals for this display/screen */
	vis_template.screen = screen;
	visual = XGetVisualInfo( dpy, VisualScreenMask, &vis_template, &n );
	if ( visual == NULL || n < 1 )
	{
		wireGLWarning( WIREGL_WARN_CRITICAL, "glXChooseVisual: "
					   "XGetVisualInfo() found no matches?" );
		return NULL;
	}

	/* now see if we can find a TrueColor/DirectColor visual */
	best = NULL;
	for ( i = 0; i < n; i++ )
	{
		if ( visual[i].class == TrueColor || visual[i].class == DirectColor )
		{
			best = &visual[i];
		}
	}

	if ( best ) {
		XVisualInfo *besttrue, *bestdirect;

		besttrue = bestdirect = NULL;

		/* okay, select the RGB visual with the most depth */
		for ( i = 0; i < n; i++ )
		{
			if ( visual[i].class == TrueColor &&
			       (besttrue == NULL || 
				 (visual[i].depth > besttrue->depth &&
				 visual[i].bits_per_rgb > besttrue->bits_per_rgb )) )
			{
				besttrue = &visual[i];
			}
			if ( visual[i].class == DirectColor &&
			      (bestdirect == NULL ||
				 (visual[i].depth > bestdirect->depth &&
				 visual[i].bits_per_rgb > bestdirect->bits_per_rgb )) )
			{
				bestdirect = &visual[i];
			}
		}
		best = besttrue;
		if(best == NULL ||
			(bestdirect != NULL
			   && bestdirect->bits_per_rgb > best->bits_per_rgb))
		    best = bestdirect;
	}
	else
	{
		/* no RGB visuals, select the deepest colorindex visual */
		/* NOTE: this code is likely never ever executed */
		/* we don't have to check for TrueColor/DirectColor, since
		 * we know none of those exist */
		best = &visual[0];
		for ( i = 1; i < n; i++ )
		{
			if ( visual[i].depth > best->depth )
			{
				best = &visual[i];
			}
		}
	}

	vis_template.screen = screen;
	vis_template.visualid = best->visualid;

	XFree( visual );
	
	visual = XGetVisualInfo( dpy, VisualScreenMask | VisualIDMask,
							 &vis_template, &n );

	if ( visual == NULL || n != 1 )
	{
		wireGLError( "glXChooseVisual: XGetVisualInfo( visualid=%d ) failed!",
					 vis_template.visualid );
		return NULL;
	}

	return visual;
}

XVisualInfo *
glXChooseVisual( Display *dpy, int screen, int *attribList )
{
	int *attrib, wants_rgb;

	if ( use_system_gl )
	{
		return fptable.glXChooseVisual( dpy, screen, attribList );
	}

	wants_rgb = 0;

	for ( attrib = attribList; *attrib != None; attrib++ )
	{
		switch ( *attrib )
		{
		  case GLX_USE_GL:
			/* ignored, this is mandatory */
			break;

		  case GLX_BUFFER_SIZE:
			/* this is for color-index visuals, which we don't support */
			attrib++;
			break;

		  case GLX_LEVEL:
			{
				int level = attrib[1];
				if ( level != 0 )
				{
					wireGLWarning( WIREGL_WARN_NOTICE, "glXChooseVisual: "
								   "level=%d unsupported", level );
					return NULL;
				}
			}
			attrib++;
			break;

		  case GLX_RGBA:
			wants_rgb = 1;
			break;

		  case GLX_DOUBLEBUFFER:
			/* ignored? */
			/* wants_doublebuffer = 1; */
			break;

		  case GLX_STEREO:
			wireGLWarning( WIREGL_WARN_NOTICE, "glXChooseVisual: stereo "
						   "unsupported" );
			return NULL;

		  case GLX_AUX_BUFFERS:
			{
				int aux_buffers = attrib[1];
				if ( aux_buffers != 0 )
				{
					wireGLWarning( WIREGL_WARN_NOTICE, "glXChooseVisual: "
								   "aux_buffers=%d unsupported", aux_buffers );
					return NULL;
				}
			}
			attrib++;
			break;

		  case GLX_RED_SIZE:
		  case GLX_GREEN_SIZE:
		  case GLX_BLUE_SIZE:
		  case GLX_ALPHA_SIZE:
		  case GLX_DEPTH_SIZE:
		  case GLX_STENCIL_SIZE:
			/* ignored */
			attrib++;
			break;

		  case GLX_ACCUM_RED_SIZE:
		  case GLX_ACCUM_GREEN_SIZE:
		  case GLX_ACCUM_BLUE_SIZE:
		  case GLX_ACCUM_ALPHA_SIZE:
			/* unsupported */
			{
				int accum_size = attrib[1];
				if ( accum_size != 0 )
				{
					wireGLWarning( WIREGL_WARN_NOTICE, "glXChooseVisual: "
								   "accum_size=%d unsupported", accum_size );
					return NULL;
				}
			}
			attrib++;
			break;

		  default:
			wireGLWarning( WIREGL_WARN_CRITICAL, "glXChooseVisual: bad "
						   "attrib=0x%x", *attrib );
			return NULL;
		}
	}

	if ( !wants_rgb )
	{
		wireGLWarning( WIREGL_WARN_NOTICE, "glXChooseVisual: didn't request"
					   "RGB visual?" );
		return NULL;
	}

	return ReasonableVisual( dpy, screen );
}

/* There is a problem with glXCopyContext.
** IRIX and Mesa both define glXCopyContext
** to have the mask argument being a 
** GLuint.  XFree 4 and oss.sgi.com
** define it to be an unsigned long.
** Solution: We don't support
** glXCopyContext anyway so we'll just
** #ifdef out the code.
*/

#if 0
void
glXCopyContext( Display *dpy, GLXContext src, GLXContext dst, GLuint mask )
{
	if ( use_system_gl )
	{
		fptable.glXCopyContext( dpy, src, dst, mask );
		return;
	}

	wireGLSimpleError( "Unsupported GLX Call: glXCopyContext()" );
}
#endif

static char myGLXContext[] = "foo";
static char *myGLXCurrentContext = NULL;

GLXContext
glXCreateContext( Display *dpy, XVisualInfo *vis, GLXContext share,
                  Bool direct )
{
	static int first = 1;

	if ( use_system_gl )
	{
		return fptable.glXCreateContext( dpy, vis, share, direct );
	}
	//wireGLCreateContext( );
	if(!first)
		wireGLWarning( WIREGL_WARN_NOTICE, "glXCreateContext() called more than once, but WireGL only allows 1 window!" );
	first = 0;
	return (GLXContext) myGLXContext;
}

GLXPixmap
glXCreateGLXPixmap( Display *dpy, XVisualInfo *vis, Pixmap pixmap )
{
	if ( use_system_gl )
	{
		return fptable.glXCreateGLXPixmap( dpy, vis, pixmap );
	}

	wireGLSimpleError( "Unsupported GLX Call: glXCreateGLXPixmap()" );
	return (GLXPixmap) 0;
}

void
glXDestroyContext( Display *dpy, GLXContext ctx )
{
	if ( use_system_gl )
	{
		fptable.glXDestroyContext( dpy, ctx );
		return;
	}

	wireGLSimpleError( "Unsupported GLX Call: glXDestroyContext()" );
}

void
glXDestroyGLXPixmap( Display *dpy, GLXPixmap pix )
{
	if ( use_system_gl )
	{
		fptable.glXDestroyGLXPixmap( dpy, pix );
		return;
	}

	wireGLSimpleError( "Unsupported GLX Call: glXDestroyGLXPixmap()" );
}

int
glXGetConfig( Display *dpy, XVisualInfo *vis, int attrib, int *value )
{
	if ( use_system_gl )
	{
		return fptable.glXGetConfig( dpy, vis, attrib, value );
	}

    switch ( attrib ) {

      case GLX_USE_GL:
        *value = 1;
        break;

      case GLX_BUFFER_SIZE:
        *value = 32;
        break;

      case GLX_LEVEL:
        *value = 0;
        break;

      case GLX_RGBA:
        *value = 1;
        break;

      case GLX_DOUBLEBUFFER:
        *value = 1;
        break;

      case GLX_STEREO:
        *value = 0;
        break;

      case GLX_AUX_BUFFERS:
        *value = 0;
        break;

      case GLX_RED_SIZE:
        *value = 8;
        break;

      case GLX_GREEN_SIZE:
        *value = 8;
        break;

      case GLX_BLUE_SIZE:
        *value = 8;
        break;

      case GLX_ALPHA_SIZE:
        *value = 8;
        break;

      case GLX_DEPTH_SIZE:
        *value = 24;
        break;

      case GLX_STENCIL_SIZE:
        *value = 8;
        break;

      case GLX_ACCUM_RED_SIZE:
        *value = 0;
        break;

      case GLX_ACCUM_GREEN_SIZE:
        *value = 0;
        break;

      case GLX_ACCUM_BLUE_SIZE:
        *value = 0;
        break;

      case GLX_ACCUM_ALPHA_SIZE:
        *value = 0;
        break;

      default:
        return GLX_BAD_ATTRIBUTE;
    }

    return 0;
}

GLXContext
glXGetCurrentContext( void )
{
	static int first = 1;

	if ( use_system_gl )
	{
		return fptable.glXGetCurrentContext( );
	}

	if(first) {
		wireGLWarning( WIREGL_WARN_NOTICE, "glXGetCurrentContext() is a baldface stub!" );
		first = 0;
	}
	return (GLXContext) myGLXCurrentContext; /* Bogus but non-NULL value */
}

GLXDrawable
glXGetCurrentDrawable( void )
{
	if ( use_system_gl )
	{
		return fptable.glXGetCurrentDrawable( );
	}

	wireGLSimpleError( "Unsupported GLX Call: glXGetCurrentDrawable()" );
	return (GLXDrawable) 0;
}

Bool
glXIsDirect( Display *dpy, GLXContext ctx )
{
	if ( use_system_gl )
	{
		return fptable.glXIsDirect( dpy, ctx );
	}

	return 1;
}

void
__glX_WindowGetSize( Display *display, GLXDrawable drawable,
                     unsigned int *width_retval,
                     unsigned int *height_retval )
{
    Window       root;
    int          x, y;
    unsigned int width, height, border, depth;

    if ( !XGetGeometry( display, drawable, &root, &x, &y,
                        &width, &height, &border, &depth ) )
        wireGLSimpleError( "__glX_WindowGetSize( ): XGetGeometry() failed!" );

    *width_retval  = width;
    *height_retval = height;
}


Bool
glXMakeCurrent( Display *dpy, GLXDrawable drawable, GLXContext ctx )
{
	if ( use_system_gl )
	{
		return fptable.glXMakeCurrent( dpy, drawable, ctx );
	}

	if ( __wiregl_globals.glx_display != dpy )
	{
		__wiregl_globals.glx_display = dpy;
		__wiregl_globals.glx_drawable = drawable;
		wireGLCreateContext( 0 ); // moved here by Humper.
	}
	wireGLMakeCurrent( );

	myGLXCurrentContext = myGLXContext;
	return True;
}

Bool
glXQueryExtension( Display *dpy, int *errorBase, int *eventBase )
{
	if ( use_system_gl )
	{
		return fptable.glXQueryExtension( dpy, errorBase, eventBase );
	}

	return 1; // You BET we do...
}

Bool
glXQueryVersion( Display *dpy, int *major, int *minor )
{
	if ( use_system_gl )
	{
		return fptable.glXQueryVersion( dpy, major, minor );
	}

    *major = 1;
    *minor = 3;

    return 1;
}

void
glXSwapBuffers( Display *dpy, GLXDrawable drawable )
{
	if ( use_system_gl )
	{
		fptable.glXSwapBuffers( dpy, drawable );
		return;
	}

	wireGLSwapBuffers();
}

#define REPLACE_GLXUSEXFONT 1

#ifdef REPLACE_GLXUSEXFONT

#include <stdlib.h>
#include <stdio.h>
#include <GL/gl.h>
#include <X11/Xlib.h>

/*
 * This code swiped from vtk3.1's imaging/vtkXMesaTextMapper.c
 * to give a replacement for glXUseXFont(), which wiregl doesn't offer.
 * This looks as though it'll work anywhere plain X and OpenGL do.
 * Stuart Levy, slevy@ncsa.uiuc.edu, Oct 2000.
 */

static void
vtkFillBitmap (Display *dpy, Window win, GC gc,
	       unsigned int width, unsigned int height,
	       int x0, int y0, char c, GLubyte *bitmap)
{
  XImage *image;
  unsigned int x, y;
  Pixmap pixmap;

  pixmap = XCreatePixmap (dpy, win, 8*width, height, 1);
  XSetForeground(dpy, gc, 0);
  XFillRectangle (dpy, pixmap, gc, 0, 0, 8*width, height);
  XSetForeground(dpy, gc, 1);
  XDrawString (dpy, pixmap, gc, x0, y0, &c, 1);

  image = XGetImage (dpy, pixmap, 0, 0, 8*width, height, 1, XYPixmap);

  /* Fill the bitmap (X11 and OpenGL are upside down wrt each other).  */
  for (y = 0; y < height; y++)
    for (x = 0; x < 8*width; x++)
      if (XGetPixel (image, x, y))
	bitmap[width*(height - y - 1) + x/8] |= (1 << (7 - (x % 8)));
  
  XFreePixmap (dpy, pixmap);
  XDestroyImage (image);
}

void replacement_glXUseXFont( Font font, int first, int count, int listbase )
{
  Pixmap pixmap;
  GC gc;
  XGCValues values;
  unsigned long valuemask;
  static Display *dpy = NULL;

  XFontStruct *fs;

  GLint swapbytes, lsbfirst, rowlength;
  GLint skiprows, skippixels, alignment;

  unsigned int max_width, max_height, max_bm_width, max_bm_height;
  GLubyte *bm;

  int i;

  if(dpy == NULL) {
    dpy = XOpenDisplay(NULL);
    if(dpy == NULL) {
	fprintf(stderr, "glXUseXFont(): can't open default X display??\n");
	return;
    }
  }

  fs = XQueryFont (dpy, font);  
  if (!fs)
    {
      return;
    }

  /* Allocate a bitmap that can fit all characters.  */
  max_width = fs->max_bounds.rbearing - fs->min_bounds.lbearing;
  max_height = fs->max_bounds.ascent + fs->max_bounds.descent;
  max_bm_width = (max_width + 7) / 8;
  max_bm_height = max_height;

  bm = (GLubyte *) malloc ((max_bm_width * max_bm_height) * sizeof (GLubyte));
  if (!bm)
    {
      return;
    }

  /* Save the current packing mode for bitmaps.  */
  glGetIntegerv	(GL_UNPACK_SWAP_BYTES, &swapbytes);
  glGetIntegerv	(GL_UNPACK_LSB_FIRST, &lsbfirst);
  glGetIntegerv	(GL_UNPACK_ROW_LENGTH, &rowlength);
  glGetIntegerv	(GL_UNPACK_SKIP_ROWS, &skiprows);
  glGetIntegerv	(GL_UNPACK_SKIP_PIXELS, &skippixels);
  glGetIntegerv	(GL_UNPACK_ALIGNMENT, &alignment);

  /* Enforce a standard packing mode which is compatible with
     fill_bitmap() from above.  This is actually the default mode,
     except for the (non)alignment.  */
  glPixelStorei	(GL_UNPACK_SWAP_BYTES, GL_FALSE);
  glPixelStorei	(GL_UNPACK_LSB_FIRST, GL_FALSE);
  glPixelStorei	(GL_UNPACK_ROW_LENGTH, 0);
  glPixelStorei	(GL_UNPACK_SKIP_ROWS, 0);
  glPixelStorei	(GL_UNPACK_SKIP_PIXELS, 0);
  glPixelStorei	(GL_UNPACK_ALIGNMENT, 1);

  pixmap = XCreatePixmap (dpy, RootWindow(dpy,DefaultScreen(dpy)), 10, 10, 1);
  values.foreground = BlackPixel (dpy, DefaultScreen (dpy));
  values.background = WhitePixel (dpy, DefaultScreen (dpy));
  values.font = fs->fid;
  valuemask = GCForeground | GCBackground | GCFont;
  gc = XCreateGC (dpy, pixmap, valuemask, &values);
  XFreePixmap (dpy, pixmap);

  for (i = 0; i < count; i++)
    {
      unsigned int width, height, bm_width, bm_height;
      GLfloat x0, y0, dx, dy;
      XCharStruct *ch;
      int x, y;
      unsigned int c = first + i;
      int list = listbase + i;

      if (fs->per_char
	  && (c >= fs->min_char_or_byte2) && (c <= fs->max_char_or_byte2))
	ch = &fs->per_char[c-fs->min_char_or_byte2];
      else
	ch = &fs->max_bounds;
      
      /* glBitmap()' parameters:
	 straight from the glXUseXFont(3) manpage.  */
      width = ch->rbearing - ch->lbearing;
      height = ch->ascent + ch->descent;
      x0 = - ch->lbearing;
      y0 = ch->descent - 1;
      dx = ch->width;
      dy = 0;

      /* X11's starting point.  */
      x = - ch->lbearing;
      y = ch->ascent;
      
      /* Round the width to a multiple of eight.  We will use this also
	 for the pixmap for capturing the X11 font.  This is slightly
	 inefficient, but it makes the OpenGL part real easy.  */
      bm_width = (width + 7) / 8;
      bm_height = height;

      glNewList (list, GL_COMPILE);
      if ((c >= fs->min_char_or_byte2) && (c <= fs->max_char_or_byte2)
	  && (bm_width > 0) && (bm_height > 0))
	{
	memset(bm, '\0', bm_width * bm_height);
	vtkFillBitmap (dpy, RootWindow(dpy,DefaultScreen(dpy)), 
		       gc, bm_width, bm_height, x, y, c, bm);
	glBitmap (width, height, x0, y0, dx, dy, bm);
	}
      else
	glBitmap (0, 0, 0.0, 0.0, dx, dy, NULL);
      glEndList ();
    }

  free (bm);
  XFreeFontInfo( NULL, fs, 0 );
  XFreeGC (dpy, gc);

  /* Restore saved packing modes.  */    
  glPixelStorei(GL_UNPACK_SWAP_BYTES, swapbytes);
  glPixelStorei(GL_UNPACK_LSB_FIRST, lsbfirst);
  glPixelStorei(GL_UNPACK_ROW_LENGTH, rowlength);
  glPixelStorei(GL_UNPACK_SKIP_ROWS, skiprows);
  glPixelStorei(GL_UNPACK_SKIP_PIXELS, skippixels);
  glPixelStorei(GL_UNPACK_ALIGNMENT, alignment);
}
#endif /*REPLACE_GLXUSEXFONT*/

void
glXUseXFont( Font font, int first, int count, int listBase )
{
	if ( use_system_gl )
	{
		fptable.glXUseXFont( font, first, count, listBase );
		return;
	}

#ifndef REPLACE_GLXUSEXFONT
	wireGLSimpleError( "Unsupported GLX Call: glxusexfont" );
#else
	replacement_glXUseXFont( font, first, count, listBase );
#endif
}

void
glXWaitGL( void )
{
	static int first_call = 1;

	if ( use_system_gl )
	{
		fptable.glXWaitGL( );
		return;
	}

	if ( first_call )
	{
		wireGLWarning( WIREGL_WARN_DEBUG, "Ignoriing unsupported GLX call, "
					   "glXWaitGL()" );
		first_call = 0;
	}
}

void
glXWaitX( void )
{
	static int first_call = 1;

	if ( use_system_gl )
	{
		fptable.glXWaitX( );
		return;
	}
	
	if ( first_call )
	{
		wireGLWarning( WIREGL_WARN_DEBUG, "Ignoriing unsupported GLX call, "
					   "glXWaitX()" );
		first_call = 0;
	}
}

const char *
glXQueryExtensionsString( Display *dpy, int screen )
{
    static const char *retval = "";

	if ( use_system_gl )
	{
		return fptable.glXQueryExtensionsString( dpy, screen );
	}

    return retval;
}

const char *
glXGetClientString( Display *dpy, int name )
{
    const char *retval;

	if ( use_system_gl )
	{
		return fptable.glXGetClientString( dpy, name );
	}

    switch ( name ) {

      case GLX_VENDOR:
        retval  = "WireGL";
        break;

      case GLX_VERSION:
        retval  = "1.0 WireGL";
        break;

      case GLX_EXTENSIONS:
        retval  = "";
        break;

      default:
        retval  = NULL;
    }

    return retval;
}

const char *
glXQueryServerString( Display *dpy, int screen, int name )
{
    const char *retval;

	if ( use_system_gl )
	{
		return fptable.glXQueryServerString( dpy, screen, name );
	}

    switch ( name ) {

      case GLX_VENDOR:
        retval  = "WireGL";
        break;

      case GLX_VERSION:
        retval  = "1.0 WireGL";
        break;

      case GLX_EXTENSIONS:
        retval  = "";
        break;

      default:
        retval  = NULL;
    }

    return retval;
}
