/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifdef WINDOWS

#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_common.h"
#include "wiregl/include/wiregl_bbox.h"

#include "wiregl/util/floatlib.c"
#include "wiregl/util/matrix.c"

#include "length_table.c"
#include "geometry.h"
#include <math.h>
#include <float.h>
#include <assert.h>

typedef struct
{
	GLmatrix compositeMatrix;
	GLboolean ishint;
	GLvectorf bounds_min;
	GLvectorf bounds_max;
} TransformationDataT;

typedef struct
{
	GLfloat m00, m01, m02;
	GLfloat m10, m11, m12;
	GLfloat m20, m21, m22;
} DerivativeMatrixT;

/* Global data for compression routines.
 *
 * extentdata-> Most recent guess as to how large the bounding box is.
 * offsetdata-> Most recent guess as to how far from the left edge the
 *				first point is.
 * startpoint-> Starting vertex of most recent geometry set.
 * latesttransform-> Most recent composite matrix and hint information,
 *					 if any.
 * fastcompressdata-> Precalculated quantities for the per-vertex
 *					  computations.
 * compressdata-> High-level quantization information.
 * compressiondatavalid-> True only when fastcompressdata has been
 *						  filled out properly.
 */

GLfloat __wiregl_extentdata[] = {0.0, 0.0, 0.0};
GLfloat __wiregl_offsetdata[] = {0.0, 0.0, 0.0};
GLfloat __wiregl_startpoint[3];
TransformationDataT __wiregl_latesttransform;
FastCompressionDataT __wiregl_fastcompressdata;
CompressionDataT __wiregl_compressdata = {0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0, 0, 0};
GLboolean __wiregl_compressiondatavalid = 0;

#define COMPRESS_DATA_SIZE ((sizeof(CompressionDataT) + 3) & ~0x3)

/*
 * is   mi0 * xo + mi1 * yo + mi2 * zo + mi3   num
 * -- = ------------------------------------ = ---
 * ws   m30 * xo + m31 * yo + m32 * zo + m33   den
 *
 * d(is / ws)   mij   m3j * num(i)
 * ---------- = --- - ------------
 *    d(jo)     den      den^2
 *
 *
 * outData[i][j] -> i = Screen Coordinate
 *                  j = Object Coordinate
 */

/* calculateDerivativeMatrix
 * -------------------------
 * This function takes in a given transformation matrix and computes the
 * derivative matrix at the given point. The contents of the derivative
 * matrix are explained in the previous comment.
 * An entry in the derivative matrix gives you how far along the given 
 * screen-space axis you would move if you moved one unit in the given
 * object-space axis.
 */

static __inline void __calculateDerivativeMatrix(DerivativeMatrixT *outData, GLvectorf *nearPos, TransformationDataT *inData)
{
	GLfloat den;
	GLfloat denSqr;
	GLfloat numXO, numYO, numZO;
	GLmatrix *mat = &inData->compositeMatrix;

	den = mat->m03 * nearPos->x + mat->m13 * nearPos->y + mat->m23 * nearPos->z + mat->m33;
	denSqr = den * den;

	numXO = mat->m00 * nearPos->x + mat->m10 * nearPos->y + mat->m20 * nearPos->z + mat->m30;
	numYO = mat->m01 * nearPos->x + mat->m11 * nearPos->y + mat->m21 * nearPos->z + mat->m31;
	numZO = mat->m02 * nearPos->x + mat->m12 * nearPos->y + mat->m22 * nearPos->z + mat->m32;

	outData->m00 = mat->m00 / den - mat->m03 * numXO / denSqr;
	outData->m01 = mat->m10 / den - mat->m13 * numXO / denSqr;
	outData->m02 = mat->m20 / den - mat->m23 * numXO / denSqr;

	outData->m10 = mat->m01 / den - mat->m03 * numYO / denSqr;
	outData->m11 = mat->m11 / den - mat->m13 * numYO / denSqr;
	outData->m12 = mat->m21 / den - mat->m23 * numYO / denSqr;

	outData->m20 = mat->m02 / den - mat->m03 * numZO / denSqr;
	outData->m21 = mat->m12 / den - mat->m13 * numZO / denSqr;
	outData->m22 = mat->m22 / den - mat->m23 * numZO / denSqr;
}

static __inline GLfloat min2(GLfloat a, GLfloat b)
{
	return(a<b?a:b);
}

static __inline GLfloat min3(GLfloat a, GLfloat b, GLfloat c)
{
	return(min2(min2(a,b),c));
}

/* determinePrecision
 * ------------------
 * This function calculates the quantization level that will be used
 * to encode the vertex data. The way this function decides this level
 * is by calculating the number of pixels spanned by each object-space
 * coordinate. The number of bits required to represent this number of
 * pixels is the used as the quantization level for that axis.  
 */

static __inline void __determinePrecision(CompressionDataT *outData, DerivativeMatrixT *inMatrix)
{
	GLfloat xoxs, xoys, xozs, xprec;
	GLfloat yoxs, yoys, yozs, yprec;
	GLfloat zoxs, zoys, zozs, zprec;

	char xdiffE = (char)__extractProperExponent(outData->scale[0]);
	char ydiffE = (char)__extractProperExponent(outData->scale[1]);
	char zdiffE = (char)__extractProperExponent(outData->scale[2]);

	GLviewportstate *viewport = &__wiregl_globals.context->viewport;
	int depthBits = __wiregl_globals.context->buffer.depthbits;
	unsigned int depth;
	
	/* Allow for less precise depth buffer if desired. */
	
	if(__wiregl_globals.compression_depthbits == -1)
		depth = depthBits;
	else
		depth = 1 << __wiregl_globals.compression_depthbits;

	/* Calculate pixel span for each combination of axes. */

	xoxs = (GLfloat)(1 / fabs(inMatrix->m00) / viewport->v_w);
	xoys = (GLfloat)(1 / fabs(inMatrix->m10) / viewport->v_h);
	xozs = (GLfloat)(1 / fabs(inMatrix->m20) / depth);

	yoxs = (GLfloat)(1 / fabs(inMatrix->m01) / viewport->v_w);
	yoys = (GLfloat)(1 / fabs(inMatrix->m11) / viewport->v_h);
	yozs = (GLfloat)(1 / fabs(inMatrix->m21) / depth);

	zoxs = (GLfloat)(1 / fabs(inMatrix->m02) / viewport->v_w);
	zoys = (GLfloat)(1 / fabs(inMatrix->m12) / viewport->v_h);
	zozs = (GLfloat)(1 / fabs(inMatrix->m22) / depth);

	/* Find screen-space axis most closely aligned with each object-space axis. */
	
	xprec = min3(xoxs,xoys,xozs);
	yprec = min3(yoxs,yoys,yozs);
	zprec = min3(zoxs,zoys,zozs);

	/* Find precisions. Do not allow negative precisions, and clamp precision at
	 * 16 to avoid messiness with the compression algorithm. */
	
	outData->mantissaPrecision[0] = (char) (xdiffE - __extractProperExponent(xprec));
	if(outData->mantissaPrecision[0] < 0)
		outData->mantissaPrecision[0] = 0;
	if(outData->mantissaPrecision[0] > 16)
		outData->mantissaPrecision[0] = 16;

	outData->mantissaPrecision[1] = (char) (ydiffE - __extractProperExponent(yprec));
	if(outData->mantissaPrecision[1] < 0)
		outData->mantissaPrecision[1] = 0;
	if(outData->mantissaPrecision[1] > 16)
		outData->mantissaPrecision[1] = 16;

	outData->mantissaPrecision[2] = (char) (zdiffE - __extractProperExponent(zprec));
	if(outData->mantissaPrecision[2] < 0)
		outData->mantissaPrecision[2] = 0;
	if(outData->mantissaPrecision[2] > 16)
		outData->mantissaPrecision[2] = 16;
}

/* calculateObjectSpaceCenter
 * --------------------------
 * This function transforms the center of screen space into object space and returns
 * the point in object space that corresponds to the center of screen space.
 */

static __inline void __calculateObjectSpaceCenter(TransformationDataT *inData, GLvectorf *center)
{
	GLmatrix m;
	
	__gl_Matrix4x4AdjointInverse(&m, &inData->compositeMatrix);

	center->x = m.m30 / m.m33;
	center->y = m.m31 / m.m33;
	center->z = m.m32 / m.m33;
}

/* calculateFastCompressionData
 * ----------------------------
 * Precomputes information necessary for the per-vertex step of the compression.
 * We want to do as little work as possible inside the glVertex call, so we do
 * as much as we can here.
 */

static __inline void __calculateFastCompressionData( TransformationDataT *inData )
{
	GLfloat mult[3];
	GLvectorf center;
	int bits, i;

	__wiregl_fastcompressdata.compressionFailed = 0;
	__calculateObjectSpaceCenter(inData, &center);

	for(i = 0; i < 3; i++)
	{
		__wiregl_fastcompressdata.mask[i] = (1 << __wiregl_compressdata.mantissaPrecision[i]) - 1;
		__wiregl_fastcompressdata.nmask[i] = ~__wiregl_fastcompressdata.mask[i];

		mult[i] = (GLfloat)__wiregl_fastcompressdata.mask[i];

		/* If we let mult[i] be #INF.0, then we're in trouble because the floating-point operations
		 * take forever, so just set it to 0.0 to keep the speed up. */

		if(__wiregl_compressdata.scale[i] != 0.0f)
			__wiregl_fastcompressdata.mult[i] = mult[i] / __wiregl_compressdata.scale[i];
		else
			__wiregl_fastcompressdata.mult[i] = 0.0f;
		
		/* This function ensures compliance with OpenGL invariance rules. Essentially, for a given
		 * transformation matrix, the precision and scale produced will always be the same, but the
		 * translate parameter will be different. This difference could potentially cause two identical
		 * verticies to come out to different points at the end of the algorithm.
		 * To fix this problem, we calculate where the center of screen space is in object space and ensure
		 * that this point is always the origin of our quantization. We are then assured that the
		 * algorithm is OpenGL compliant since we are measuring each vertex against a fixed point in
		 * object space instead of the side of the bounding box. */
		
		if(__wiregl_fastcompressdata.mult[i] != 0.0 && __wiregl_compressdata.scale[i] != 0.0)
			__wiregl_compressdata.translate[i] = (GLfloat)(floor((__wiregl_compressdata.translate[i] - center.x) * __wiregl_fastcompressdata.mult[i]) / __wiregl_fastcompressdata.mult[i] + center.x);
		else
			__wiregl_fastcompressdata.compressionFailed = 1;
		
		__wiregl_fastcompressdata.add[i] = - __wiregl_compressdata.translate[i];
	}

	__wiregl_fastcompressdata.shiftLeft[0] = __wiregl_compressdata.mantissaPrecision[0];
	__wiregl_fastcompressdata.shiftLeft[1] = __wiregl_compressdata.mantissaPrecision[0] + __wiregl_compressdata.mantissaPrecision[1];

	bits = __wiregl_compressdata.mantissaPrecision[0] + __wiregl_compressdata.mantissaPrecision[1] + __wiregl_compressdata.mantissaPrecision[2];
	if(bits > 32)
		__wiregl_fastcompressdata.bitsLeft = 32 - __wiregl_fastcompressdata.shiftLeft[1];
	else
		__wiregl_fastcompressdata.bitsLeft = 0;

	__wiregl_fastcompressdata.bytes = (bits + 7) >> 3;
	__wiregl_fastcompressdata.bytes = ((__wiregl_fastcompressdata.bytes + 3) & ~0x3);

	/* We cannot have no data, or the opcode buffer will overflow, so we send four dead bytes of data
	 * to satisfy the packer. */
	
	if(__wiregl_fastcompressdata.bytes == 0)
		__wiregl_fastcompressdata.bytes = 4;
}

/* processCompressionData
 * ----------------------
 * This function is called when a object-space bounding box hint is provided to WireGL. It
 * is similar to GuessCompressionData, but it uses the given bounding box instead of trying
 * to guess it.
 */

static __inline void __processCompressionData(CompressionDataT *outData, TransformationDataT *inData)
{
	DerivativeMatrixT dMatrix;
	GLvectorf nearPos;

	/* Calculate derivative matrix from the center of the bounding box. */
	
	nearPos.x = (inData->bounds_max.x + inData->bounds_min.x) / 2;
	nearPos.y = (inData->bounds_max.y + inData->bounds_min.y) / 2;
	nearPos.z = (inData->bounds_max.z + inData->bounds_min.z) / 2;
	
	outData->scale[0] = inData->bounds_max.x - inData->bounds_min.x;
	outData->scale[1] = inData->bounds_max.y - inData->bounds_min.y;
	outData->scale[2] = inData->bounds_max.z - inData->bounds_min.z;
		
	outData->translate[0] = inData->bounds_min.x;
	outData->translate[1] = inData->bounds_min.y;
	outData->translate[2] = inData->bounds_min.z;

	__calculateDerivativeMatrix(&dMatrix, &nearPos, inData);
	__determinePrecision(outData, &dMatrix);
}

/* SendCompressionData
 * -------------------
 * This function is the pack function for SetVertexCompressionData.
 */

void __wiregl_SendCompressionData( void )
{
	unsigned char *data_ptr;
	GET_BUFFERED_POINTER(COMPRESS_DATA_SIZE);
	WRITE_DATA(0, CompressionDataT, __wiregl_compressdata);
	WRITE_OPCODE(WIREGL_SETVERTEXCOMPRESSIONDATA_OPCODE);
}

/* CompressionPostprocess
 * ----------------------
 * This function adjusts the guess of the bounding box given the most
 * recent bounding box. If the guess was too small, it expands the
 * guess to match the most recent one's size. It then inflates the
 * box slightly to account for precision errors.
 */

void __wiregl_CompressionPostprocess( GLfloat *bbox )
{
	GLfloat newMin[3], newMax[3];
	GLfloat oldMin[3], oldMax[3];
	GLfloat expand;
	GLboolean doExpand[3] = {GL_FALSE, GL_FALSE, GL_FALSE};

	/* Check the left sides of the bounding-box */

	oldMin[0] = __wiregl_compressdata.translate[0];
	newMin[0] = bbox[0];
	
	oldMin[1] = __wiregl_compressdata.translate[1];
	newMin[1] = bbox[2];

	oldMin[2] = __wiregl_compressdata.translate[2];
	newMin[2] = bbox[4];

	/* If any of the guesses were too small, expand the bouding-box. */

	if(newMin[0] < oldMin[0])
	{
		__wiregl_extentdata[0] += oldMin[0] - newMin[0];
		__wiregl_offsetdata[0] = __wiregl_startpoint[0] - newMin[0];
		oldMin[0] = newMin[0];
		doExpand[0] = GL_TRUE;
	}

	if(newMin[1] < oldMin[1])
	{
		__wiregl_extentdata[1] += oldMin[1] - newMin[1];
		__wiregl_offsetdata[1] = __wiregl_startpoint[1] - newMin[1];
		oldMin[1] = newMin[1];
		doExpand[1] = GL_TRUE;
	}

	if(newMin[2] < oldMin[2])
	{
		__wiregl_extentdata[2] += oldMin[2] - newMin[2];
		__wiregl_offsetdata[2] = __wiregl_startpoint[2] - newMin[2];
		oldMin[2] = newMin[2];
		doExpand[2] = GL_TRUE;
	}

	/* Check the right sides. */

	oldMax[0] = oldMin[0] + __wiregl_compressdata.scale[0];
	newMax[0] = bbox[1];
	
	oldMax[1] = oldMin[1] + __wiregl_compressdata.scale[1];
	newMax[1] = bbox[3];

	oldMax[2] = oldMin[2] + __wiregl_compressdata.scale[2];
	newMax[2] = bbox[5];

	if(newMax[0] > oldMax[0])
	{
		__wiregl_extentdata[0] = bbox[1] - oldMin[0];
		doExpand[0] = GL_TRUE;
	}

	if(newMax[1] > oldMax[1])
	{
		__wiregl_extentdata[1] = bbox[3] - oldMin[1];
		doExpand[1] = GL_TRUE;
	}

	if(newMax[2] > oldMax[2])
	{
		__wiregl_extentdata[2] = bbox[5] - oldMin[2];
		doExpand[2] = GL_TRUE;
	}

	/* If the bounding-box was adjusted, expand it slightly. */

	if(doExpand[0])
	{
		expand = __wiregl_extentdata[0] * 1e-5f;
		__wiregl_extentdata[0] += 2.0f * expand;
		__wiregl_offsetdata[0] += expand;
	}

	if(doExpand[1])
	{
		expand = __wiregl_extentdata[1] * 1e-5f;
		__wiregl_extentdata[1] += 2.0f * expand;
		__wiregl_offsetdata[1] += expand;
	}

	if(doExpand[2])
	{
		expand = __wiregl_extentdata[2] * 1e-5f;
		__wiregl_extentdata[2] += 2.0f * expand;
		__wiregl_offsetdata[2] += expand;
	}
}

/* CompressionPreprocess
 * ---------------------
 * Store data about the current transformation for GuessCompressData, or if the hint
 * is active, just use that right away.
 */

void __wiregl_CompressionPreprocess( GLmatrix composite, GLint ishint )
{
	__wiregl_compressiondatavalid = GL_FALSE;
	__wiregl_latesttransform.compositeMatrix = composite;
	__wiregl_latesttransform.ishint = (char) (ishint);
	if(ishint)
	{
		__wiregl_latesttransform.bounds_min = __wiregl_globals.bounds_min;
		__wiregl_latesttransform.bounds_max = __wiregl_globals.bounds_max;
		__processCompressionData(&__wiregl_compressdata, &__wiregl_latesttransform);

		__calculateFastCompressionData(&__wiregl_latesttransform);
		__wiregl_compressiondatavalid = GL_TRUE;
	}
}

/* GuessCompressData
 * -----------------
 * Guesses the bounding-box based on previous bounding boxes and the first point.
 */

void __wiregl_GuessCompressData( const GLfloat *x, const GLfloat *y, const GLfloat *z )
{
	DerivativeMatrixT dMatrix;
	GLvectorf nearPos;
	static int average = 0, num = 0;

	__wiregl_compressdata.scale[0] = __wiregl_extentdata[0];
	__wiregl_compressdata.scale[1] = __wiregl_extentdata[1];
	__wiregl_compressdata.scale[2] = __wiregl_extentdata[2];

	__wiregl_compressdata.translate[0] = *x - __wiregl_offsetdata[0];
	__wiregl_compressdata.translate[1] = *y - __wiregl_offsetdata[1];
	__wiregl_compressdata.translate[2] = *z - __wiregl_offsetdata[2];

	__wiregl_startpoint[0] = nearPos.x = *x;
	__wiregl_startpoint[1] = nearPos.y = *y;
	__wiregl_startpoint[2] = nearPos.z = *z;

	__calculateDerivativeMatrix(&dMatrix, &nearPos, &__wiregl_latesttransform);
	__determinePrecision(&__wiregl_compressdata, &dMatrix);

	__calculateFastCompressionData(&__wiregl_latesttransform);
	__wiregl_compressiondatavalid = 1;
}

#endif
