/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifndef WIREGL_UTIL_H
#define WIREGL_UTIL_H

#ifndef WINDOWS
#define OPENGL_APIENTRY
#else
#ifndef OPENGL_APIENTRY
#define OPENGL_APIENTRY __stdcall
#endif
#endif

#ifndef WINDOWS
#define UTIL_DECL
#else
#ifndef UTIL_DECL
#define UTIL_DECL __declspec(dllimport)
#endif
#endif

#ifdef __GNUC__
#define NORETURN __attribute__ ((noreturn))
#else
#define NORETURN
#endif

#if defined (WINDOWS)
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <winbase.h>
#include <winsock.h>
#include <errno.h>
#endif

#if defined(LINUX) || defined(IRIX) || defined(IRIX64) || defined(AIX)
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <string.h>
#include <netinet/in.h>
#include <unistd.h>
#endif

#include <stdio.h> // for FILE *

#include "wiregl/include/wiregl_common.h"

#ifdef __cplusplus
extern "C" {
#endif

UTIL_DECL void *wireGLAlloc( unsigned int bytes );
UTIL_DECL void wireGLRealloc( void **ptr, unsigned int bytes );
UTIL_DECL void wireGLFree( void *ptr );

UTIL_DECL void __wireGLAssert( const char *assertion, const char *file, int line );
UTIL_DECL void __wireGLAbort( const char *file, int line );

#ifndef NDEBUG
#define wireGLAssert(EX)  ((EX)?((void)0):__wireGLAssert( # EX , __FILE__, __LINE__))
#else
#define wireGLAssert(EX)        ((void) 0)
#endif

#define wireGLAbort()     __wireGLAbort( __FILE__, __LINE__ )

typedef enum {
  WIREGL_WARN_NONE=0,
  WIREGL_WARN_CRITICAL,
  WIREGL_WARN_IMPORTANT,
  WIREGL_WARN_NOTICE,
  WIREGL_WARN_CHATTY,
  WIREGL_WARN_DEBUG,
  WIREGL_WARN_PACKET_DEBUG,
  WIREGL_WARN_VERBOSE_DEBUG
} WireGLWarningLevel;

UTIL_DECL void wireGLError( const char *format, ... );
UTIL_DECL void wireGLSimpleError( const char *format, ... );
UTIL_DECL void wireGLPrintError( const char *format, ... );
UTIL_DECL void wireGLWarning( WireGLWarningLevel level, const char *format, ... );
UTIL_DECL void wireGLInitWarnings( const char *warn_str, int gui );
UTIL_DECL void wireGLInitOS(void);
UTIL_DECL void wireGLSleep( unsigned int msecs );

UTIL_DECL char wireGLDetermineEndianness( void );

#if defined(LINUX) || defined(WINDOWS)
#define WIREGL_UNALIGNED_ACCESS_OKAY
#else
#undef WIREGL_UNALIGNED_ACCESS_OKAY
#endif
UTIL_DECL void wireGLWriteUnalignedDouble( void *buffer, double d );
UTIL_DECL double wireGLReadUnalignedDouble( void *buffer );

typedef struct WireGLBufferPool {
    unsigned int   num;
    unsigned int   max;
    void         **buf;
} WireGLBufferPool;

void  wireGLBufferPoolInit( WireGLBufferPool *pool, unsigned int max );
void  wireGLBufferPoolLoad( WireGLBufferPool *pool, void *mem,
							unsigned int stride, unsigned int count );
void  wireGLBufferPoolPush( WireGLBufferPool *pool, void *buf );
void *wireGLBufferPoolPop( WireGLBufferPool *pool );

typedef struct WireGLConnection WireGLConnection;

typedef enum {
	WIREGL_NO_CONNECTION,
	WIREGL_TCPIP,
	WIREGL_DROP_PACKETS,
	WIREGL_FILE_TRACE,
	WIREGL_GM,
	WIREGL_MULTI_PACKET
} WireGLConnectionType;

#if defined(WINDOWS)
typedef SOCKET WireGLSocket;
#else
typedef int    WireGLSocket;
#endif

typedef void (*WireGLVoidFunc)( void );
typedef void (*WireGLNetReceiveFunc)( WireGLConnection *conn, void *buf, unsigned int len );
typedef void (*WireGLNetConnectFunc)( WireGLConnection *conn, WireGLConnectionRequest *request );
typedef void (*WireGLNetCloseFunc)( unsigned int sender_id );

UTIL_DECL void wireGLNetInit( WireGLNetReceiveFunc recvFunc, WireGLNetCloseFunc closeFunc );
UTIL_DECL void *wireGLNetAlloc( WireGLConnection *conn );
UTIL_DECL void wireGLNetSend( WireGLConnection *conn, void **bufp, void *start, unsigned int len );
UTIL_DECL int  wireGLNetRecv( void );
UTIL_DECL void wireGLNetFree( WireGLConnection *conn, void *buf );
UTIL_DECL unsigned int wireGLNetMaxSend( void );

struct WireGLConnection {
	WireGLConnectionType type;
	unsigned int sender_id;

	char *hostname;

	void *(*Alloc)( void );
	void  (*Send)( WireGLConnection *conn, void **buf, void *start, unsigned int len );
	void  (*Free)( WireGLConnection *conn, void *buf );

	/* pending writebacks */
	int pending_writebacks;

	/* logging */
	int total_bytes;

	/* credits for flow control */
	int send_credits;
	int recv_credits;

	/* TCP/IP */
	WireGLSocket tcp_socket;

	/* Myrinet GM */
	unsigned int gm_node_id;
	unsigned int gm_port_num;
};

UTIL_DECL void wireGLCloseSocket( WireGLSocket sock );

extern unsigned int __wiregl_max_send;

UTIL_DECL void wireGLNetBytesToString( char *string, int nstring, void *data, int ndata );
UTIL_DECL void wireGLNetWordsToString( char *string, int nstring, void *data, int ndata );

UTIL_DECL void wireGLBecomeServer( unsigned short port, WireGLNetConnectFunc connectFunc );
UTIL_DECL void wireGLTcpipBecomeServer( WireGLSocket sock, WireGLVoidFunc connectFunc );

UTIL_DECL WireGLConnection *wireGLConnectToServer( char *server, unsigned short default_port, WireGLConnectionRequest *request );

UTIL_DECL void wireGLGmInit( WireGLNetReceiveFunc recvFunc, 
							 WireGLNetCloseFunc   closeFunc );
UTIL_DECL unsigned int wireGLGmNodeId( void );
UTIL_DECL unsigned int wireGLGmPortNum( void );
UTIL_DECL void wireGLGmConnection( WireGLConnection *conn );
UTIL_DECL int  wireGLGmRecv( void );

UTIL_DECL void wireGLTcpipInit( WireGLNetReceiveFunc recvFunc, 
								WireGLNetCloseFunc   closeFunc );
UTIL_DECL void wireGLTcpipConnection( WireGLConnection *conn );
UTIL_DECL int  wireGLTcpipRecv( void );

UTIL_DECL void  wireGLTcpipWriteExact( WireGLSocket sock, void *buf,
									   unsigned int len );
UTIL_DECL void  wireGLTcpipReadExact( WireGLSocket sock, void *buf,
									  unsigned int len );

UTIL_DECL int  wireGLTcpipErrno( void );
UTIL_DECL char *wireGLTcpipErrorString( int err );

UTIL_DECL void wireGLDevNullInit( WireGLNetReceiveFunc recvFunc, 
WireGLNetCloseFunc   closeFunc );
UTIL_DECL void wireGLDevNullConnection( WireGLConnection *conn );
UTIL_DECL int  wireGLDevNullRecv( void );

typedef void (*WireGLThreadFunc)( void * );
UTIL_DECL void wireGLCreateThread( WireGLThreadFunc entry, void *arg );

#define WIREGL_NUM_BUCKETS 1047

typedef struct WireGLHashNode {
        char is_string;
        unsigned int key;
        char *str_key;
        void *data;
        struct WireGLHashNode *next;
} WireGLHashNode;

typedef struct WireGLHashTable {
        unsigned int num_elements;
        WireGLHashNode *buckets[WIREGL_NUM_BUCKETS];
} WireGLHashTable;

UTIL_DECL WireGLHashTable *wireGLAllocHashtable( void );
UTIL_DECL void wireGLFreeHashtable( WireGLHashTable *hash );
UTIL_DECL void wireGLHashtableAdd( WireGLHashTable *h, unsigned int key, void *data );
UTIL_DECL void wireGLHashtableStringAdd( WireGLHashTable *h, char *key, void *data );
UTIL_DECL void wireGLHashtableDelete( WireGLHashTable *h, unsigned int key );
UTIL_DECL void wireGLHashtableStringDelete( WireGLHashTable *h, char *key );
UTIL_DECL void *wireGLHashtableSearch( WireGLHashTable *h, unsigned int key );
UTIL_DECL void *wireGLHashtableStringSearch( WireGLHashTable *h, char * key );
UTIL_DECL void wireGLHashtableReplace( WireGLHashTable *h, unsigned int key, void *data, int free_mem );
UTIL_DECL void wireGLHashtableStringReplace( WireGLHashTable *h, char * key, void *data, int free_mem );
UTIL_DECL unsigned int wireGLHashtableNumElements( WireGLHashTable *h) ;

#define WIREGL_HASHTABLE_WALK( h, t ) {         \
  WireGLHashNode *t;                            \
  int _;                                        \
  for (_ = 0 ; _ < WIREGL_NUM_BUCKETS ; _++) {  \
    for (t = h->buckets[_] ; t; t = t->next)   {


#define WIREGL_HASHTABLE_WALK_END( h )          \
  }                                             \
 }                                              \
}


#ifndef WINDOWS
typedef unsigned long long iotimer64_t;
typedef unsigned int iotimer32_t;
#endif

typedef struct Timer {
        double time0, elapsed;
        char running;

        int fd;
#if defined (IRIX) || defined( IRIX64 )
        unsigned long long counter64;
        unsigned int counter32;
        unsigned int cycleval;

        volatile iotimer64_t *iotimer_addr64;
        volatile iotimer32_t *iotimer_addr32;

        void *unmapLocation;
        int unmapSize;
#elif defined(WINDOWS)
        LARGE_INTEGER performance_counter, performance_frequency;
        double one_over_frequency;
#elif defined( Linux ) || defined (AIX)
        struct timeval timeofday;
#endif
} WireGLTimer;

UTIL_DECL WireGLTimer *wireGLTimerNewTimer( void );
UTIL_DECL void wireGLDestroyTimer( WireGLTimer *t );
UTIL_DECL void wireGLStartTimer( WireGLTimer *t );
UTIL_DECL void wireGLStopTimer( WireGLTimer *t );
UTIL_DECL void wireGLResetTimer( WireGLTimer *t );
UTIL_DECL double wireGLTimerTime( WireGLTimer *t );

typedef enum {
        TO_SCREEN,
        TO_FILE
} WireGLInstrumentationType;

//#define INSTRUMENT

UTIL_DECL void wireGLInitInstrumentation( WireGLInstrumentationType type, ...);
UTIL_DECL void wireGLInstrumentToScreen( void );
UTIL_DECL void wireGLInstrumentToFile( char *fname );
UTIL_DECL void wireGLInstrumentShutDown( void );

#if defined(IGNORE_INSTRUMENT_FLAG) || defined(INSTRUMENT)

UTIL_DECL void wireGLInstrumentRegisterCounter( int id, char *name );
UTIL_DECL void wireGLInstrumentRegisterTimer( int id, char *name );
UTIL_DECL void wireGLInstrumentRegisterPerFrameCounter( int id, char *name );
UTIL_DECL void wireGLInstrumentRegisterPerServerCounter( int id, int num, char *name );

UTIL_DECL void wireGLInstrumentIncrPerFrameCounter( int id, float amount );
UTIL_DECL void wireGLInstrumentIncrPerServerCounter( int id, int which, float amount );
UTIL_DECL void wireGLInstrumentIncrCounter( int id, float amount );
UTIL_DECL void wireGLInstrumentSetCounter( int id, float amount );

UTIL_DECL void wireGLInstrumentResetTimer( int id );
UTIL_DECL void wireGLInstrumentStartTimer( int id );
UTIL_DECL void wireGLInstrumentStopTimer( int id );

UTIL_DECL float wireGLInstrumentGetCounter( int id );
UTIL_DECL double wireGLInstrumentGetTime( int id );

UTIL_DECL void wireGLInstrumentPrintPerServerCounter( int id );
UTIL_DECL void wireGLInstrumentPrintPerFrameCounter( int id );
UTIL_DECL void wireGLInstrumentPrintCounter( int id );
UTIL_DECL void wireGLInstrumentPrintTimer( int id );
UTIL_DECL void wireGLInstrumentNextFrame( void );

UTIL_DECL void wireGLInstrumentPrintf( const char *format, ... );

UTIL_DECL void wireGLInstrumentPrintAll( void );

#else

#define wireGLInstrumentRegisterCounter( id, name ) ((void) 0)
#define wireGLInstrumentRegisterTimer( id, name )  ((void) 0)
#define wireGLInstrumentRegisterPerServerCounter( id, num, name )  ((void) 0)
#define wireGLInstrumentRegisterPerFrameCounter( id, name )  ((void) 0)
#define wireGLInstrumentIncrPerServerCounter( id, val, name )  ((void) 0)
#define wireGLInstrumentIncrPerFrameCounter( id, val )  ((void) 0)
#define wireGLInstrumentIncrCounter( id, amount )  ((void) 0)
#define wireGLInstrumentSetCounter( id, amount )  ((void) 0)

#define wireGLInstrumentResetTimer( id )  ((void) 0)
#define wireGLInstrumentStartTimer( id )  ((void) 0)
#define wireGLInstrumentStopTimer( id )  ((void) 0)

#define wireGLInstrumentGetCounter( id )  ((void) 0)
#define wireGLInstrumentGetTime( id )  0

#define wireGLInstrumentPrintCounter( id )  ((void) 0)
#define wireGLInstrumentPrintTimer( id )  ((void) 0)

#define wireGLInstrumentNextFrame() ((void) 0)

#define wireGLInstrumentPrintPerServerCounter( id )  ((void) 0)
#define wireGLInstrumentPrintPerFrameCounter( id )  ((void) 0)
#define wireGLInstrumentPrintf __wireGLInstrumentNOP

#define wireGLInstrumentPrintAll()  ((void) 0)
#endif

UTIL_DECL int wireGLParseURL( char *url, char *protocol, char *hostname,
                                                          unsigned short *port,
                                                          unsigned short default_port );

UTIL_DECL char *wireGLGetLine( FILE *fp, char *buf );

#ifndef WINDOWS
#define WIREGL_PATH_SEPARATOR ":"
#else
#define WIREGL_PATH_SEPARATOR ";"
#endif

#define WIREGL_LITTLE_ENDIAN 0
#define WIREGL_BIG_ENDIAN 1

typedef struct {
        char *dllname;
#if defined(WINDOWS)
        HINSTANCE hinstLib;
#elif defined(IRIX) || defined(IRIX64) || defined( Linux ) || defined(AIX)
        void *hinstLib;
#else
#error DSOs, please
#endif
} WireGLDSO;

typedef void (OPENGL_APIENTRY * WireGLDSOFunc) ();
UTIL_DECL WireGLDSO *wireGLDSOOpen( const char *dllname );
UTIL_DECL WireGLDSOFunc wireGLDSOGet( WireGLDSO *dso, const char *symname );
UTIL_DECL WireGLDSOFunc wireGLDSOGetNoError( WireGLDSO *dso, const char *symname );
UTIL_DECL void wireGLDSOClose( WireGLDSO *dso );

typedef struct {
        char *str;
        void (*parser)( char *buf );
} WireGLParseStruct;

UTIL_DECL void wireGLParseFile( char *fname, WireGLParseStruct *ps );

#define WIREGL_UNUSED(x)	((void) x)

#ifdef __cplusplus
} /* extern "C" */
#endif

#endif /* WIREGL_UTIL_H */
