/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include <stdlib.h>
#include <stdio.h>
#include <memory.h>
#include <string.h>
#include <assert.h>

#include "glcontext.h"
#include "glerror.h"
#include "glhw.h"

GLcontext *		__hwcontext = NULL;
GLcontext *		__currentcontext = NULL;
GLstatebits	*	__statebits;
GLbitvalue		__bitID = GLBITS_ONES;
GLboolean		__glinitflag = GL_FALSE;
GLconfig  *		__config = NULL;

void
__glInit (GLconfig *c) {

	assert (c);

	__statebits = (GLstatebits *) malloc (sizeof (GLstatebits));
	memset ((void *) __statebits, 0, sizeof (GLstatebits));

	/* Call all the initbits functions */
	__gltrans_initbits	    (&(__statebits->trans), c);
	__gllighting_initbits	(&(__statebits->lighting), c);
	__glclient_initbits		(&(__statebits->client), c);

	__glhw_RegisterApi(c);
	__glhw_LoadSwitchApi(c->packapi);
	__glhw_LoadStateApi();
	__glhw_UnloadBeginEndApi();

	if ( c->broadcast )
		__glhw_LoadVertexPackApi();
	else 
		__glhw_LoadVertexBBPackApi();

	if ( c->pack_only )
		__glhw_LoadPackApi();

	__config = __glconfig_CopyConfig(c);

	__glinitflag = GL_TRUE;
}

void
__glDestroy (void) {
	__gllighting_destroybits (&(__statebits->lighting));
	__glinitflag = GL_FALSE;
}

GLcontext *
__glCreateVirtualContext(void) {
	return __glCreateContext_id(0, -1);
}

GLcontext *
__glCreateContext(void) {
	int i;
	GLbitvalue a;
	GLcontext *g;

	/* Get a bitID */
	for (i=0, a=1; i < GLBITS_LENGTH && !(__bitID & a); i++, a = a<<1);

	if (i == GLBITS_LENGTH) 
		if (__glerror(__LINE__, __FILE__, GL_NO_ERROR, 
			"No more context bitIDs available")) {
			return NULL;
		}

	g = __glCreateContext_id(a, i);
	
	return g;
}

GLcontext *  
__glCreateContext_id(GLbitvalue id, int i) {
	GLcontext *g;

	/* Initialize */
	if (__glinitflag == GL_FALSE) {
		__glerror(__LINE__, __FILE__, GL_NO_ERROR,
			"Must initialize GL first with __glInit");
		exit(1);
	}
	
	__config->id = i;

	/* Allocate a new context */
	g = (GLcontext *) malloc (sizeof (GLcontext));
	memset( (void *) g, 0, sizeof (GLcontext));

	/* Set the bitID */
	g->bitID = id;
	g->nbitID = ~id;
	__bitID &= ~id;
	g->ID = i;
	g->update = GLBITS_ONES;

	/* Init the components */
	__glbucket_init(&(g->bucket), __config);
	__glbuffer_init(&(g->buffer), __config);
	__glclient_init(&(g->client), __config);
	__glcurrent_init(&(g->current), __config);
	__gleval_init(&(g->eval), __config);
	__glfog_init(&(g->fog), __config);
	__glhint_init(&(g->hint), __config);
	__glimaging_init(&(g->imaging), __config);
	__gllighting_init(&(g->lighting), __config);
	__glline_init(&(g->line), __config);
	__gllists_init(&(g->lists), __config);
	__glpixel_init(&(g->pixel), __config);
	__glpolygon_init(&(g->polygon), __config);
	__glstencil_init(&(g->stencil), __config);
	__gltexture_init(&(g->texture), __config);
	__gltrans_init(&(g->trans), __config);
	__glviewport_init(&(g->viewport), __config);
	
	if (__config->vendor) {
		g->vendor = (char *) malloc (strlen(__config->vendor) + 1);
		strcpy (g->vendor, __config->vendor);
	}
	if (__config->renderer) {
		g->renderer = (char *) malloc (strlen(__config->renderer) + 1);
		strcpy (g->renderer, __config->renderer);
	}
	if (__config->version) {
		g->version = (char *) malloc (strlen(__config->version) + 1);
		strcpy (g->version, __config->version);
	}
	if (__config->extensions) {
		g->extensions = (char *) malloc (strlen(__config->extensions) + 1);
		strcpy (g->extensions, __config->extensions);
	}

	g->error = GL_NO_ERROR;
	g->config = __config;

	/* Set initial dirties 
	** Note: We should move these into the init functions...
	*/
	__statebits->trans.base = GLBITS_ONES;
	__statebits->trans.dirty = GLBITS_ONES;

	return g;
}

void
__glMakeCurrent(GLcontext *g) {
	if (__currentcontext == g) 
		return;

	__currentcontext = g;

	/* initialize viewport and scissor if invalid */
	if (g) {
		if (!g->viewport.v_valid && !g->viewport.s_valid)
			__glviewport_MakeCurrent (&g->viewport, &__statebits->viewport);
	}

	if (__hwcontext && g)
		__glcontext_switch (__hwcontext, g);

	if (g) {
		__hwcontext = g;
		__bktactive = g->bucket.active;
	}
}

const GLflushinfo *
__glBeginFlush () {
	const GLflushinfo *ret;
	GLcontext *g = GetCurrentContext();
	GLbitvalue bitID = g->bitID;
	static const GLflushinfo pack_only_flushinfo =
			{GLBITS_ONES, 1,
			{-1.0f, -1.0f, -1.0f, 1.0f},
			{1.0f, 1.0f, 1.0f, 1.0f},
			{-1.0f, -1.0f, -1.0f, 1.0f},
			{1.0f, 1.0f, 1.0f, 1.0f}};
	
	if (__config->pack_only) 
		return &pack_only_flushinfo;

	ret = __glbucket_flush();
	__glcurrent_recover (&(g->current), &(__statebits->current), bitID);
	if (__config->vpinch)
		__config->vpinch (g);

	return ret;
}

void
__glDiffContext(GLcontext *current, GLcontext *target) {
	if (__config->pack_only) 
		return;

	__glcontext_sync (current, target);
}

void
__glEndFlush () {
	GLcontext *g = GetCurrentContext();
	GLbitvalue bitID = g->bitID;

	if (__config->pack_only) 
		return;

	/* Recover any client state */
	__glclient_finish(&(g->client), &(__statebits->client), bitID);

	/* Reset current pointers and restart the triangle engine */
	__glcurrent_finish(&(g->current), &(__statebits->current), bitID);
}


void
__glCloseBeginEnd( GLcontext *target) {
	if (__config->pack_only) 
		return;

	if (target->current.beginend == GL_TRUE) {
		__glhw_End();
	}
}

void
__glFixupPointers( int offset ) {
	GLcontext *g = GetCurrentContext();
	__glcurrent_fixup_pointers (&(g->current), offset);
}

void
__glExtractCurrentState() {
	GLcontext *g = GetCurrentContext();
	GLbitvalue bitID = g->bitID;
	
	if (__config->pack_only) 
		return;
	
	__glcurrent_recover (&(g->current), &(__statebits->current), bitID);
		/* Clear the current pointers */
	__glcurrent_null_pointers (&(g->current));

}

void
__glSetOutputBounds (GLcontext *g,
					 GLrecti *outputwindow,
					 GLrecti *imagespace,
					 GLrecti *imagewindow) {

	GLrecti p,q;

#if 0
	printf ("SetOutputBounds\n");
	printf ("outputwindow: (%d, %d) -> (%d, %d)\n", outputwindow->x1, outputwindow->y1,
		outputwindow->x2, outputwindow->y2);
	printf ("imagespace:   (%d, %d) -> (%d, %d)\n", imagespace->x1, imagespace->y1,
		imagespace->x2, imagespace->y2);
	printf ("imagewindow:  (%d, %d) -> (%d, %d)\n", imagewindow->x1, imagewindow->y1,
		imagewindow->x2, imagewindow->y2);
#endif
	__glviewport_setoutputbounds (&(g->viewport), outputwindow, imagespace, imagewindow, &p, &q);
	__gltrans_setoutputbounds (&(g->trans), outputwindow, &p, &q);
}

void
__glApplyViewTranform (GLcontext *g, GLboolean z) {
	GLstatebits *s = GetStateBits();

	g->trans.usebase = z;
	s->trans.base = GLBITS_ONES;
	s->trans.dirty = GLBITS_ONES;
}

void 
__glcontext_switch(GLcontext *current, GLcontext *target) {
	GLbitvalue bitID = target->bitID;

	if (__statebits->trans.dirty & bitID)
		__gltrans_switch	(&(__statebits->trans), bitID,
							 &(current->trans), &(target->trans));
	if (__statebits->pixel.dirty & bitID)
		__glpixel_switch	(&(__statebits->pixel), bitID,
							 &(current->pixel), &(target->pixel));
	if (__statebits->current.dirty & bitID)
		__glcurrent_switch	(&(__statebits->current), bitID,
							 &(current->current), &(target->current));
	if (__statebits->viewport.dirty & bitID)
		__glviewport_switch	(&(__statebits->viewport), bitID,
								 &(current->viewport), &(target->viewport));
	if (__statebits->fog.dirty & bitID)
		__glfog_switch	(&(__statebits->fog), bitID,
							 &(current->fog), &(target->fog));
	if (__statebits->texture.dirty & bitID)
		__gltexture_switch	(&(__statebits->texture), bitID,
							 &(current->texture), &(target->texture));
	if (__statebits->lists.dirty & bitID)
		__gllists_switch	(&(__statebits->lists), bitID,
							 &(current->lists), &(target->lists));
	if (__statebits->client.dirty & bitID)
		__glclient_switch	(&(__statebits->client), bitID,
							 &(current->client), &(target->client));
	if (__statebits->buffer.dirty & bitID)
		__glbuffer_switch	(&(__statebits->buffer), bitID,
							 &(current->buffer), &(target->buffer));
	if (__statebits->hint.dirty & bitID)
		__glhint_switch	(&(__statebits->hint), bitID,
							 &(current->hint), &(target->hint));
	if (__statebits->lighting.dirty & bitID)
		__gllighting_switch	(&(__statebits->lighting), bitID,
								 &(current->lighting), &(target->lighting));
	if (__statebits->line.dirty & bitID)
		__glline_switch	(&(__statebits->line), bitID,
							 &(current->line), &(target->line));
	if (__statebits->polygon.dirty & bitID)
		__glpolygon_switch	(&(__statebits->polygon), bitID,
							 &(current->polygon), &(target->polygon));
	if (__statebits->stencil.dirty & bitID)
		__glstencil_switch	(&(__statebits->stencil), bitID,
							 &(current->stencil), &(target->stencil));
	if (__statebits->eval.dirty & bitID)
		__gleval_switch	(&(__statebits->eval), bitID,
							 &(current->eval), &(target->eval));
	if (__statebits->imaging.dirty & bitID)
		__glimaging_switch	(&(__statebits->imaging), bitID,
							 &(current->imaging), &(target->imaging));
	if (__statebits->lists.dirty & bitID)
		__gllists_switch	(&(__statebits->lists), bitID,
							 &(current->lists), &(target->lists));
	if (__statebits->selection.dirty & bitID)
		__glselection_switch	(&(__statebits->selection), bitID,
								 &(current->selection), &(target->selection));
}

void 
__glcontext_sync(GLcontext *current, GLcontext *target) {
	GLbitvalue bitID = current->bitID;
	GLbitvalue update = target->update;

	if (update & GLUPDATE_TRANS && __statebits->trans.dirty & bitID)
		__gltrans_sync	(&(__statebits->trans), bitID,
							 &(current->trans), &(target->trans));
	if (update & GLUPDATE_PIXEL && __statebits->pixel.dirty & bitID)
		__glpixel_sync	(&(__statebits->pixel), bitID,
							 &(current->pixel), &(target->pixel));
	if (update & GLUPDATE_VIEWPORT && __statebits->viewport.dirty & bitID)
		__glviewport_sync	(&(__statebits->viewport), bitID,
								 &(current->viewport), &(target->viewport));
	if (update & GLUPDATE_FOG && __statebits->fog.dirty & bitID)
		__glfog_sync	(&(__statebits->fog), bitID,
							 &(current->fog), &(target->fog));
	if (update & GLUPDATE_TEXTURE && __statebits->texture.dirty & bitID)
		__gltexture_sync	(&(__statebits->texture), bitID,
							 &(current->texture), &(target->texture));
	if (update & GLUPDATE_LISTS && __statebits->lists.dirty & bitID)
		__gllists_sync	(&(__statebits->lists), bitID,
							 &(current->lists), &(target->lists));
	if (update & GLUPDATE_CLIENT && __statebits->client.dirty & bitID)
		__glclient_sync	(&(__statebits->client), bitID,
							 &(current->client), &(target->client));
	if (update & GLUPDATE_BUFFER && __statebits->buffer.dirty & bitID)
		__glbuffer_sync	(&(__statebits->buffer), bitID,
							 &(current->buffer), &(target->buffer));
	if (update & GLUPDATE_HINT && __statebits->hint.dirty & bitID)
		__glhint_sync	(&(__statebits->hint), bitID,
							 &(current->hint), &(target->hint));
	if (update & GLUPDATE_LIGHTING && __statebits->lighting.dirty & bitID)
		__gllighting_sync	(&(__statebits->lighting), bitID,
								 &(current->lighting), &(target->lighting));
	if (update & GLUPDATE_LINE && __statebits->line.dirty & bitID)
		__glline_sync	(&(__statebits->line), bitID,
							 &(current->line), &(target->line));
	if (update & GLUPDATE_POLYGON && __statebits->polygon.dirty & bitID)
		__glpolygon_sync	(&(__statebits->polygon), bitID,
							 &(current->polygon), &(target->polygon));
	if (update & GLUPDATE_STENCIL && __statebits->stencil.dirty & bitID)
		__glstencil_sync	(&(__statebits->stencil), bitID,
							 &(current->stencil), &(target->stencil));
	if (update & GLUPDATE_EVAL && __statebits->eval.dirty & bitID)
		__gleval_sync	(&(__statebits->eval), bitID,
							 &(current->eval), &(target->eval));
	if (update & GLUPDATE_IMAGING && __statebits->imaging.dirty & bitID)
		__glimaging_sync	(&(__statebits->imaging), bitID,
							 &(current->imaging), &(target->imaging));
	if (update & GLUPDATE_SELECTION && __statebits->selection.dirty & bitID)
		__glselection_sync	(&(__statebits->selection), bitID,
								 &(current->selection), &(target->selection));

	if (update & GLUPDATE_CURRENT && __statebits->current.dirty & bitID)
		__glcurrent_sync	(&(__statebits->current), bitID,
							 &(current->current), &(target->current));
	
	if (__config->draw_bbox)
		__glbucket_drawbbox();

	
}


GLcontext *
__glGetCurrentContext (void) {
	return __currentcontext;
}

void
__glDestroyContext(GLcontext *g) {
	__glbucket_destroy(&(g->bucket));
	__glbuffer_destroy(&(g->buffer));
	__glclient_destroy(&(g->client));
	__glcurrent_destroy(&(g->current));
	__gleval_destroy(&(g->eval));
	__glfog_destroy(&(g->fog));
	__glhint_destroy(&(g->hint));
	__glimaging_destroy(&(g->imaging));
	__gllighting_destroy(&(g->lighting));
	__glline_destroy(&(g->line));
	__gllists_destroy(&(g->lists));
	__glpixel_destroy(&(g->pixel));
	__glpolygon_destroy(&(g->polygon));
	__glstencil_destroy(&(g->stencil));
	__gltexture_destroy(&(g->texture));
	__gltrans_destroy(&(g->trans));
	__glviewport_destroy(&(g->viewport));

	if (g->vendor) free(g->vendor);
	if (g->renderer) free(g->renderer);
	if (g->version) free(g->version);
	if (g->extensions) free(g->extensions);

	__bitID |= g->bitID;
	
	free(g);
}


