/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include <stdlib.h>
#include <stdio.h>
#include <memory.h>
#include "glcontext.h"
#include "glerror.h"
#include "glhw.h"

#define GLCLIENT_NUMPOINTERS 6
#define GLCLIENT_INDEX_ALLOC 1024
#define GLCLIENT_DATA_ALLOC 1024
#define GLCLIENT_CACHE_ALLOC 1024
#define GLCLIENT_LIST_ALLOC 1024
#define GLCLIENT_BIT_ALLOC 1024

void (*__glhw_UpdateClientPointer) (int array, int index_size, int *index, int data_size, unsigned char * data);
void __glhw_UpdateClientPointer_nop (int array, int index_size, int * index, int data_size, unsigned char * data) { 
	UNUSED(array);
	UNUSED(index_size);
	UNUSED(index);
	UNUSED(data_size);
	UNUSED(data);
}

void 
__glclient_initbits (GLclientbits *c, GLconfig * cfg) {
	UNUSED(cfg);
	c->v = (GLbitvalue *) malloc (GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	c->n = (GLbitvalue *) malloc (GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	c->c = (GLbitvalue *) malloc (GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	c->i = (GLbitvalue *) malloc (GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	c->t = (GLbitvalue *) malloc (GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	c->e = (GLbitvalue *) malloc (GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	memset(c->v, 0, GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	memset(c->n, 0, GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	memset(c->c, 0, GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	memset(c->i, 0, GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	memset(c->t, 0, GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	memset(c->e, 0, GLCLIENT_BIT_ALLOC*sizeof(GLbitvalue));
	c->valloc = GLCLIENT_BIT_ALLOC;
	c->nalloc = GLCLIENT_BIT_ALLOC;
	c->calloc = GLCLIENT_BIT_ALLOC;
	c->ialloc = GLCLIENT_BIT_ALLOC;
	c->talloc = GLCLIENT_BIT_ALLOC;
	c->ealloc = GLCLIENT_BIT_ALLOC;
}

void
__glclient_init(GLclientstate *c, GLconfig *cfg) {
	GLclientpointer *lookup[6];
	GLclientpointer *cp;
	int i;

	lookup[0] = &(c->v);
	lookup[1] = &(c->c);
	lookup[2] = &(c->e);
	lookup[3] = &(c->i);
	lookup[4] = &(c->n);
	lookup[5] = &(c->t);

	__glhw_UpdateClientPointer = __glhw_UpdateClientPointer_nop;
	if (cfg->updateclientpointer)
		__glhw_UpdateClientPointer = cfg->updateclientpointer;
	
	c->maxelementsindices = cfg->maxelementsindices;
	c->maxelementsvertices = cfg->maxelementsvertices;

	c->list_alloc = GLCLIENT_LIST_ALLOC;
	c->list_size = 0;
	c->list = (int *) malloc (c->list_alloc * sizeof (int));

	for (i=0; i<GLCLIENT_NUMPOINTERS; i++) {
		cp = lookup[i];
		cp->p = NULL;
		cp->size = 0;
		cp->type = GL_NONE;
		cp->stride = 0;
		cp->enabled = 0;

		cp->cache_alloc = GLCLIENT_CACHE_ALLOC;
		cp->cache =	(unsigned char *) malloc (cp->cache_alloc);
		memset ((void *)(cp->cache), 0, cp->cache_alloc);
	}
}

void 
__glclient_destroy(GLclientstate *c) {
	GLclientpointer *lookup[6];
	GLclientpointer *cp;
	int i;

	lookup[0] = &(c->v);
	lookup[1] = &(c->c);
	lookup[2] = &(c->e);
	lookup[3] = &(c->i);
	lookup[4] = &(c->n);
	lookup[5] = &(c->t);


	free (c->list);

	for (i=0; i<GLCLIENT_NUMPOINTERS; i++) {
		cp = lookup[i];
		free(cp->cache);
	}
}

void
__glclient_finish(GLclientstate *c, GLclientbits *cb, GLbitvalue bitID) {
	UNUSED(cb);
	UNUSED(bitID);
	c->list_size = 0;
}

void 
__glclient_setclientstate(GLclientstate *c, GLclientbits *cb, GLbitvalue nbitID, 
						  GLenum array, GLboolean state) {
	switch (array) {
	case GL_VERTEX_ARRAY:
		c->v.enabled = state;
	break;
	case GL_COLOR_ARRAY:
		c->c.enabled = state;
	break;
	case GL_NORMAL_ARRAY:
		c->n.enabled = state;
	break;
	case GL_INDEX_ARRAY:
		c->i.enabled = state;
	break;
	case GL_TEXTURE_COORD_ARRAY:
		c->t.enabled = state;
	break;
	case GL_EDGE_FLAG_ARRAY:
		c->e.enabled = state;
	break;	
	default:
		if (__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "Invalid Enum passed to Enable/Disable Client State"))
			return;
	}
	cb->dirty = nbitID;
	cb->enableclientstate = nbitID;
}

void 
__glclient_SendUpdates(GLclientbits *c, GLbitvalue bitID, 
						GLclientstate * current, GLclientstate *target) {
	int i,j;
	GLbitvalue *invalidbits[6];
	GLclientpointer *cpointers[6];
	GLbitvalue nbitID = ~bitID;

	UNUSED(current);

	invalidbits[0] = c->v;
	invalidbits[1] = c->n;
	invalidbits[2] = c->c;
	invalidbits[3] = c->i;
	invalidbits[4] = c->t;
	invalidbits[5] = c->e;

	cpointers[0] = &(target->v);
	cpointers[1] = &(target->n);
	cpointers[2] = &(target->c);
	cpointers[3] = &(target->i);
	cpointers[4] = &(target->t);
	cpointers[5] = &(target->e);

	for (i=0; i<GLCLIENT_NUMPOINTERS; i++) {
		GLbitvalue *pbit = invalidbits[i];
		GLclientpointer *cp = cpointers[i];
		GLint list_size;
		GLint *list;
		unsigned char *p_cache;
		int *p_index;
		unsigned char *p_data;
		int bytesperindex;
		
		int index_size, data_size;
		static int index_alloc = GLCLIENT_INDEX_ALLOC;
		static int data_alloc = GLCLIENT_DATA_ALLOC;
		static int *index = NULL;
		static unsigned char *data = NULL;

		if (!index) index = (int *) malloc(index_alloc*sizeof(GLint));
		if (!data) data = (unsigned char *) malloc(data_alloc);

		if (!cp->enabled)
			continue;
		
		list_size = target->list_size;

		/* Check to make sure the index is big enough */
		while (index_alloc <= list_size) {
			index_alloc *= 2;
			index = (int *) realloc ((void *)index, index_alloc*sizeof(GLint));
		}

		/* Check to make sure the data is big enough */
		while (data_alloc <= list_size*cp->bytesperindex) {
			data_alloc *= 2;
			data = (unsigned char *) realloc ((void *)data, data_alloc);
		}

		/* Dereference outside of the loop */
		p_index = index;
		p_data = data;
		p_cache = cp->cache;
		bytesperindex = cp->bytesperindex;

		/* Compute what the update stream should be */
		for (j=0, list=target->list; j<list_size; j++, list++) {
			GLint elem = *list;
			GLbitvalue *bit = pbit+elem;

			/* Do we need to send this element? */
			if (*bit & bitID) {
				unsigned char *p;
				int k;
				p = p_cache + bytesperindex*elem;
				*p_index++ = elem;
				for (k=0; k<bytesperindex; k++)
					*p_data++ = *p++;

				/* Clear the bit */
				*bit &= nbitID;
			}
		}

		/* Compute the number of indexes */
		index_size = ((int) (((char *)p_index) - ((char *)index)))/sizeof(int);
		data_size = (int) (p_data - data);
		__glhw_UpdateClientPointer(i, index_size, index, data_size, data);
	}
	c->element &= nbitID;
}


void GLSTATE_DECL
__glstate_LockArraysEXT (GLint first, GLsizei count) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);
	unsigned char *p;
	int i, j;
	int index;
	int bytesperindex;
	GLclientpointer *lookup[6];
	GLbitvalue *invalidbits[6];
	GLint invalidbits_alloc[6];

	lookup[0] = &(c->v);
	lookup[1] = &(c->n);
	lookup[2] = &(c->c);
	lookup[3] = &(c->i);
	lookup[4] = &(c->t);
	lookup[5] = &(c->e);

	invalidbits[0] = cb->v;
	invalidbits[1] = cb->n;
	invalidbits[2] = cb->c;
	invalidbits[3] = cb->i;
	invalidbits[4] = cb->t;
	invalidbits[5] = cb->e;

	invalidbits_alloc[0] = cb->valloc;
	invalidbits_alloc[1] = cb->nalloc;
	invalidbits_alloc[2] = cb->calloc;
	invalidbits_alloc[3] = cb->ialloc;
	invalidbits_alloc[4] = cb->talloc;
	invalidbits_alloc[5] = cb->ealloc;



	if (IsBucketingActive()) 
		g->bucket.flush(g);

	/* add it to the list */	
	while (c->list_alloc <= c->list_size+count) {
		c->list_alloc *= 2;
		c->list = (int *) realloc (c->list, c->list_alloc*sizeof(GLint));
	}

	for (index=first; index<first+count; index++) {

		c->list[c->list_size++] = index;

		for (i=0; i<GLCLIENT_NUMPOINTERS; i++) {
			GLclientpointer *cp = lookup[i];
			unsigned char *p_cache;
			
			if (!cp->enabled)
				continue;

			bytesperindex = cp->bytesperindex;

			/* Compare it to the cache value */
			p = cp->p + cp->stride*index;
			p_cache = cp->cache + bytesperindex*index;
			j=0;
			if (cp->cache_alloc < bytesperindex*index)
				for (; j<bytesperindex; j++)
					if (*p++ != *p_cache++)
						break;

			/* If invalid, copy to cache*/
			if (j!=bytesperindex)	{
				/* Check to make sure the cache is big enough */
				while (cp->cache_alloc <= index*bytesperindex) {
					cp->cache = (unsigned char *) realloc ((void*)cp->cache, cp->cache_alloc*2);
					memset((void*) (cp->cache+cp->cache_alloc), 0, cp->cache_alloc);
					cp->cache_alloc *= 2;
				}

				/* Copy to the cache */
				p = cp->p + cp->stride*index;
				p_cache = cp->cache + bytesperindex*index;
				for (j=0;j<bytesperindex; j++)
					*p_cache++ = *p++;

				/* check to see if the bit vector is long enough */
				while (invalidbits_alloc[i] <= index) {	
					GLbitvalue **invalidbitsptr[6];
					GLint *invalidbits_allocptr[6];

					invalidbitsptr[0] = &(cb->v);
					invalidbitsptr[1] = &(cb->n);
					invalidbitsptr[2] = &(cb->c);
					invalidbitsptr[3] = &(cb->i);
					invalidbitsptr[4] = &(cb->t);
					invalidbitsptr[5] = &(cb->e);

					invalidbits_allocptr[0] = &(cb->valloc);
					invalidbits_allocptr[1] = &(cb->nalloc);
					invalidbits_allocptr[2] = &(cb->calloc);
					invalidbits_allocptr[3] = &(cb->ialloc);
					invalidbits_allocptr[4] = &(cb->talloc);
					invalidbits_allocptr[5] = &(cb->ealloc);
					
					*invalidbitsptr[i] = (GLbitvalue *) realloc((void *)invalidbits[i], invalidbits_alloc[i]*2*sizeof(GLbitvalue));
					*invalidbits_allocptr[i]*=2;
					invalidbits[i] = *invalidbitsptr[i];
				}

				/* set the invalid bits */
				*(invalidbits[i]+index) = g->nbitID;
				cb->element = g->nbitID;
				cb->dirty = g->nbitID;
			}

		}
	}

	__glhw_LockArrays();
}

void GLSTATE_DECL
__glstate_UnlockArraysEXT (void) {
	__glhw_UnlockArrays();
}

void GLSTATE_DECL
__glstate_EnableClientState (GLenum array) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	__glclient_setclientstate(c, cb, g->nbitID, array, GL_TRUE);
}

void GLSTATE_DECL
__glstate_DisableClientState (GLenum array) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	__glclient_setclientstate(c, cb, g->nbitID, array, GL_FALSE);
}

void 
__glclient_setcppointer (GLclientpointer *cp, GLint size, GLenum type, GLsizei stride, const GLvoid *pointer) {
	cp->p = (unsigned char *) pointer;
	cp->size = size;
	cp->type = type;
	// Calculate the bytes per index for address calculation
	cp->bytesperindex = size;
	switch (type) {
	case GL_BYTE:
	case GL_UNSIGNED_BYTE:
		break;
	case GL_SHORT:
	case GL_UNSIGNED_SHORT:
		cp->bytesperindex *= sizeof(GLshort);
		break;
	case GL_INT:
	case GL_UNSIGNED_INT:
		cp->bytesperindex *= sizeof(GLint);
		break;
	case GL_FLOAT:
		cp->bytesperindex *= sizeof(GLfloat);
		break;
	case GL_DOUBLE:
		cp->bytesperindex *= sizeof(GLdouble);
		break;
	}

	/* 
	**  Note: If stide==0 then we set the 
	**  stride equal address offset
	**  therefore stride can never equal
	**  zero.
	*/
	cp->stride = stride;
	if (!stride) cp->stride = cp->bytesperindex;
}

void GLSTATE_DECL
__glstate_VertexPointer(GLint size, GLenum type, GLsizei stride, const GLvoid *p) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	if (size != 2 &&
		size != 3 &&
		size != 4)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "glVertexPointer: invalid size: %d", size))
			return;
	if (type != GL_SHORT &&
		type != GL_INT &&
		type != GL_FLOAT &&
		type != GL_DOUBLE)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glVertexPointer: invalid type: %d", type))
			return;
	if (stride < 0) 
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "glVertexPointer: stride was negative: %d", stride))
			return;
	
	__glclient_setcppointer(&(c->v), size, type, stride, p);
	cb->dirty = g->nbitID;
	cb->clientpointer = g->nbitID;
}

void GLSTATE_DECL
__glstate_ColorPointer(GLint size, GLenum type, GLsizei stride, const GLvoid *p) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	if (size != 3 &&
		size != 4)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "glColorPointer: invalid size: %d", size))
			return;
	if (type != GL_BYTE &&
		type != GL_UNSIGNED_BYTE &&
		type != GL_SHORT &&
		type != GL_UNSIGNED_SHORT &&
		type != GL_INT &&
		type != GL_UNSIGNED_INT &&
		type != GL_FLOAT &&
		type != GL_DOUBLE)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glColorPointer: invalid type: %d", type))
			return;
	if (stride < 0) 
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "glColorPointer: stride was negative: %d", stride))
			return;

	__glclient_setcppointer(&(c->c), size, type, stride, p);
	cb->dirty = g->nbitID;
	cb->clientpointer = g->nbitID;
}

void GLSTATE_DECL
__glstate_IndexPointer(GLenum type, GLsizei stride, const GLvoid *p) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	if (type != GL_SHORT &&
		type != GL_INT &&
		type != GL_FLOAT &&
		type != GL_DOUBLE)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glIndexPointer: invalid type: %d", type))
			return;
	if (stride < 0) 
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "glIndexPointer: stride was negative: %d", stride))
			return;

	__glclient_setcppointer(&(c->i), 1, type, stride, p);
	cb->dirty = g->nbitID;
	cb->clientpointer = g->nbitID;
}

void GLSTATE_DECL
__glstate_NormalPointer(GLenum type, GLsizei stride, const GLvoid *p) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	if (type != GL_BYTE &&
		type != GL_SHORT &&
		type != GL_INT &&
		type != GL_FLOAT &&
		type != GL_DOUBLE)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glNormalPointer: invalid type: %d", type))
			return;
	if (stride < 0) 
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "glNormalPointer: stride was negative: %d", stride))
			return;

	__glclient_setcppointer(&(c->n), 3, type, stride, p);
	cb->dirty = g->nbitID;
	cb->clientpointer = g->nbitID;
}

void GLSTATE_DECL
__glstate_TexCoordPointer(GLint size, GLenum type, GLsizei stride, const GLvoid *p) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	if (size != 1 &&
		size != 2 &&
		size != 3 &&
		size != 4)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "glTexCoordPointer: invalid size: %d", size))
			return;
	if (type != GL_SHORT &&
		type != GL_INT &&
		type != GL_FLOAT &&
		type != GL_DOUBLE)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glTexCoordPointer: invalid type: %d", type))
			return;
	if (stride < 0) 
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "glTexCoordPointer: stride was negative: %d", stride))
			return;

	__glclient_setcppointer(&(c->t), size, type, stride, p);
	cb->dirty = g->nbitID;
	cb->clientpointer = g->nbitID;
}

void GLSTATE_DECL
__glstate_EdgeFlagPointer(GLsizei stride, const GLvoid *p) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	if (stride < 0) 
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE, "glTexCoordPointer: stride was negative: %d", stride))
			return;

	__glclient_setcppointer(&(c->e), 1, GL_UNSIGNED_BYTE, stride, p);
	cb->dirty = g->nbitID;
	cb->clientpointer = g->nbitID;
}


/* 
** Currently I treat Interleaved Arrays as if the 
** user uses them as separate arrays.
** Certainly not the most efficient method but it 
** lets me use the same glDrawArrays method.
*/
void GLSTATE_DECL 
__glstate_InterleavedArrays(GLenum format, GLsizei stride, const GLvoid *p) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);
	GLclientpointer *cp;
	unsigned char *base = (unsigned char *) p;

	if (g->current.beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION, "glInterleavedArrays called in begin/end"))
			return;

	if (IsBucketingActive()) 
		g->bucket.flush(g);

	if (stride < 0)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION, "glInterleavedArrays: stride < 0: %d", stride))
			return;

	switch (format) {
	case GL_T4F_C4F_N3F_V4F:
	case GL_T2F_C4F_N3F_V3F:
	case GL_C4F_N3F_V3F:
	case GL_T4F_V4F:
	case GL_T2F_C3F_V3F:
	case GL_T2F_N3F_V3F:
	case GL_C3F_V3F:
	case GL_N3F_V3F:
	case GL_T2F_C4UB_V3F:
	case GL_T2F_V3F:
	case GL_C4UB_V3F:
	case GL_V3F:
	case GL_C4UB_V2F:
	case GL_V2F:
		break;
	default:
		if(__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glInterleavedArrays: Unrecognized format: %d", format))
			return;
	}

	cb->dirty = g->nbitID;
	cb->clientpointer = g->nbitID;

/* p, size, type, stride, enabled, bytesperindex */
/*
**  VertexPointer 
*/
	
	cp = &(c->v);
	cp->type = GL_FLOAT;
	cp->enabled = GL_TRUE;

	switch (format) {
	case GL_T4F_C4F_N3F_V4F:
		cp->p = base+4*sizeof(GLfloat)+4*sizeof(GLfloat)+3*sizeof(GLfloat);
		cp->size = 4;
		break;
	case GL_T2F_C4F_N3F_V3F:
		cp->p = base+2*sizeof(GLfloat)+4*sizeof(GLfloat)+3*sizeof(GLfloat);
		cp->size = 3;
		break;
	case GL_C4F_N3F_V3F:
		cp->p = base+4*sizeof(GLfloat)+3*sizeof(GLfloat);
		cp->size = 3;
		break;
	case GL_T4F_V4F:
		cp->p = base+4*sizeof(GLfloat);
		cp->size = 4;
		break;
	case GL_T2F_C3F_V3F:
		cp->p = base+2*sizeof(GLfloat)+3*sizeof(GLfloat);
		cp->size = 3;
		break;
	case GL_T2F_N3F_V3F:
		cp->p = base+2*sizeof(GLfloat)+3*sizeof(GLfloat);
		cp->size = 3;
		break;
	case GL_C3F_V3F:
		cp->p = base+3*sizeof(GLfloat);
		cp->size = 3;
		break;
	case GL_N3F_V3F:
		cp->p = base+3*sizeof(GLfloat);
		cp->size = 3;
		break;
	case GL_T2F_C4UB_V3F:
		cp->p = base+2*sizeof(GLfloat)+4*sizeof(GLubyte);
		cp->size = 3;
		break;
	case GL_T2F_V3F:
		cp->p = base+2*sizeof(GLfloat);
		cp->size = 2;
		break;
	case GL_C4UB_V3F:
		cp->p = base+4*sizeof(GLubyte);
		cp->size = 3;
		break;
	case GL_V3F:
		cp->p = base;
		cp->size = 3;
		break;
	case GL_C4UB_V2F:
		cp->p = base+4*sizeof(GLubyte);
		cp->size = 2;
		break;
	case GL_V2F:
		cp->p = base;
		cp->size = 2;
		break;
	default:
		if(__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glInterleavedArrays: Unrecognized format: %d", format))
			return;
	}

	cp->bytesperindex = cp->size * sizeof (GLfloat);

	if (stride)
		cp->stride = stride + (cp->p - base);
	else
		cp->stride = cp->bytesperindex + (cp->p - base);

/*
**  NormalPointer
*/

	cp = &(c->n);
	cp->enabled = GL_TRUE;
	switch (format) {
	case GL_T4F_C4F_N3F_V4F:
		cp->p = base+4*sizeof(GLfloat)+4*sizeof(GLfloat);
		cp->stride = 4*sizeof(GLfloat)+4*sizeof(GLfloat)+stride;
		if (!stride) cp->stride += 3*sizeof(GLfloat)+4*sizeof(GLfloat);
		break;
	case GL_T2F_C4F_N3F_V3F:
		cp->p = base+2*sizeof(GLfloat)+4*sizeof(GLfloat);
		cp->stride = 2*sizeof(GLfloat)+4*sizeof(GLfloat)+stride;
		if (!stride) cp->stride += 3*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_C4F_N3F_V3F:
		cp->p = base+4*sizeof(GLfloat);
		cp->stride = 4*sizeof(GLfloat)+stride;
		if (!stride) cp->stride += 3*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_T2F_N3F_V3F:
		cp->p = base+2*sizeof(GLfloat);
		cp->stride = 2*sizeof(GLfloat)+stride;
		if (!stride) cp->stride += 3*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_N3F_V3F:
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride += 3*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_T4F_V4F:
	case GL_T2F_C3F_V3F:
	case GL_C3F_V3F:
	case GL_T2F_C4UB_V3F:
	case GL_T2F_V3F:
	case GL_C4UB_V3F:
	case GL_V3F:
	case GL_C4UB_V2F:
	case GL_V2F:
		cp->enabled = GL_FALSE;
	default:
		if(__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glInterleavedArrays: Unrecognized format: %d", format))
			return;
	}

	if (cp->enabled) {
		cp->type = GL_FLOAT;
		cp->size = 3;
		cp->bytesperindex = cp->size * sizeof (GLfloat);
	}
	
/*
**  ColorPointer
*/

	cp = &(c->c);
	cp->enabled = GL_TRUE;
	switch (format) {
	case GL_T4F_C4F_N3F_V4F:
		cp->size = 4;
		cp->type = GL_FLOAT;
		cp->bytesperindex = cp->size * sizeof(GLfloat);
		cp->p = base+4*sizeof(GLfloat);
		cp->stride = 4*sizeof(GLfloat)+stride;
		if (!stride) cp->stride += 4*sizeof(GLfloat)+3*sizeof(GLfloat)+4*sizeof(GLfloat);
		break;
	case GL_T2F_C4F_N3F_V3F:
		cp->size = 4;
		cp->type = GL_FLOAT;
		cp->bytesperindex = cp->size * sizeof(GLfloat);
		cp->p = base+2*sizeof(GLfloat);
		cp->stride = 2*sizeof(GLfloat)+stride;
		if (!stride) cp->stride += 4*sizeof(GLfloat)+3*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_C4F_N3F_V3F:
		cp->size = 4;
		cp->type = GL_FLOAT;
		cp->bytesperindex = cp->size * sizeof(GLfloat);
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride += 4*sizeof(GLfloat)+3*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_T2F_C3F_V3F:
		cp->size = 3;
		cp->type = GL_FLOAT;
		cp->bytesperindex = cp->size * sizeof(GLfloat);
		cp->p = base+2*sizeof(GLfloat);
		cp->stride = 2*sizeof(GLfloat)+stride;
		if (!stride) cp->stride += 3*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_C3F_V3F:
		cp->size = 3;
		cp->type = GL_FLOAT;
		cp->bytesperindex = cp->size * sizeof(GLfloat);
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride += 3*sizeof(GLfloat) + 3*sizeof(GLfloat);
		break;
	case GL_T2F_C4UB_V3F:
		cp->size = 4;
		cp->type = GL_UNSIGNED_BYTE;
		cp->bytesperindex = cp->size * sizeof(GLubyte);
		cp->p = base+2*sizeof(GLfloat);
		cp->stride = 2*sizeof(GLfloat)+stride;
		if (!stride) cp->stride += 4*sizeof(GLubyte)+2*sizeof(GLfloat);
		break;
	case GL_C4UB_V3F:
		cp->size = 4;
		cp->type = GL_UNSIGNED_BYTE;
		cp->bytesperindex = cp->size * sizeof(GLubyte);
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride += 4*sizeof(GLubyte)+3*sizeof(GLfloat);
		break;
	case GL_C4UB_V2F:
		cp->size = 4;
		cp->type = GL_UNSIGNED_BYTE;
		cp->bytesperindex = cp->size * sizeof(GLubyte);
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride += 4*sizeof(GLubyte)+2*sizeof(GLfloat);
		break;
	case GL_T2F_N3F_V3F:
	case GL_N3F_V3F:
	case GL_T4F_V4F:
	case GL_T2F_V3F:
	case GL_V3F:
	case GL_V2F:
		cp->enabled = GL_FALSE;
	default:
		if(__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glInterleavedArrays: Unrecognized format: %d", format))
			return;
	}

/*
**  TexturePointer
*/

	cp = &(c->t);
	cp->enabled = GL_TRUE;
	switch (format) {
	case GL_T4F_C4F_N3F_V4F:
		cp->size = 4;
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride = 4*sizeof(GLfloat)+4*sizeof(GLfloat)+3*sizeof(GLfloat)+4*sizeof(GLfloat);
		break;
	case GL_T2F_C4F_N3F_V3F:
		cp->size = 2;
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride = 2*sizeof(GLfloat)+4*sizeof(GLfloat)+3*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_T2F_C3F_V3F:
	case GL_T2F_N3F_V3F:
		cp->size = 2;
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride = 2*sizeof(GLfloat)+3*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_T2F_C4UB_V3F:
		cp->size = 2;
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride = 2*sizeof(GLfloat)+4*sizeof(GLubyte)+3*sizeof(GLfloat);
		break;
	case GL_T4F_V4F:
		cp->size = 4;
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride = 4*sizeof(GLfloat)+4*sizeof(GLfloat);
		break;
	case GL_T2F_V3F:
		cp->size = 2;
		cp->p = base;
		cp->stride = stride;
		if (!stride) cp->stride = 2*sizeof(GLfloat)+3*sizeof(GLfloat);
		break;
	case GL_C4UB_V3F:
	case GL_C4UB_V2F:
	case GL_C3F_V3F:
	case GL_C4F_N3F_V3F:
	case GL_N3F_V3F:
	case GL_V3F:
	case GL_V2F:
		cp->enabled = GL_FALSE;
	default:
		if(__glerror(__LINE__, __FILE__, GL_INVALID_ENUM, "glInterleavedArrays: Unrecognized format: %d", format))
			return;
	}

	if (cp->enabled) {
		cp->type = GL_FLOAT;
		cp->bytesperindex = cp->size * sizeof (GLfloat);
	}	
}


void GLSTATE_DECL
__glstate_ArrayElement_cache(GLint index) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLstatebits *sb = GetStateBits();
	GLclientbits *cb = &(sb->client);
	GLclientpointer *cp = &(c->v);
	unsigned char *p;
	unsigned char *p_cache;
	int i, j;
	float x=0.0f, y=0.0f, z=0.0f, w=1.0f;
	int bytesperindex = c->v.bytesperindex;
	GLclientpointer *lookup[6];
	GLbitvalue *invalidbits[6];
	GLint invalidbits_alloc[6];

	GLbucketstate *bkt = &(g->bucket);

	lookup[0] = &(c->v);
	lookup[1] = &(c->n);
	lookup[2] = &(c->c);
	lookup[3] = &(c->i);
	lookup[4] = &(c->t);
	lookup[5] = &(c->e);

	invalidbits[0] = cb->v;
	invalidbits[1] = cb->n;
	invalidbits[2] = cb->c;
	invalidbits[3] = cb->i;
	invalidbits[4] = cb->t;
	invalidbits[5] = cb->e;

	invalidbits_alloc[0] = cb->valloc;
	invalidbits_alloc[1] = cb->nalloc;
	invalidbits_alloc[2] = cb->calloc;
	invalidbits_alloc[3] = cb->ialloc;
	invalidbits_alloc[4] = cb->talloc;
	invalidbits_alloc[5] = cb->ealloc;



   	/* Begin the bucketing */
	if (!IsBucketingActive()) 
		__glbucket_begin();
	else if (*bkt->count > *bkt->max)
		bkt->flush(g);

	if (index < 0)
		UNIMPLEMENTED();

	__glhw_ArrayElement(index);

	/* add it to the list */
	while (c->list_alloc <= c->list_size) {
		c->list_alloc *= 2;
		c->list = (int *) realloc (c->list, c->list_alloc*sizeof(GLint));
	}
	c->list[c->list_size++] = index;
	
	/* Update the cacheing */
	for (i=0; i<GLCLIENT_NUMPOINTERS; i++) {
		cp = lookup[i];
		if (cp->enabled) {

			bytesperindex = cp->bytesperindex;

			/* Compare it to the cache value */
			p = cp->p + cp->stride*index;
			p_cache = cp->cache + bytesperindex*index;
			j=0;
			if (cp->cache_alloc < bytesperindex*index)
				for (; j<bytesperindex; j++)
					if (*p++ != *p_cache++)
						break;

			/* If invalid, copy to cache */
			if (j!=bytesperindex)	{
				/* Check to make sure the cache is big enough */
				while (cp->cache_alloc <= index*bytesperindex) {
					cp->cache = (unsigned char *) realloc ((void*)cp->cache, cp->cache_alloc*2);
					memset((void*) (cp->cache+cp->cache_alloc), 0, cp->cache_alloc);
					cp->cache_alloc *= 2;
				}

				/* Copy to the cache */
				p = cp->p + cp->stride*index;
				p_cache = cp->cache + bytesperindex*index;
				for (j=0;j<bytesperindex; j++)
					*p_cache++ = *p++;

				/* check to see if the bit vector is long enough */
				while (invalidbits_alloc[i] <= index) {	
					GLbitvalue **invalidbitsptr[6];
					GLint *invalidbits_allocptr[6];

					invalidbitsptr[0] = &(cb->v);
					invalidbitsptr[1] = &(cb->n);
					invalidbitsptr[2] = &(cb->c);
					invalidbitsptr[3] = &(cb->i);
					invalidbitsptr[4] = &(cb->t);
					invalidbitsptr[5] = &(cb->e);

					invalidbits_allocptr[0] = &(cb->valloc);
					invalidbits_allocptr[1] = &(cb->nalloc);
					invalidbits_allocptr[2] = &(cb->calloc);
					invalidbits_allocptr[3] = &(cb->ialloc);
					invalidbits_allocptr[4] = &(cb->talloc);
					invalidbits_allocptr[5] = &(cb->ealloc);

					
					*invalidbitsptr[i] = (GLbitvalue *) realloc((void *)invalidbits[i], invalidbits_alloc[i]*2*sizeof(GLbitvalue));
					*invalidbits_allocptr[i]*=2;
					invalidbits[i] = *invalidbitsptr[i];
				}

				/* set the invalid bits */
				*(invalidbits[i]+index) = g->nbitID;
				cb->element = g->nbitID;
				cb->dirty = g->nbitID;
			}

			/* update the bucketing */
			if (!i) {
				p = cp->cache + (bytesperindex * index);
				switch (cp->type) {
				case GL_SHORT:
					x = (float) *((GLshort *) p);
					y = (float) *((GLshort *) (p+2));
					z = (float) *((GLshort *) (p+4));
					break;
				case GL_INT:
					x = (float) *((GLint *) p);
					y = (float) *((GLint *) (p+4));
					z = (float) *((GLint *) (p+8));
					break;
				case GL_FLOAT:
					x = *((GLfloat *) p);
					y = *((GLfloat *) (p+4));
					z = *((GLfloat *) (p+8));
					break;
				case GL_DOUBLE:
					x = (float) *((GLdouble *) p);
					y = (float) *((GLdouble *) (p+8));
					z = (float) *((GLdouble *) (p+16));
					break;
				default:
					UNIMPLEMENTED();
				}

				if (cp->size == 4) {
					switch (cp->type) {
					case GL_SHORT:
						w = (float) *((GLshort *) (p+6));
						break;
					case GL_INT:
						w = (float) *((GLint *) (p+12));
						break;
					case GL_FLOAT:
						w = *((GLfloat *) (p+12));
						break;
					case GL_DOUBLE:
						w = (float) *((GLdouble *) (p+24));
						break;
					default:
						UNIMPLEMENTED();
					}
					x /= w;
					y /= w;
					z /= w;
				}

				if (cp->size == 2)
					z = 0.0f;
				/* Update the bounding box */
				if (x < bkt->bmin->x) bkt->bmin->x = x;
				if (y < bkt->bmin->y) bkt->bmin->y = y;
				if (z < bkt->bmin->z) bkt->bmin->z = z;
				if (x > bkt->bmax->x) bkt->bmax->x = x;
				if (y > bkt->bmax->y) bkt->bmax->y = y;
				if (z > bkt->bmax->z) bkt->bmax->z = z;
				(*bkt->count)++;
			}
		}	
	}
}


void GLSTATE_DECL
__glstate_ArrayElement_locked(GLint index) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLclientpointer *cp;
	unsigned char *p;
	int bytesperindex;
	float x=0.0f, y=0.0f, z=0.0f, w=1.0f;
	GLbucketstate *bkt = &(g->bucket);

   	/* Begin the bucketing */
	if (!IsBucketingActive()) 
		__glbucket_begin();
	else if (*bkt->count > *bkt->max)
		bkt->flush(g);

	if (index < 0)
		UNIMPLEMENTED();

	__glhw_ArrayElement(index);

	cp = &(c->v);
	if (cp->enabled) {
		/* update the bucketing */
		bytesperindex = cp->bytesperindex;
		p = cp->cache + (bytesperindex * index);
		switch (cp->type) {
		case GL_SHORT:
			x = (float) *((GLshort *) p);
			y = (float) *((GLshort *) (p+2));
			z = (float) *((GLshort *) (p+4));
			break;
		case GL_INT:
			x = (float) *((GLint *) p);
			y = (float) *((GLint *) (p+4));
			z = (float) *((GLint *) (p+8));
			break;
		case GL_FLOAT:
			x = *((GLfloat *) p);
			y = *((GLfloat *) (p+4));
			z = *((GLfloat *) (p+8));
			break;
		case GL_DOUBLE:
			x = (float) *((GLdouble *) p);
			y = (float) *((GLdouble *) (p+8));
			z = (float) *((GLdouble *) (p+16));
			break;
		default:
			UNIMPLEMENTED();
		}

		if (cp->size == 4) {
			switch (cp->type) {
			case GL_SHORT:
				w = (float) *((GLshort *) (p+6));
				break;
			case GL_INT:
				w = (float) *((GLint *) (p+12));
				break;
			case GL_FLOAT:
				w = *((GLfloat *) (p+12));
				break;
			case GL_DOUBLE:
				w = (float) *((GLdouble *) (p+24));
				break;
			default:
				UNIMPLEMENTED();
			}
			x /= w;
			y /= w;
			z /= w;
		}

		if (cp->size == 2)
			z = 0.0f;
		/* Update the bounding box */
		if (x < bkt->bmin->x) bkt->bmin->x = x;
		if (y < bkt->bmin->y) bkt->bmin->y = y;
		if (z < bkt->bmin->z) bkt->bmin->z = z;
		if (x > bkt->bmax->x) bkt->bmax->x = x;
		if (y > bkt->bmax->y) bkt->bmax->y = y;
		if (z > bkt->bmax->z) bkt->bmax->z = z;
		(*bkt->count)++;
	}
}


/*
** ArrayElement_resolve
** This code is for debugging purposes 
** only.  It resolves the vertex array 
** information on the client for comparison
** testing of the caching versions.
*/
void GLSTATE_DECL
__glstate_ArrayElement (GLint index) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLbucketstate *bkt = &(g->bucket);
	float x = 0.0f, y = 0.0f, z = 0.0f;
	float w = 1.0f;
	unsigned char *p;

	if (index < 0)
		UNIMPLEMENTED();

   	/* Begin the bucketing */
	if (!IsBucketingActive()) 
		__glbucket_begin();
	else if (*bkt->count > *bkt->max)
		bkt->flush(g);

	if (c->e.enabled) {
		__glhw_EdgeFlagv(c->e.p + index*c->e.stride);
	}
	if (c->t.enabled) {
		p = c->t.p + index*c->t.stride;
		switch (c->t.type) {
		case GL_SHORT:
			switch (c->t.size) {
			case 1:
				__glhw_TexCoord1sv((GLshort *)p);
				break;
			case 2:
				__glhw_TexCoord2sv((GLshort *)p);
				break;
			case 3:
				__glhw_TexCoord3sv((GLshort *)p);
				break;
			case 4:
				__glhw_TexCoord4sv((GLshort *)p);
				break;
			}
			break;
		case GL_INT:
			switch (c->t.size) {
			case 1:
				__glhw_TexCoord1iv((GLint *)p);
				break;
			case 2:
				__glhw_TexCoord2iv((GLint *)p);
				break;
			case 3:
				__glhw_TexCoord3iv((GLint *)p);
				break;
			case 4:
				__glhw_TexCoord4iv((GLint *)p);
				break;
			}
			break;
		case GL_FLOAT:
			switch (c->t.size) {
			case 1:
				__glhw_TexCoord1fv((GLfloat *)p);
				break;
			case 2:
				__glhw_TexCoord2fv((GLfloat *)p);
				break;
			case 3:
				__glhw_TexCoord3fv((GLfloat *)p);
				break;
			case 4:
				__glhw_TexCoord4fv((GLfloat *)p);
				break;
			}
			break;
		case GL_DOUBLE:
			switch (c->t.size) {
			case 1:
				__glhw_TexCoord1dv((GLdouble *)p);
				break;
			case 2:
				__glhw_TexCoord2dv((GLdouble *)p);
				break;
			case 3:
				__glhw_TexCoord3dv((GLdouble *)p);
				break;
			case 4:
				__glhw_TexCoord4dv((GLdouble *)p);
				break;
			}
			break;
		}
	}
	if (c->i.enabled) {
		p = c->i.p + index*c->i.stride;
		switch (c->i.type) {
		case GL_SHORT:
			__glhw_Indexsv((GLshort *)p);
			break;
		case GL_INT:
			__glhw_Indexiv((GLint *)p);
			break;
		case GL_FLOAT:
			__glhw_Indexfv((GLfloat *)p);
			break;
		case GL_DOUBLE:
			__glhw_Indexdv((GLdouble *)p);
			break;
		}
	}
	if (c->c.enabled) {
		p = c->c.p + index*c->c.stride;
		switch (c->c.type) {
		case GL_BYTE:
			switch (c->c.size) {
			case 3:
				__glhw_Color3bv((GLbyte *)p);
				break;
			case 4:
				__glhw_Color4bv((GLbyte *)p);
				break;
			}
			break;
		case GL_UNSIGNED_BYTE:
			switch (c->c.size) {
			case 3:
				__glhw_Color3ubv((GLubyte *)p);
				break;
			case 4:
				__glhw_Color4ubv((GLubyte *)p);
				break;
			}
			break;
		case GL_SHORT:
			switch (c->c.size) {
			case 3:
				__glhw_Color3sv((GLshort *)p);
				break;
			case 4:
				__glhw_Color4sv((GLshort *)p);
				break;
			}
			break;
		case GL_UNSIGNED_SHORT:
			switch (c->c.size) {
			case 3:
				__glhw_Color3usv((GLushort *)p);
				break;
			case 4:
				__glhw_Color4usv((GLushort *)p);
				break;
			}
			break;
		case GL_INT:
			switch (c->c.size) {
			case 3:
				__glhw_Color3iv((GLint *)p);
				break;
			case 4:
				__glhw_Color4iv((GLint *)p);
				break;
			}
			break;
		case GL_UNSIGNED_INT:
			switch (c->c.size) {
			case 3:
				__glhw_Color3uiv((GLuint *)p);
				break;
			case 4:
				__glhw_Color4uiv((GLuint *)p);
				break;
			}
			break;
		case GL_FLOAT:
			switch (c->c.size) {
			case 3:
				__glhw_Color3fv((GLfloat *)p);
				break;
			case 4:
				__glhw_Color4fv((GLfloat *)p);
				break;
			}
			break;
		case GL_DOUBLE:
			switch (c->c.size) {
			case 3:
				__glhw_Color3dv((GLdouble *)p);
				break;
			case 4:
				__glhw_Color4dv((GLdouble *)p);
				break;
			}
			break;
		}
	}
	if (c->n.enabled) {
		p = c->n.p + index*c->n.stride;
		switch (c->n.type) {
		case GL_BYTE:
			__glhw_Normal3bv((GLbyte *)p);
			break;
		case GL_SHORT:
			__glhw_Normal3sv((GLshort *)p);
			break;
		case GL_INT:
			__glhw_Normal3iv((GLint *)p);
			break;
		case GL_FLOAT:
			__glhw_Normal3fv((GLfloat *)p);
			break;
		case GL_DOUBLE:
			__glhw_Normal3dv((GLdouble *)p);
			break;
		}
	}
	if (c->v.enabled) {
		p = c->v.p + (index*c->v.stride);

		switch (c->v.type) {
		case GL_SHORT:
			x = (float) *((GLshort *) p);
			y = (float) *((GLshort *) (p+2));
			switch (c->v.size) {
			case 2:
				__glhw_Vertex2sv((GLshort *)p);
				break;
			case 3:
				__glhw_Vertex3sv((GLshort *)p);
				z = (float) *((GLshort *) (p+4));
				break;
			case 4:
				__glhw_Vertex4sv((GLshort *)p);
				z = (float) *((GLshort *) (p+4));
				w = (float) *((GLshort *) (p+6));
				break;
			}
			break;
		case GL_INT:
			x = (float) *((GLint *) p);
			y = (float) *((GLint *) (p+4));
			switch (c->v.size) {
			case 2:
				__glhw_Vertex2iv((GLint *)p);
				break;
			case 3:
				__glhw_Vertex3iv((GLint *)p);
				z = (float) *((GLint *) (p+8));
				break;
			case 4:
				__glhw_Vertex4iv((GLint *)p);
				z = (float) *((GLint *) (p+8));
				w = (float) *((GLint *) (p+12));
				break;
			}
			break;
		case GL_FLOAT:
			x = (float) *((GLfloat *) p);
			y = (float) *((GLfloat *) (p+4));
			switch (c->v.size) {
			case 2:
				__glhw_Vertex2fv((GLfloat *)p);
				break;
			case 3:
				__glhw_Vertex3fv((GLfloat *)p);
				z = (float) *((GLfloat *) (p+8));
				break;
			case 4:
				__glhw_Vertex4fv((GLfloat *)p);
				z = (float) *((GLfloat *) (p+8));
				w = (float) *((GLfloat *) (p+12));
				break;
			}
			break;
		case GL_DOUBLE:
			x = (float) *((GLdouble *) p);
			y = (float) *((GLdouble *) (p+8));
			switch (c->v.size) {
			case 2:
				__glhw_Vertex2dv((GLdouble *)p);
				break;
			case 3:
				__glhw_Vertex3dv((GLdouble *)p);
				z = (float) *((GLdouble *) (p+16));
				break;
			case 4:
				__glhw_Vertex4dv((GLdouble *)p);
				z = (float) *((GLdouble *) (p+16));
				w = (float) *((GLdouble *) (p+24));
				break;
			}
			break;
		}
		/* update the bucketing */
		x /= w;
		y /= w;
		z /= w;
		if (x < bkt->bmin->x) bkt->bmin->x = x;
		if (y < bkt->bmin->y) bkt->bmin->y = y;
		if (z < bkt->bmin->z) bkt->bmin->z = z;
		if (x > bkt->bmax->x) bkt->bmax->x = x;
		if (y > bkt->bmax->y) bkt->bmax->y = y;
		if (z > bkt->bmax->z) bkt->bmax->z = z;
		(*bkt->count)++;
	}
}

void 
__glclient_UpdateClientPointer (int array, int index_size, int *index, int data_size, unsigned char * data) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	GLclientpointer *cp;
	GLclientpointer *lookup[6];
	
	unsigned char *p;
	unsigned char *cache;
	int i, j;
	int bytesperindex;

	UNUSED(data_size);

	lookup[0] = &(c->v);
	lookup[1] = &(c->n);
	lookup[2] = &(c->c);
	lookup[3] = &(c->i);
	lookup[4] = &(c->t);
	lookup[5] = &(c->e);

	/*
	printf("UpdateClientPointer:\n");
	printf("\tarray:%d\n", array);
	printf("\tindex_size:%d\n", index_size);
	for (i=0; i<index_size; i++)
		printf("\t\t%d\n", index[i]);
    printf("\tdata_size:%d\n", data_size);
	*/

	cp = lookup[array];
	if (cp->enabled) {
		bytesperindex = cp->bytesperindex;
		cache = cp->cache;
		for (i=0; i<index_size; i++) {
			int elem = index[i];
	
			/* Increase the cache size if necessary */
			while (cp->cache_alloc <= elem * cp->bytesperindex) {
				cp->cache = (unsigned char *) realloc ((void *) cp->cache, cp->cache_alloc*2);
				memset((void *) (cp->cache + cp->cache_alloc), 0, cp->cache_alloc);
				cp->cache_alloc *= 2;
				if (cache != cp->cache) {
					cache = cp->cache;
					cp->p = cp->cache;
					switch (array) {
					case 0:
						__glhw_VertexPointer(cp->size, cp->type, 0, cp->p);
						break;
					case 1:
						__glhw_NormalPointer(cp->type, 0, cp->p);
						break;
					case 2:
						__glhw_ColorPointer(cp->size, cp->type, 0, cp->p);
						break;
					case 3:
						__glhw_IndexPointer(cp->type, 0, cp->p);
						break;
					case 4:
						__glhw_TexCoordPointer(cp->size, cp->type, 0, cp->p);
						break;
					case 5:
						__glhw_EdgeFlagPointer(0, cp->p);
						break;
					default:					
						__glerror(__LINE__, __FILE__, GL_NO_ERROR,
							"Client pointers > 6!");
					}
				}
			}
			p = cache + elem * bytesperindex;
			for (j=0; j<bytesperindex; j++)
				*p++ = *data++;
		}
	}
}

void GLSTATE_DECL
__glstate_GetPointerv(GLenum pname, GLvoid * * params) {
	GLcontext *g = GetCurrentContext();
	GLclientstate *c = &(g->client);
	
	if (g->current.beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION,
			"GetPointerv called in begin/end"))
			return;

	switch (pname) {
	case GL_VERTEX_ARRAY:
		*params = (GLvoid *) c->v.p;
	break;
	case GL_COLOR_ARRAY:
		*params = (GLvoid *) c->c.p;
	break;
	case GL_NORMAL_ARRAY:
		*params = (GLvoid *) c->n.p;
	break;
	case GL_INDEX_ARRAY:
		*params = (GLvoid *) c->i.p;
	break;
	case GL_TEXTURE_COORD_ARRAY:
		*params = (GLvoid *) c->t.p;
	break;
	case GL_EDGE_FLAG_ARRAY:
		*params = (GLvoid *) c->e.p;
	break;
	default:
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION,
			"glGetPointerv: invalid pname: %d", pname))
			return;
	}
}

void GLSTATE_DECL
__glstate_DrawArrays(GLenum mode, GLint first, GLsizei count) {
	GLcontext *g = GetCurrentContext();
	int i;
	
	if (g->current.beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION,
			"DrawArrays called in begin/end"))
			return;

	if (count < 0)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE,
			"DrawArrays passed negative count: %d", count))
			return;

	if (mode > GL_POLYGON)
		if (__glerror (__LINE__, __FILE__, GL_INVALID_ENUM,
			"DrawArrays called with invalid mode: %d", mode))
			return;

	__glstate_Begin (mode);
	for (i=0; i<count; i++) 
		__glstate_ArrayElement(first++);
	__glstate_End();
}

void GLSTATE_DECL
__glstate_DrawElements( GLenum mode,  GLsizei count,  GLenum type,  const GLvoid *indices) {
	GLcontext *g = GetCurrentContext();
	int i;
	GLubyte *p = (GLubyte *)indices;

	if (g->current.beginend)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_OPERATION,
			"DrawElements called in begin/end"))
			return;

	if (count < 0)
		if (__glerror(__LINE__, __FILE__, GL_INVALID_VALUE,
			"DrawArrays passed negative count: %d", count))
			return;

	if (mode > GL_POLYGON)
		if (__glerror (__LINE__, __FILE__, GL_INVALID_ENUM,
			"DrawArrays called with invalid mode: %d", mode))
			return;

	if (type != GL_UNSIGNED_BYTE &&
		type != GL_UNSIGNED_SHORT &&
		type != GL_UNSIGNED_INT)
		if (__glerror (__LINE__, __FILE__, GL_INVALID_ENUM,
			"DrawArrays called with invalid type: %d", type))
			return;
	
	__glstate_Begin (mode);
	switch (type) {
	case GL_UNSIGNED_BYTE:
		for (i=0; i<count; i++)
			__glstate_ArrayElement((GLint) *p++);
		break;
	case GL_UNSIGNED_SHORT:
		for (i=0; i<count; i++) {
			__glstate_ArrayElement((GLint) * (GLushort *) p);
			p+=sizeof (GLushort);
		}
		break;
	case GL_UNSIGNED_INT:
		for (i=0; i<count; i++) {
			__glstate_ArrayElement((GLint) * (GLuint *) p);
			p+=sizeof (GLuint);
		}
		break;
	default:
		UNIMPLEMENTED();
	}
	__glstate_End();
}
