/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "__utilsource.h"

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#pragma warning( push, 3 )
#include <winsock2.h>
#pragma warning( pop )
#pragma warning( disable : 4514 )
#pragma warning( disable : 4127 )
typedef int ssize_t;
#define write(a,b,c) send(a,b,c,0)
#else
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#endif

#include <limits.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>

#include "wiregl/include/wiregl_util.h"

typedef enum {
	WireGLTcpipMemory,
	WireGLTcpipMemoryBig
} WireGLTcpipBufferKind;

#define WIREGL_TCPIP_BUFFER_MAGIC 0x89134532

typedef struct WireGLTcpipBuffer {
	unsigned int          magic;
	WireGLTcpipBufferKind kind;
	unsigned int          len;
	unsigned int          pad;
} WireGLTcpipBuffer;

#ifdef WINDOWS

#undef  ECONNRESET
#define ECONNRESET  WSAECONNRESET
#undef  EINTR
#define EINTR       WSAEINTR

UTIL_DECL int wireGLTcpipErrno( void )
{
	return WSAGetLastError( );
}

UTIL_DECL char *wireGLTcpipErrorString( int err )
{
    static char buf[512], *temp;

	sprintf( buf, "err=%d", err );

#define X(x)	strcpy(buf,x); break

	switch ( err )
	{
	  case WSAECONNREFUSED: X( "connection refused" );
	  case WSAECONNRESET:   X( "connection reset" );
	  default:
		FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER |
					   FORMAT_MESSAGE_FROM_SYSTEM |
					   FORMAT_MESSAGE_MAX_WIDTH_MASK, NULL, err,
					   MAKELANGID( LANG_NEUTRAL, SUBLANG_DEFAULT ),
					   (LPTSTR) &temp, 0, NULL );
		if ( temp )
		{
			strncpy( buf, temp, sizeof(buf)-1 );
			buf[ sizeof(buf)-1 ] = '\0';
		}
	}

#undef X

	temp = buf + strlen(buf) - 1;
	while ( temp > buf && isspace( *temp ) )
	{
		*temp = '\0';
		temp--;
	}

	return buf;
}

#else

UTIL_DECL int wireGLTcpipErrno( void )
{
	int err = errno;
	errno = 0;
	return err;
}

UTIL_DECL char *wireGLTcpipErrorString( int err )
{
	static char buf[512], *temp;
	
	temp = strerror( err );
	if ( temp )
	{
		strncpy( buf, temp, sizeof(buf)-1 );
		buf[ sizeof(buf)-1 ] = '\0';
	}
	else
	{
		sprintf( buf, "err=%d", err );
	}

	return buf;
}

#endif

UTIL_DECL void 
wireGLCloseSocket( WireGLSocket sock )
{
	int fail;
#ifdef WINDOWS
    fail = ( closesocket( sock ) != 0 );
#else
	shutdown( sock, 2 /* RDWR */ );
	fail = ( close( sock ) != 0 );
#endif
	if ( fail )
	{
		int err = wireGLTcpipErrno( );
		wireGLWarning( WIREGL_WARN_DEBUG, "wireGLCloseSocket( sock=%d ): %s",
					   sock, wireGLTcpipErrorString( err ) );
	}
}

#if 0
/**************************************************************************/

static FILE *log_file = NULL;

static void
wireGLLogOpen( void )
{
	char *file;
	log_file = NULL;
	file = getenv( "WIREGL_LOG" );
	if ( file )
	{
		wireGLWarning( WIREGL_WARN_DEBUG, "wireGLLogOpen: logging to \"%s\"",
					   file );
		log_file = fopen( file, "wb" );
		if ( log_file )
		{
			unsigned __int64 freq;
			QueryPerformanceFrequency( (LARGE_INTEGER *) &freq );
			fprintf( log_file, "freq %I64u\n", freq );
		}
		else
		{
			wireGLWarning( WIREGL_WARN_CRITICAL, "wireGLLogOpen: trouble "
						   "opening \"%s\"", file );
		}
	}
}

static void
wireGLLogRead( unsigned int len )
{
	if ( log_file ) 
	{
		unsigned __int64 time;
		QueryPerformanceCounter( (LARGE_INTEGER *) &time );
		fprintf( log_file, "read %u %I64u\n", len, time );
	}
}

static void
wireGLLogWrite( unsigned int len )
{
	if ( log_file ) 
	{
		unsigned __int64 time;
		QueryPerformanceCounter( (LARGE_INTEGER *) &time );
		fprintf( log_file, "write %u %I64u\n", len, time );
	}
}

static void
wireGLLogClose( void )
{
	if ( log_file )
	{
		fclose( log_file );
		log_file = NULL;
	}
}

#endif

/**************************************************************************/

static struct {
	int                  initialized;
	int                  num_conns;
	WireGLConnection   **conns;
	WireGLBufferPool     bufpool;
	WireGLNetReceiveFunc recv;
	WireGLNetCloseFunc   close;
	struct {
		WireGLSocket     sock;
		WireGLVoidFunc   connect;
	} server;
} wiregl_tcpip;

static int 
TcpipReadExact( WireGLSocket sock, void *buf, unsigned int len )
{
    char *dst = (char *) buf;

    while ( len > 0 )
    {
        int num_read = recv( sock, dst, (int) len, 0 );

#ifdef WINDOWS_XXXX
		/* MWE: why is this necessary for windows???  Does it return a
           "good" value for num_bytes despite having a reset
           connection? */
		if ( wireGLTcpipErrno( ) == ECONNRESET )
            return -1;
#endif

        if ( num_read < 0 )
		{
			if ( wireGLTcpipErrno( ) == EINTR )
			{
				wireGLWarning( WIREGL_WARN_DEBUG, "read interrupted, "
							   "looping for more data" );
				continue;
			}
            return -1;
		}
        
		if ( num_read == 0 ) 
		{
			/* client exited gracefully */
			return 0;
        }

		dst += num_read;
		len -= num_read;
    }

    return 1;
}

UTIL_DECL void 
wireGLTcpipReadExact( WireGLSocket sock, void *buf, unsigned int len )
{
	int retval = TcpipReadExact( sock, buf, len );
	if ( retval <= 0 )
	{
		int err = wireGLTcpipErrno( );
		wireGLSimpleError( "wireGLTcpipReadExact: %s", 
						   wireGLTcpipErrorString( err ) );
	}
}

static int 
TcpipWriteExact( WireGLSocket sock, void *buf, unsigned int len )
{
    char *src = (char *) buf;

    while ( len > 0 )
    {
		int num_written = write( sock, src, len );
        if ( num_written <= 0 )
        {
			if ( wireGLTcpipErrno( ) == EINTR )
			{
				wireGLWarning( WIREGL_WARN_DEBUG, "TcpipWriteExact: "
							   "caught an EINTR, continuing" );
				continue;
			}
			return -1;
        }

        len -= num_written;
        src += num_written;
    }

    return 1;
}

UTIL_DECL void 
wireGLTcpipWriteExact( WireGLSocket sock, void *buf, unsigned int len )
{
	int retval = TcpipWriteExact( sock, buf, len );
	if ( retval <= 0 )
	{
		int err = wireGLTcpipErrno( );
		wireGLSimpleError( "wireGLTcpipWriteExact: %s", 
						   wireGLTcpipErrorString( err ) );
	}
}

UTIL_DECL void *
wireGLTcpipAlloc( void )
{
	WireGLTcpipBuffer *buf = (WireGLTcpipBuffer *)
		wireGLBufferPoolPop( &wiregl_tcpip.bufpool );
	if ( buf == NULL )
	{
		buf = (WireGLTcpipBuffer *) 
			wireGLAlloc( sizeof(WireGLTcpipBuffer) + __wiregl_max_send );
		buf->magic = WIREGL_TCPIP_BUFFER_MAGIC;
		buf->kind  = WireGLTcpipMemory;
		buf->pad   = 0;
	}
	return (void *)( buf + 1 );
}

UTIL_DECL void 
wireGLTcpipSend( WireGLConnection *conn, void **bufp,
				 void *start, unsigned int len )
{
	WireGLTcpipBuffer *tcpip_buffer;
	unsigned int      *lenp;

	if ( bufp == NULL )
	{
		/* we are doing synchronous sends from user memory, so no need
		 * to get fancy.  Simply write the length & the payload and
		 * return. */
		wireGLTcpipWriteExact( conn->tcp_socket, &len, sizeof(len) );
		wireGLTcpipWriteExact( conn->tcp_socket, start, len );
		return;
	}

	tcpip_buffer = (WireGLTcpipBuffer *)(*bufp) - 1;

	wireGLAssert( tcpip_buffer->magic == WIREGL_TCPIP_BUFFER_MAGIC );

	/* All of the buffers passed to the send function were allocated
     * with wireGLTcpipAlloc(), which includes a header with a 4 byte
     * length field, to insure that we always have a place to write
     * the length field, even when start == *bufp. */
	lenp = (unsigned int *) start - 1;
	*lenp = len;

	if ( TcpipWriteExact( conn->tcp_socket, lenp, len + sizeof(int) ) < 0 )
	{
		int err = wireGLTcpipErrno( );
		wireGLSimpleError( "wireGLTcpipSend: %s",
						   wireGLTcpipErrorString( err ) );
	}

#if 0
	wireGLLogWrite( len );
#endif

	/* reclaim this pointer for reuse and try to keep the client from
	   accidentally reusing it directly */
	wireGLBufferPoolPush( &wiregl_tcpip.bufpool, tcpip_buffer );
	*bufp = NULL;
}

static void 
TcpipDeadConnection( WireGLConnection *conn )
{
#if 1
	wireGLWarning( WIREGL_WARN_DEBUG, "Dead connection (sock=%d, host=%s)",
				   conn->tcp_socket, conn->hostname );
	if ( wiregl_tcpip.server.connect )
	{
		wireGLWarning( WIREGL_WARN_DEBUG, "Closing server socket" );
		wireGLCloseSocket( wiregl_tcpip.server.sock );
	}
	exit( 0 );
#else
    WireGLConnection *conn = wireGLLookupClientConnBySocket( sock );
    handlers->disconnect( conn->sender_id );
    wireGLCloseSocket( sock );
    wireGLDeleteClientConn( conn );
    wireGLFree( conn );
#endif
}

static int
wireGLTcpipSelect( int n, fd_set *readfds, struct timeval *timeout )
{
	for ( ; ; ) 
	{
		int err, num_ready;

		num_ready = select( n, readfds, NULL, NULL, timeout );

		if ( num_ready >= 0 )
		{
			return num_ready;
		}

		err = wireGLTcpipErrno( );
		if ( err == EINTR )
		{
			wireGLWarning( WIREGL_WARN_CRITICAL, "select interruped, "
						   "trying again" );
		}
		else
		{
			wireGLSimpleError( "select failed: %s", 
							   wireGLTcpipErrorString( err ) );
		}
	}
}


UTIL_DECL int
wireGLTcpipRecv( void )
{
	int    num_ready, max_fd;
	fd_set read_fds;
	int i;

	max_fd = 0;
	FD_ZERO( &read_fds );
	if ( wiregl_tcpip.server.connect )
	{
		FD_SET( wiregl_tcpip.server.sock, &read_fds );
		max_fd = (int) wiregl_tcpip.server.sock + 1;
	}
	for ( i = 0; i < wiregl_tcpip.num_conns; i++ )
	{
		WireGLConnection *conn = wiregl_tcpip.conns[i];
		if ( conn->recv_credits > 0 || conn->type != WIREGL_TCPIP )
		{
			/* NOTE: may want to always puts the FD in the descriptor
               set so we'll notice broken connections.  Down in the
               loop that iterates over the ready sockets only peek
               (MSG_PEEK flag to recv()?) if the connection isn't
               enabled. */
			WireGLSocket sock = conn->tcp_socket;
			if ( (int) sock + 1 > max_fd )
				max_fd = (int) sock + 1;
			FD_SET( sock, &read_fds );
		}
	}

	if ( wiregl_tcpip.num_conns )
	{
		struct timeval timeout;
		timeout.tv_sec = 0;
		timeout.tv_usec = 0;
		num_ready = wireGLTcpipSelect( max_fd, &read_fds, &timeout );
	}
	else
	{
		wireGLWarning( WIREGL_WARN_DEBUG, "Waiting for first connection..." );
		num_ready = wireGLTcpipSelect( max_fd, &read_fds, NULL );
	}

	if ( num_ready == 0 )
		return 0;

	if ( wiregl_tcpip.server.connect && 
		 FD_ISSET( wiregl_tcpip.server.sock, &read_fds ) )
	{
		wiregl_tcpip.server.connect( );
	}

	for ( i = 0; i < wiregl_tcpip.num_conns; i++ )
	{
		WireGLTcpipBuffer *tcpip_buffer;
		unsigned int       len;
		int                read_ret;
		WireGLConnection  *conn = wiregl_tcpip.conns[i];
		WireGLSocket       sock = conn->tcp_socket;

		if ( !FD_ISSET( sock, &read_fds ) )
			continue;

		read_ret = TcpipReadExact( sock, &len, sizeof(len) );
		if ( read_ret <= 0 )
		{
			TcpipDeadConnection( conn );
			i--;
			continue;
		}

		wireGLAssert( len > 0 );

		if ( len <= __wiregl_max_send )
		{
			tcpip_buffer = (WireGLTcpipBuffer *) wireGLTcpipAlloc( ) - 1;
		}
		else
		{
			tcpip_buffer = (WireGLTcpipBuffer *) 
				wireGLAlloc( sizeof(*tcpip_buffer) + len );
			tcpip_buffer->magic = WIREGL_TCPIP_BUFFER_MAGIC;
			tcpip_buffer->kind  = WireGLTcpipMemoryBig;
			tcpip_buffer->pad   = 0;
		}

		tcpip_buffer->len = len;

		read_ret = TcpipReadExact( sock, tcpip_buffer + 1, len );
		if ( read_ret <= 0 )
		{
			wireGLFree( tcpip_buffer );
			TcpipDeadConnection( conn );
			i--;
			continue;
		}

#if 0
		wireGLLogRead( len );
#endif

		conn->recv_credits -= len;
		wiregl_tcpip.recv( conn, tcpip_buffer + 1, len );
	}

	return 1;
}

UTIL_DECL void
wireGLTcpipFree( WireGLConnection *conn, void *buf )
{
	WireGLTcpipBuffer *tcpip_buffer = (WireGLTcpipBuffer *) buf - 1;

	wireGLAssert( tcpip_buffer->magic == WIREGL_TCPIP_BUFFER_MAGIC );
	conn->recv_credits += tcpip_buffer->len;

	switch ( tcpip_buffer->kind )
	{
	  case WireGLTcpipMemory:
		wireGLBufferPoolPush( &wiregl_tcpip.bufpool, tcpip_buffer );
		break;

	  case WireGLTcpipMemoryBig:
		wireGLFree( tcpip_buffer );
		break;

	  default:
		wireGLAbort( );
	}

}

UTIL_DECL void
wireGLTcpipInit( WireGLNetReceiveFunc recvFunc, WireGLNetCloseFunc closeFunc )
{
	if ( wiregl_tcpip.initialized )
	{
		wireGLAssert( wiregl_tcpip.recv == recvFunc );
		wireGLAssert( wiregl_tcpip.close == closeFunc );
		return;
	}

#if 0
	wireGLLogOpen( );
#endif

	wiregl_tcpip.num_conns = 0;
	wiregl_tcpip.conns     = NULL;
	
	wiregl_tcpip.server.sock    = 0;
	wiregl_tcpip.server.connect = NULL;

	wireGLBufferPoolInit( &wiregl_tcpip.bufpool, 16 );

	wiregl_tcpip.recv = recvFunc;
	wiregl_tcpip.close = closeFunc;

	wiregl_tcpip.initialized = 1;
}

UTIL_DECL void 
wireGLTcpipConnection( WireGLConnection *conn )
{
	int n_bytes;

	wireGLAssert( wiregl_tcpip.initialized );

    conn->type  = WIREGL_TCPIP;
	conn->Alloc = wireGLTcpipAlloc;
	conn->Send  = wireGLTcpipSend;
	conn->Free  = wireGLTcpipFree;

	n_bytes = ( wiregl_tcpip.num_conns + 1 ) * sizeof(*wiregl_tcpip.conns);
	wireGLRealloc( (void **) &wiregl_tcpip.conns, n_bytes );

	wiregl_tcpip.conns[wiregl_tcpip.num_conns++] = conn;
}

UTIL_DECL void
wireGLTcpipBecomeServer( WireGLSocket sock, WireGLVoidFunc connectFunc )
{
	wireGLAssert( wiregl_tcpip.server.connect == NULL );
	wireGLAssert( connectFunc != NULL );

	wiregl_tcpip.server.sock    = sock;
	wiregl_tcpip.server.connect = connectFunc;
}
