/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "__utilsource.h"

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#pragma warning( push, 3 )
#include <winsock2.h>
#pragma warning( pop )
#pragma warning( disable : 4514 )
#else
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>

#include "wiregl/include/wiregl_util.h"
#include "wiregl/include/wiregl_common.h"
#include "wiregl/include/wiregl_protocol.h"

#ifdef WINDOWS
#define EADDRINUSE   WSAEADDRINUSE
#define ECONNREFUSED WSAECONNREFUSED
#endif

#define WIREGL_BIND_TRIES 8
#define WIREGL_INITIAL_RECV_CREDITS ( 1 << 21 )

unsigned int __wiregl_max_send = 0;

static struct {
	int                  initialized;
	WireGLNetReceiveFunc recv;
	WireGLNetCloseFunc   close;
	int                  use_gm;
	int                  num_clients;
	struct {
		WireGLSocket         sock;
		WireGLNetConnectFunc connect;
	} server;
} wiregl_net;

UTIL_DECL void
wireGLNetBytesToString( char *string, int nstring, void *data, int ndata )
{
	int i, offset;
	unsigned char *udata;

	offset = 0;
	udata = (unsigned char *) data;
	for ( i = 0; i < ndata && ( offset + 4 <= nstring ); i++ ) 
	{
		offset += sprintf( string + offset, "%02x ", udata[i] );
	}

	if ( i == ndata && offset > 0 )
		string[offset-1] = '\0';
	else
		strcpy( string + offset - 3, "..." );
}

UTIL_DECL void
wireGLNetWordsToString( char *string, int nstring, void *data, int ndata )
{
	int i, offset, ellipsis;
	unsigned int *udata;

	/* turn byte count into word count */
	ndata /= 4;

	/* we need an ellipsis if all the words won't fit in the string */
	ellipsis = ( ndata * 9 > nstring );
	if ( ellipsis )
	{
		ndata = nstring / 9;

		/* if the ellipsis won't fit then print one less word */
		if ( ndata * 9 + 3 > nstring )
			ndata--;
	}
		
	offset = 0;
	udata = (unsigned int *) data;
	for ( i = 0; i < ndata; i++ ) 
	{
		offset += sprintf( string + offset, "%08x ", udata[i] );
	}

	if ( ellipsis )
		strcpy( string + offset, "..." );
	else if ( offset > 0 )
		string[offset-1] = 0;
}

static void
wireGLFrobSocket( WireGLSocket sock )
{
    int sndbuf = 64*1024;
	int rcvbuf = sndbuf;
	int tcp_nodelay = 1;

    if ( setsockopt( sock, SOL_SOCKET, SO_SNDBUF,
					 (char *) &sndbuf, sizeof(sndbuf) ) )
	{
		int err = wireGLTcpipErrno( );
		wireGLWarning( WIREGL_WARN_CRITICAL, "setsockopt( SO_SNDBUF=%d ) : %s",
					   sndbuf, wireGLTcpipErrorString( err ) );
	}

    if ( setsockopt( sock, SOL_SOCKET, SO_RCVBUF,
					 (char *) &rcvbuf, sizeof(rcvbuf) ) )
	{
		int err = wireGLTcpipErrno( );
		wireGLWarning( WIREGL_WARN_CRITICAL, "setsockopt( SO_RCVBUF=%d ) : %s",
					   rcvbuf, wireGLTcpipErrorString( err ) );
	}

	if ( setsockopt( sock, IPPROTO_TCP, TCP_NODELAY,
					 (char *) &tcp_nodelay, sizeof(tcp_nodelay) ) )
	{
		int err = wireGLTcpipErrno( );
		wireGLWarning( WIREGL_WARN_CRITICAL, "setsockopt( TCP_NODELAY=%d )"
					   " : %s", tcp_nodelay, wireGLTcpipErrorString( err ) );
	}
}

UTIL_DECL WireGLConnection *
wireGLConnectToServer( char *server, unsigned short default_port, WireGLConnectionRequest *request )
{
    char hostname[4096], protocol[4096];
    unsigned short port;
	unsigned int request_size;
    WireGLConnection *conn;
    WireGLConnectionRequest local_request;
	WireGLConnectionRequest *p;
    WireGLConnectionResponse response;
	int i;

	wireGLAssert( wiregl_net.initialized );
	wireGLTcpipInit( wiregl_net.recv, wiregl_net.close );

    wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "Connecting to %s", server );

    if (!request)
    {
        request = &local_request;
    }
    else
    {
        // windows doesn't let us directly set this global variable
        // (cross DLL mumble-mumble), so sneak in the value
        wireGLAssert( __wiregl_max_send == 0 ||
                      __wiregl_max_send == request->max_send );
        __wiregl_max_send = request->max_send;
    }

    request->magic    = WIREGL_CONNECTION_MAGIC;
	request->size     = sizeof(*request);
    request->max_send = __wiregl_max_send;

    if ( !wireGLParseURL( server, protocol, hostname, &port, default_port ) )
    {
        wireGLSimpleError( "Malformed URL: \"%s\"", server );
    }

    conn = (WireGLConnection *) wireGLAlloc( sizeof(*conn) );

	conn->type         = WIREGL_NO_CONNECTION;
	conn->sender_id    = 0;
	conn->pending_writebacks = 0;
	conn->total_bytes  = 0;
	conn->send_credits = 0;
	conn->recv_credits = WIREGL_INITIAL_RECV_CREDITS;
    conn->hostname     = strdup( hostname );
	conn->Alloc        = NULL;
	conn->Send         = NULL;
	conn->Free         = NULL;
	conn->tcp_socket   = 0;
	conn->gm_node_id   = 0;

    request->please_use_gm = 0;
	request->gm_node_id    = 0; /* GM_NO_SUCH_NODE_ID */
	request->gm_port_num   = 0;

    if ( !strcmp( protocol, "devnull" ) )
    {
		wireGLDevNullInit( wiregl_net.recv, wiregl_net.close );
		wireGLDevNullConnection( conn );
        return conn;
    }
#ifdef GM_SUPPORT
    else if ( !strcmp( protocol, "gm" ) )
    {
		/* just note that we are trying to setup GM, we'll have to
           finish this later... */
        conn->type = WIREGL_GM;
        request->please_use_gm = 1;

		wireGLGmInit( wiregl_net.recv, wiregl_net.close );
		request->gm_node_id = wireGLGmNodeId( );
		request->gm_port_num = wireGLGmPortNum( );
    }
#endif
    else if ( !strcmp( protocol, "tcpip" ) )
    {
		conn->type = WIREGL_TCPIP;
	}
	else
	{
        wireGLSimpleError( "Unknown Protocol: \"%s\"", protocol );
    }

	for ( i = 0; i < WIREGL_BIND_TRIES; i++ )
	{
		struct sockaddr_in servaddr;
		struct hostent *hp;
		int err;

		conn->tcp_socket = socket( AF_INET, SOCK_STREAM, 0 );
		if ( conn->tcp_socket < 0 )
		{
			int err = wireGLTcpipErrno( );
			wireGLSimpleError( "socket error: %s", 
							   wireGLTcpipErrorString( err ) );
		}

		wireGLFrobSocket( conn->tcp_socket );

		hp = gethostbyname( hostname );
		if ( !hp )
		{
			wireGLSimpleError( "Unknown host: \"%s\"", hostname );
		}

		memset( &servaddr, 0, sizeof(servaddr) );
		servaddr.sin_family = AF_INET;
		servaddr.sin_port = htons( port );

		memcpy( (char *) &servaddr.sin_addr, hp->h_addr,
				sizeof(servaddr.sin_addr) );
		if ( !connect( conn->tcp_socket, (struct sockaddr *) &servaddr,
					  sizeof(servaddr) ) )
			break;

		err = wireGLTcpipErrno( );
		if ( err == EADDRINUSE || err == ECONNREFUSED )
		{
			wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, 
						   "Couldn't connect to %s:%d, %s",
						   hostname, port, wireGLTcpipErrorString( err ) );
			port += 100;
		}
		else if ( err == EINTR )
		{
			wireGLWarning( WIREGL_WARN_DEBUG, "connection to %s:%d "
						   "interruped, trying again", hostname, port );
		}
		else
		{
			wireGLSimpleError( "Couldn't connect to %s:%d, %s",
							   hostname, port, wireGLTcpipErrorString( err ) );
		}
	}

	if ( i == WIREGL_BIND_TRIES )
	{
		wireGLSimpleError( "Couldn't find server %s", hostname );
	}

	if ( i > 0 )
	{
		wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "Connected on alternate "
					   "port %u", port );
	}

    if ( request->max_send < 1024 )
    {
        wireGLSimpleError( "wireGLConnectToServer: max_send too small (%d)\n",
                           request->max_send );
    }


	/* calculate the request size */
	request_size = sizeof(*request);
	for (p = request; p->peer[0]; p++)
		request_size += sizeof(*request);

	wireGLTcpipWriteExact( conn->tcp_socket, request, request_size );

	wireGLTcpipReadExact( conn->tcp_socket, &response, sizeof(response) );

    if ( response.magic != WIREGL_CONNECTION_MAGIC )
    {
        wireGLSimpleError( "wireGLFinishConnectToServer: connection "
                           "magic=0x%x, expected 0x%x",
                           response.magic, WIREGL_CONNECTION_MAGIC );
    }

	if ( response.size != sizeof(response) )
	{
		wireGLSimpleError( "wireGLFinishConnectToServer: server claims "
						   "response is %u bytes, I expect %u bytes\n",
						   response.size, sizeof(response) );
	}

    if ( response.max_send < __wiregl_max_send )
    {
        wireGLSimpleError( "wireGLFinishConnectToServer: server has a "
                           "max_send=%d, but client is using %d",
                           response.max_send, __wiregl_max_send );
    }

    conn->sender_id = response.client_id;

	switch ( conn->type )
	{
#ifdef GM_SUPPORT
	  case WIREGL_GM:
		/* set this up as a TCPIP connection first, so that we can
         * detect a client disconnect via socket death */
		wireGLTcpipConnection( conn );

		/* and now set it up correctly for GM */
		conn->gm_node_id  = response.gm_node_id;
		conn->gm_port_num = response.gm_port_num;
		wiregl_net.use_gm++;
        wireGLGmConnection( conn );
		break;
#endif
	  case WIREGL_TCPIP:
		wireGLTcpipConnection( conn );
		break;

	  default:
		wireGLError( "wireGLFinishConnectToServer: conn->type=%u?",
					 conn->type );
	}

    wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "Done connecting to server." );

	return conn;
}

#if defined( WINDOWS ) || defined( IRIX ) || defined( IRIX64 )
typedef int socklen_t;
#endif

UTIL_DECL void
wireGLNetAcceptClient( void )
{
    WireGLConnection         *conn;
	WireGLConnectionRequest  *request;
    WireGLConnectionResponse  response;
	struct sockaddr           addr;
	socklen_t                 addr_length;
	struct hostent           *host;
	struct in_addr            sin_addr;

	int i;

	conn = (WireGLConnection *) wireGLAlloc( sizeof( *conn ) );

    addr_length = sizeof(addr);
    conn->tcp_socket = accept( wiregl_net.server.sock,
                               (struct sockaddr *) &addr,
                               &addr_length );
    if ( conn->tcp_socket == -1 )
    {
		int err = wireGLTcpipErrno( );
        wireGLSimpleError( "accept() failed: %s",
						   wireGLTcpipErrorString( err ) );
    }

	sin_addr = ((struct sockaddr_in *) &addr)->sin_addr;
	host = gethostbyaddr( (char *) &sin_addr, sizeof(sin_addr), AF_INET );
	if ( host == NULL )
	{
		char *temp = inet_ntoa( sin_addr );
		conn->hostname = (char *) wireGLAlloc( strlen(temp) + 1 );
		strcpy( conn->hostname, temp );
	}
	else
	{
		char *temp;

		conn->hostname = (char *) wireGLAlloc( strlen( host->h_name ) + 1 );
		strcpy( conn->hostname, host->h_name );

		/* remove the trailing part of the hostname */
		temp = conn->hostname;
		while ( *temp && *temp != '.' )
			temp++;
		*temp = '\0';
	}

	wireGLWarning( WIREGL_WARN_DEBUG, "Accepting connection from \"%s\"",
				   conn->hostname );

    conn->type         = WIREGL_TCPIP;
	conn->sender_id    = 0;
	conn->pending_writebacks = 0;
	conn->total_bytes  = 0;
	conn->send_credits = 0;
	conn->recv_credits = WIREGL_INITIAL_RECV_CREDITS;
					   
	conn->Alloc        = NULL;
	conn->Send         = NULL;
	conn->Free         = NULL;
					   
	conn->gm_node_id   = 0; /* GM_NO_SUCH_NODE_ID */

	request = (WireGLConnectionRequest*) wireGLAlloc( sizeof(*request) );
	for ( i = 0; ; i++ ) 
	{
		wireGLTcpipReadExact( conn->tcp_socket, request+i, sizeof(*request) );
		if ( !request[i].peer[0] ) break;
		wireGLRealloc ((void **) &request, sizeof(*request) * ( i + 2 ) );
	}

    if ( request->magic != WIREGL_CONNECTION_MAGIC )
    {
        wireGLSimpleError( "wireGLAcceptClient: connection "
                           "magic=0x%x, expected 0x%x\n",
                           request->magic, WIREGL_CONNECTION_MAGIC );
    }

	if ( request->size != sizeof(*request) )
	{
		wireGLSimpleError( "wireGLAcceptClient: client claims "
						   "request is %u bytes, I expect %u bytes\n",
						   request->size, sizeof(*request) );
	}

    if ( __wiregl_max_send == 0 )
    {
        __wiregl_max_send = request->max_send;
        wireGLWarning( WIREGL_WARN_NOTICE, "pipeserver: setting max_send=%d",
                       __wiregl_max_send );
    }
    else
    {
        if ( request->max_send > __wiregl_max_send )
        {
            wireGLSimpleError( "wireGLAcceptClient: client specified "
                               "max_send=%d, but server already "
                               "using %d\n", request->max_send,
                               __wiregl_max_send );
        }
    }

    if ( request->please_use_gm )
    {
#ifdef GM_SUPPORT
        conn->type        = WIREGL_GM;
		conn->gm_node_id  = request->gm_node_id;
		conn->gm_port_num = request->gm_port_num;
#else
        wireGLSimpleError( "client has requested GM, but we don't "
						   "support it." );
#endif
    }

    conn->sender_id = wiregl_net.num_clients++;

	wiregl_net.server.connect( conn, request );

	wireGLFree( request );

    response.magic       = WIREGL_CONNECTION_MAGIC;
	response.size        = sizeof(response);
    response.max_send    = __wiregl_max_send;
    response.client_id   = conn->sender_id;
    response.gm_node_id  = 0; /* GM_NO_SUCH_NODE_ID */
	response.gm_port_num = 0;
#ifdef GM_SUPPORT
    if ( conn->type == WIREGL_GM )
    {
		wireGLGmInit( wiregl_net.recv, wiregl_net.close );
		response.gm_node_id  = wireGLGmNodeId( );
		response.gm_port_num = wireGLGmPortNum( );
    }
#endif

	wireGLTcpipWriteExact( conn->tcp_socket, &response, sizeof(response) );

    switch( conn->type )
    {
#ifdef GM_SUPPORT
	  case WIREGL_GM:
		wireGLTcpipConnection( conn );
		wireGLGmConnection( conn );
		wiregl_net.use_gm++;
		break;
#endif
	  case WIREGL_TCPIP:
		wireGLTcpipConnection( conn );
		break;

	  default:
		wireGLSimpleError( "wireGLFinishAcceptClient: "
						   "unknown conn->type=%u", conn->type );
		break;
    }
}

UTIL_DECL void
wireGLBecomeServer( unsigned short port, WireGLNetConnectFunc connectFunc )
{
	int i;

	wireGLAssert( wiregl_net.initialized );
	wireGLAssert( connectFunc != NULL );
	wireGLTcpipInit( wiregl_net.recv, wiregl_net.close );

	wiregl_net.server.sock = socket( AF_INET, SOCK_STREAM, 0 );
    if ( wiregl_net.server.sock == -1 )
    {
		int err = wireGLTcpipErrno( );
		wireGLSimpleError( "Couldn't create socket: %s", 
						   wireGLTcpipErrorString( err ) );
    }

	wireGLFrobSocket( wiregl_net.server.sock );

	for ( i = 0; i < WIREGL_BIND_TRIES; i++ )
	{
		int err;
		struct sockaddr_in addr;

		addr.sin_family = AF_INET;
		addr.sin_addr.s_addr = INADDR_ANY;
		addr.sin_port = htons( port );
		if ( !bind( wiregl_net.server.sock, (struct sockaddr *) &addr,
					sizeof(addr) ) )
			break;

		err = wireGLTcpipErrno( );
		if ( err == EADDRINUSE )
		{
			wireGLWarning( WIREGL_WARN_DEBUG, "Couldn't bind to socket "
						   "(port=%d): %s", port,
						   wireGLTcpipErrorString( err ) );
			port += 100;
		}
		else
		{
			wireGLSimpleError( "Couldn't bind to socket (port=%d): %s",
							   port, wireGLTcpipErrorString( err ) );
		}
	}

	if ( i == WIREGL_BIND_TRIES )
	{
		wireGLSimpleError( "Couldn't find a port to listen on." );
	}

	if ( i > 0 )
	{
		wireGLWarning( WIREGL_WARN_DEBUG, "Bound to alternate port %u",
					   port );
	}

    if ( listen( wiregl_net.server.sock, 100 /* max pending connections */ ) )
	{
		int err = wireGLTcpipErrno( );
        wireGLSimpleError( "Couldn't listen on socket: %s",
						   wireGLTcpipErrorString( err ) );
	}

	wiregl_net.server.connect = connectFunc;

	wireGLTcpipBecomeServer( wiregl_net.server.sock, wireGLNetAcceptClient );
}

UTIL_DECL void
wireGLNetInit( WireGLNetReceiveFunc recvFunc, WireGLNetCloseFunc closeFunc )
{
	if ( wiregl_net.initialized )
		return;

	wiregl_net.recv        = recvFunc;
	wiregl_net.close       = closeFunc;
	wiregl_net.use_gm      = 0;
	wiregl_net.num_clients = 0;

	wiregl_net.server.sock    = 0;
	wiregl_net.server.connect = NULL;

	wiregl_net.initialized = 1;
}

UTIL_DECL void *
wireGLNetAlloc( WireGLConnection *conn )
{
	wireGLAssert( conn );
	return conn->Alloc( );
}

UTIL_DECL void
wireGLNetSend( WireGLConnection *conn, void **bufp, 
			   void *start, unsigned int len )
{
	wireGLAssert( conn );
	wireGLAssert( len > 0 );
	if ( bufp ) {
		wireGLAssert( start >= *bufp );
		wireGLAssert( (unsigned char *) start + len <= 
					  (unsigned char *) *bufp + __wiregl_max_send );
	}

#ifndef NDEBUG
	if ( conn->send_credits > WIREGL_INITIAL_RECV_CREDITS )
	{
		wireGLError( "wireGLNetSend: send_credits=%u, looks like there is a "
					 "leak (max=%u)", conn->send_credits,
					 WIREGL_INITIAL_RECV_CREDITS );
	}
#endif

	conn->total_bytes += len;

	conn->Send( conn, bufp, start, len );
}

UTIL_DECL void
wireGLNetFree( WireGLConnection *conn, void *buf )
{
	wireGLAssert( conn );
	conn->Free( conn, buf );
}

UTIL_DECL int
wireGLNetRecv( void )
{
	int found_work = 0;

	found_work += wireGLTcpipRecv( );
#ifdef GM_SUPPORT
	if ( wiregl_net.use_gm )
		found_work += wireGLGmRecv( );
#endif
#if 0
	/* HACK - NEVER EVER SLEEP! */
	/* if we didn't pull anything out of the network layer then give
       up the rest of our timeslice */
	if ( !found_work )
		wireGLSleep( 0 );
#endif

	return found_work;
}

UTIL_DECL unsigned int
wireGLNetMaxSend( void )
{
	return __wiregl_max_send;
}
