/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "__utilsource.h"

#include <stdio.h>
#include <stdarg.h>

#define IGNORE_INSTRUMENT_FLAG
#include "wiregl/include/wiregl_util.h"

static WireGLHashTable *counter_hash = NULL, *timer_hash = NULL, *frame_hash = NULL, *server_hash = NULL;
static FILE *output_file;

typedef struct {
	float value;
	char *name;
} WireGLInstrumentationCounter;

typedef struct {
	WireGLTimer *timer;
	char *name;
} WireGLInstrumentationTimer;

typedef struct __node {
	float value;
	struct __node *next;
} WireGLInstrumentationArrayNode;

typedef struct {
	char *name;
	float *value;
	int num;
} WireGLInstrumentationServerCounter;

typedef struct {
	char *name;
	WireGLInstrumentationArrayNode *head, *tail;
} WireGLInstrumentationArray;

static WireGLInstrumentationCounter *__getCounter( int id )
{
	WireGLInstrumentationCounter *counter = 
		(WireGLInstrumentationCounter *) wireGLHashtableSearch( counter_hash, id );
	if (!counter)
	{
		wireGLSimpleError( "Searched for a bogus counter id: %d", id );
	}
	return counter;
}

static WireGLInstrumentationTimer *__getTimer( int id )
{
	WireGLInstrumentationTimer *timer = 
		(WireGLInstrumentationTimer *) wireGLHashtableSearch( timer_hash, id );
	if (!timer)
	{
		wireGLSimpleError( "Searched for a bogus timer id: %d", id );
	}
	return timer;
}

static WireGLInstrumentationServerCounter *__getServerCounter( int id )
{
	WireGLInstrumentationServerCounter *count = 
		(WireGLInstrumentationServerCounter *) wireGLHashtableSearch( server_hash, id );
	if (!count)
	{
		wireGLSimpleError( "Searched for a bogus server counter id: %d", id );
	}
	return count;
}

static void __bumpArray( WireGLInstrumentationArray *array )
{
	WireGLInstrumentationArrayNode *node = 
		( WireGLInstrumentationArrayNode * ) wireGLAlloc( sizeof( *node ) );
	node->value = 0;
	node->next = NULL;
	if (array->tail)
	{
		array->tail->next = node;
	}
	else
	{
		array->head = node;
	}
	array->tail = node;
}

static WireGLInstrumentationArray *__getArray( int id )
{
	WireGLInstrumentationArray *array = 
		(WireGLInstrumentationArray *) wireGLHashtableSearch( frame_hash, id );
	if (!array)
	{
		wireGLSimpleError( "Searched for a bogus array id: %d", id );
	}
	return array;
}

UTIL_DECL void wireGLInstrumentShutDown( void )
{
	fclose( output_file );
}

UTIL_DECL void wireGLInstrumentToScreen( void )
{
	output_file = stdout;
}

UTIL_DECL void wireGLInstrumentToFile( char *filename )
{
	output_file = fopen( filename, "w" );
	if (!output_file)
	{
		wireGLSimpleError( "Can't open %s for instrumentation logging!", filename );
	}
}

UTIL_DECL void wireGLInitInstrumentation( WireGLInstrumentationType type, ...)
{
	char *filename;
	va_list args;
	static int initialized = 0;
	if (initialized)
		return;
	initialized = 1;
	counter_hash = wireGLAllocHashtable();
	timer_hash = wireGLAllocHashtable();
	frame_hash = wireGLAllocHashtable();
	server_hash = wireGLAllocHashtable();
	va_start( args, type );
	switch( type )
	{
		case TO_SCREEN:
			wireGLInstrumentToScreen();
			break;
		case TO_FILE:
			filename = va_arg( args, char * );
			wireGLInstrumentToFile( filename );
			break;
	}
	va_end( args );
}

UTIL_DECL void wireGLInstrumentRegisterCounter( int id, char *name )
{
	WireGLInstrumentationCounter *counter = 
		(WireGLInstrumentationCounter *) wireGLAlloc( sizeof( *counter ) );

	counter->value = 0;
	counter->name = strdup( name );
	wireGLHashtableAdd( counter_hash, id, counter );
}

UTIL_DECL void wireGLInstrumentRegisterTimer( int id, char *name )
{
	WireGLInstrumentationTimer *counter = 
		(WireGLInstrumentationTimer *) wireGLAlloc( sizeof( *counter ) );

	counter->timer = wireGLTimerNewTimer();
	counter->name = strdup( name );
	wireGLHashtableAdd( timer_hash, id, counter );
}

UTIL_DECL void wireGLInstrumentRegisterPerFrameCounter( int id, char *name )
{
	WireGLInstrumentationArray *array = 
		(WireGLInstrumentationArray *) wireGLAlloc( sizeof( *array ) );

	array->name = strdup( name );
	array->head = array->tail = NULL;

	__bumpArray( array );
	wireGLHashtableAdd( frame_hash, id, array );
}

UTIL_DECL void wireGLInstrumentRegisterPerServerCounter( int id, int num, char *name )
{
	WireGLInstrumentationServerCounter *count = 
		(WireGLInstrumentationServerCounter *) wireGLAlloc( sizeof( *count ) );

	count->name = strdup( name );
	count->num = num;
	count->value = (float *) wireGLAlloc( num * sizeof( *(count->value) ) );
	wireGLHashtableAdd( server_hash, id, count );
}

UTIL_DECL void wireGLInstrumentIncrPerFrameCounter( int id, float value )
{
	WireGLInstrumentationArray *array = __getArray( id );
	array->tail->value += value;
}

UTIL_DECL void wireGLInstrumentIncrPerServerCounter( int id, int num, float value )
{
	WireGLInstrumentationServerCounter *count = __getServerCounter( id );
	if( num < 0 || num >= count->num)
		wireGLSimpleError( "Array out of bounds in wireGLInstrumentIncrPerServerCounter: %d", num );
	count->value[num] += value;
}

UTIL_DECL void wireGLInstrumentIncrCounter( int id, float amount )
{
	WireGLInstrumentationCounter *counter = __getCounter( id );
	counter->value += amount;
}

UTIL_DECL void wireGLInstrumentSetCounter( int id, float amount )
{
	WireGLInstrumentationCounter *counter = __getCounter( id );
	counter->value = amount;
}

UTIL_DECL void wireGLInstrumentResetTimer( int id )
{
	WireGLInstrumentationTimer *timer = __getTimer( id );
	wireGLResetTimer( timer->timer );
}

UTIL_DECL void wireGLInstrumentStartTimer( int id )
{
	WireGLInstrumentationTimer *timer = __getTimer( id );
	wireGLStartTimer( timer->timer );
}

UTIL_DECL void wireGLInstrumentStopTimer( int id )
{
	WireGLInstrumentationTimer *timer = __getTimer( id );
	wireGLStopTimer( timer->timer );
}

UTIL_DECL void wireGLInstrumentPrintCounter( int id )
{
	WireGLInstrumentationCounter *counter = __getCounter( id );
	fprintf( output_file, "%s: %f\n", counter->name, counter->value );
}

UTIL_DECL void wireGLInstrumentPrintTimer( int id )
{
	WireGLInstrumentationTimer *timer = __getTimer( id );
	fprintf( output_file, "%s: %f\n", timer->name, wireGLTimerTime( timer->timer ) );
}

UTIL_DECL void wireGLInstrumentPrintPerFrameCounter( int id )
{
	WireGLInstrumentationArrayNode *node;
	WireGLInstrumentationArray *array = __getArray( id );
	fprintf( output_file, "%s: { ", array->name );

	for (node = array->head ; node ; node = node->next )
	{
		fprintf( output_file, "%f ", node->value );
	}
	fprintf( output_file, "}\n");
}

UTIL_DECL void wireGLInstrumentPrintPerServerCounter( int id )
{
	int i;
	WireGLInstrumentationServerCounter *count = __getServerCounter( id );
	fprintf( output_file, "%s: { ", count->name );

	for ( i = 0; i < count->num; i++ )
	{
		fprintf( output_file, "%f ", count->value[i] );
	}
	fprintf( output_file, "}\n");
}

UTIL_DECL float wireGLInstrumentGetCounter( int id )
{
	WireGLInstrumentationCounter *counter = __getCounter( id );
	return counter->value;
}

UTIL_DECL double wireGLInstrumentGetTime( int id )
{
	WireGLInstrumentationTimer *timer = __getTimer( id );
	return wireGLTimerTime( timer->timer );
}

UTIL_DECL void wireGLInstrumentPrintAll( void )
{
	WIREGL_HASHTABLE_WALK( counter_hash, counter_temp )
	{
		WireGLInstrumentationCounter *counter = (WireGLInstrumentationCounter *) counter_temp->data;
		fprintf( output_file, "%s: %f\n", counter->name, counter->value );
	} WIREGL_HASHTABLE_WALK_END( counter_hash );
	WIREGL_HASHTABLE_WALK( timer_hash, timer_temp )
	{
		WireGLInstrumentationTimer *timer = (WireGLInstrumentationTimer *) timer_temp->data;
		fprintf( output_file, "%s: %f\n", timer->name, wireGLTimerTime( timer->timer ));
	} WIREGL_HASHTABLE_WALK_END( timer_hash );
	WIREGL_HASHTABLE_WALK( frame_hash, array_temp )
	{
		wireGLInstrumentPrintPerFrameCounter( array_temp->key );
	} WIREGL_HASHTABLE_WALK_END( frame_hash );
}

UTIL_DECL void wireGLInstrumentPrintf( const char *format, ... )
{
	va_list args;
	va_start( args, format );
	vfprintf( output_file, format, args );
	va_end( args );
}

UTIL_DECL void __wireGLInstrumentNOP (const char *format, ... )
{
	va_list args;
	va_start( args, format );
	va_end( args );
	WIREGL_UNUSED(format);
	WIREGL_UNUSED(args);
}

UTIL_DECL void wireGLInstrumentNextFrame( void )
{
	WIREGL_HASHTABLE_WALK( frame_hash, array_temp )
	{
		WireGLInstrumentationArray *array = (WireGLInstrumentationArray *) (array_temp->data);
		__bumpArray( array );
	} WIREGL_HASHTABLE_WALK_END( frame_hash );
}
