/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "__utilsource.h"

#include <stdio.h>
#include <stdlib.h>
#include "wiregl/include/wiregl_util.h"

UTIL_DECL WireGLHashTable *wireGLAllocHashtable( void )
{
	int i;
	WireGLHashTable *hash = (WireGLHashTable *) wireGLAlloc( sizeof( WireGLHashTable )) ;
	hash->num_elements = 0;
	for (i = 0 ; i < WIREGL_NUM_BUCKETS ; i++)
    {
		hash->buckets[i] = NULL;;
    }
	return hash;
}

UTIL_DECL void wireGLFreeHashtable( WireGLHashTable *hash )
{
	int i;

	for ( i = 0; i < WIREGL_NUM_BUCKETS; i++ )
	{
		if ( hash->buckets[i] ) {
			wireGLFree( hash->buckets[i]->data );
			wireGLFree( hash->buckets[i] );
		}
	}
	wireGLFree( hash );
}
			

static unsigned int wireGLHash( unsigned int key )
{
	return key % WIREGL_NUM_BUCKETS;
}

static unsigned int wireGLStringHash( char *key )
{
	unsigned int val = 0;
	char *temp;

	for ( temp = key; *temp; temp++ )
	{
		val += *temp;
		val %= WIREGL_NUM_BUCKETS;
	}
	return val;
}

UTIL_DECL void wireGLHashtableAdd( WireGLHashTable *h, unsigned int key, void *data )
{
	WireGLHashNode *node = (WireGLHashNode *) wireGLAlloc( sizeof( WireGLHashNode ) );
	node->is_string = 0;
	node->key = key;
	node->data = data;
	node->next = h->buckets[wireGLHash( key )];
	h->buckets[ wireGLHash( key ) ] = node;
	h->num_elements++;
}

UTIL_DECL void wireGLHashtableStringAdd( WireGLHashTable *h, char *key, void *data )
{
	WireGLHashNode *node = (WireGLHashNode *) wireGLAlloc( sizeof( WireGLHashNode ) );
	node->is_string = 1;
	node->str_key = key;
	node->data = data;
	node->next = h->buckets[wireGLStringHash( key )];
	h->buckets[wireGLStringHash( key )] = node;
	h->num_elements++;
}

UTIL_DECL void wireGLHashtableDelete( WireGLHashTable *h, unsigned int key )
{
	unsigned int index = wireGLHash( key );
	WireGLHashNode *temp, *beftemp = NULL;
	for ( temp = h->buckets[index]; temp; temp = temp->next )
    {
		if ( !temp->is_string && temp->key == key )
			break;
		beftemp = temp;
    }
	if ( !temp )
		wireGLError( "Couldn't find the hashtable element to delete" );
	if ( beftemp )
		beftemp->next = temp->next;
	else
		h->buckets[index] = temp->next;
	h->num_elements--;
	wireGLFree( temp->data );
	wireGLFree( temp );
}

UTIL_DECL void wireGLHashtableStringDelete( WireGLHashTable *h, char *key )
{
	unsigned int index = wireGLStringHash( key );
	WireGLHashNode *temp, *beftemp = NULL;
	for ( temp = h->buckets[index]; temp; temp = temp->next )
    {
		if ( temp->is_string && !strcmp( temp->str_key,key ) )
			break;
		beftemp = temp;
    }
	if ( !temp )
		wireGLError( "Couldn't find the hashtable element to delete" );
	if ( beftemp )
		beftemp->next = temp->next;
	else
		h->buckets[index] = temp->next;
	h->num_elements--;
	wireGLFree( temp->data );
	wireGLFree( temp );
}

UTIL_DECL void *wireGLHashtableSearch( WireGLHashTable *h, unsigned int key )
{
	unsigned int index = wireGLHash( key );
	WireGLHashNode *temp;
	for ( temp = h->buckets[index]; temp; temp = temp->next )
    {
		if ( !temp->is_string && temp->key == key )
			break;
    }
	if ( !temp )
    {
		return NULL;
    }
	return temp->data;
}

UTIL_DECL void *wireGLHashtableStringSearch( WireGLHashTable *h, char * key )
{
	unsigned int index = wireGLStringHash( key );
	WireGLHashNode *temp;
	for ( temp = h->buckets[index]; temp; temp = temp->next )
    {
		if ( temp->is_string && !strcmp( temp->str_key,key ) )
			break;
    }
	if ( !temp )
    {
		return NULL;
    }
	return temp->data;
}

UTIL_DECL void wireGLHashtableReplace( WireGLHashTable *h, unsigned int key, void *data, int free_mem )
{
	unsigned int index = wireGLHash( key );
	WireGLHashNode *temp;
	for ( temp = h->buckets[index]; temp; temp = temp->next )
    {
		if ( !temp->is_string && temp->key == key )
			break;
    }
	if ( !temp )
    {
		wireGLHashtableAdd( h, key, data );
		return;
    }
	if ( free_mem )
    {
		free( temp->data );
    }
	temp->data = data;
}

UTIL_DECL void wireGLHashtableStringReplace( WireGLHashTable *h, char * key, void *data, int free_mem )
{
	unsigned int index = wireGLStringHash( key );
	WireGLHashNode *temp;
	for ( temp = h->buckets[index]; temp; temp = temp->next )
    {
		if ( temp->is_string && !strcmp( temp->str_key,key ) )
			break;
    }
	if ( !temp )
    {
		wireGLHashtableStringAdd( h, key, data );
		return;
    }
	if ( free_mem )
    {
		free( temp->data );
    }
	temp->data = data;
}

UTIL_DECL unsigned int wireGLHashtableNumElements( WireGLHashTable *h) 
{
	return h->num_elements;
}
