/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "__utilsource.h"
#if defined(WINDOWS)
#define WIN32_LEAN_AND_MEAN
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#if defined(WINDOWS)
#include <assert.h>
#include <windows.h>
#include <process.h>
#else
#include <unistd.h>
#endif
#include <ctype.h>
#include "wiregl/include/wiregl_util.h"

static char               warn_hostname[256];
static int                warn_pid   = 0;
static WireGLWarningLevel warn_level = WIREGL_WARN_DEBUG;
static int                use_gui    = 0;


UTIL_DECL void 
__wireGLAssert( const char *assertion, const char *file, int line )
{
	wireGLError( "Assertion failed: %s, file %s, line %d",
				 assertion, file, line );
}

UTIL_DECL void 
__wireGLAbort( const char *file, int line )
{
    wireGLError( "Abort: file %s, line %d\n", file, line );
}

static void
GetWarningId( void )
{
	char *temp;
	if ( gethostname( warn_hostname, sizeof(warn_hostname) ) )
	{
		strcpy( warn_hostname, "foobar" );
	}
	temp = warn_hostname;
	while ( *temp && *temp != '.' )
		temp++;
	*temp = '\0';
	warn_pid   = (int) getpid( );
}

static void 
wireGLIntError( int core, int use_gui, const char *format, va_list args ) 
{
	static char txt[8092];
	int offset;
	if ( !warn_hostname[0] )
		GetWarningId( );
	offset = sprintf( txt, "WireGL Error (%s:%d): ", warn_hostname, warn_pid );
	vsprintf( txt + offset, format, args );
#ifdef WINDOWS
	if ( use_gui )
		MessageBox( NULL, txt, "WireGL Error", MB_OK );
	else
		fprintf( stderr, "%s\n", txt );
#else
	WIREGL_UNUSED(use_gui);
	fprintf( stderr, "%s\n", txt );
#endif

#if 0
	while ( core )
	{
		warn_level = WIREGL_WARN_VERBOSE_DEBUG;
		wireGLWarning( WIREGL_WARN_CRITICAL, "I'm waiting to be debugged" );
		wireGLSleep( 10 * 1000 /* milliseconds */ );
	}
#endif

	if ( core )
		abort( );

	exit( 1 );
}

UTIL_DECL void 
wireGLError( const char *format, ... )
{
	va_list args;
	va_start( args, format );
	wireGLIntError( 1, use_gui, format, args );
	va_end( args );
}

UTIL_DECL void 
wireGLSimpleError( const char *format, ... )
{
	va_list args;
	va_start( args, format );
	wireGLIntError( 0, use_gui, format, args );
	va_end( args );
}

UTIL_DECL void 
wireGLPrintError( const char *format, ... )
{
	va_list args;
	va_start( args, format );
	wireGLIntError( 0, 0, format, args );
	va_end( args );
}

#ifndef WIREGL_NO_WARN

UTIL_DECL void 
wireGLWarning( WireGLWarningLevel level, const char *format, ... )
{
	static char txt[8192];

	if (level <= warn_level)
	{
		int offset;
		va_list args;
		va_start( args, format );
		if ( !warn_hostname[0] )
			GetWarningId( );
		offset = sprintf( txt, "WireGL (%s:%d): ", warn_hostname, warn_pid );
		offset += vsprintf( txt+offset, format, args );
		strcpy( txt+offset, "\n" );
		fputs( txt, stderr );
		va_end( args );
	}
}

#endif

static int 
StringMatch( const char *a, const char *b )
{
	while ( *a && *b ) 
	{
		if ( tolower(*a) != tolower(*b) ) return 0;
		a++;
		b++;
	}

	return ( *a == 0 && *b == 0 );
}

UTIL_DECL void 
wireGLInitWarnings( const char *warn_str, int gui )
{
	warn_level = WIREGL_WARN_CRITICAL;
	use_gui = gui;
	if ( warn_str )
    {
		if ( StringMatch( warn_str , "VERBOSE_DEBUG" ) )
			warn_level = WIREGL_WARN_VERBOSE_DEBUG;
		else if ( StringMatch( warn_str, "DEBUG" ) )
			warn_level = WIREGL_WARN_DEBUG;
		else if ( StringMatch( warn_str, "CHATTY" ) )
			warn_level = WIREGL_WARN_CHATTY;
		else if ( StringMatch( warn_str, "NOTICE" ) )
			warn_level = WIREGL_WARN_NOTICE;
		else if ( StringMatch( warn_str, "IMPORTANT" ) )
			warn_level = WIREGL_WARN_IMPORTANT;
		else if ( StringMatch( warn_str, "CRITICAL" ) )
			warn_level = WIREGL_WARN_CRITICAL;
		else
		{
			warn_level = WIREGL_WARN_VERBOSE_DEBUG;
			wireGLWarning( WIREGL_WARN_NOTICE, "wireGLInitWarnings( "
						   "warn_str=\"%s\", gui=%d ) : bad warn_str",
						   warn_str, gui );
		}
    }
}
