/* volread.c */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <math.h>
#include <memory.h>
#include <assert.h>

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#else
#include <unistd.h>
#endif

#define USE_ZLIB

#ifdef USE_ZLIB
#include <zlib.h>
#endif

#include "volume.h"

#define DEN_VERSION       0x0001
#define DEN_VERSION_SWAB  0x0100

typedef struct {
    short version;

    short origMin[3];
    short origMax[3];
    short origLen[3];

    short extrMin[3];
    short extrMax[3];
    short extrLen[3];

    short mapMin[3];
    short mapMax[3];
    short mapLen[3];

    short mapWarps;
} DenHeader;

static u16
SwapShort( u16 x )
{
    return ( x << 8 ) | ( x >> 8 );
}

static u32
SwapWord( u32 x )
{
    return (( x >> 24 ) | ( x << 24 ) | (( x >> 8 ) & 0xff00 )
            | (( x << 8 ) & 0xff0000 ));
}

static void
SwapHeader( DenHeader *header )
{
    u32  i;
    u16 *temp;

    temp = (u16 *) header;
    for ( i = 0; i  != sizeof(DenHeader) / 2; i++ )
        temp[i] = SwapShort( temp[i] );
}

int
Volume_MakeSphere( u32 dim, Volume *volume )
{
    float dim_2  = dim * 0.5f;
    float radius = dim_2 - 1.0f;
    float dx, dy, dz;
    u8 temp;
    u32 n, i, j, k;

    assert( dim > 2 && dim < 256 );

    n = dim * dim * dim;

    volume->width   = dim;
    volume->height  = dim;
    volume->depth   = dim;
    volume->density = (u8 *) xmalloc( n );
    assert( volume->density );

    for ( k = 0; k != dim; k++ ) {
        for ( j = 0; j != dim; j++ ) {
            for ( i = 0; i != dim; i++ ) {
                dx = i - dim_2;
                dy = j - dim_2;
                dz = k - dim_2;
                temp = ( SquareRoot( dx * dx + dy * dy + dz * dz ) < radius ) ? 255 : 0;
                volume->density[ ( k * dim + j ) * dim + i ] = temp;
            }
        }
    }

    return 1;
}

int
Volume_MakeCube( u32 dim, Volume *volume )
{
    u32 n, i, j, k;

    assert( dim > 4 && dim < 256 );

    n = dim * dim * dim;

    volume->width   = dim;
    volume->height  = dim;
    volume->depth   = dim;
    volume->density = (u8 *) xmalloc( n );
    assert( volume->density );

    memset( volume->density, 0, n );

    for ( k = 2; k != dim - 2; k++ )
        for ( j = 2; j != dim - 2; j++ )
            for ( i = 2; i != dim - 2; i++ )
                volume->density[ ( k * dim + j ) * dim + i ] = 255;

    return 1;
}

int
Volume_MakePointField( u32 dim, Volume *volume )
{
    u32 n, i, j, k;
    u32 skip = 2;

    assert( dim > 2 && dim < 256 );

    n = dim * dim * dim;

    volume->width   = dim;
    volume->height  = dim;
    volume->depth   = dim;
    volume->density = (u8 *) xmalloc( n );
    assert( volume->density );

    memset( volume->density, 0, n );

    for ( k = 2; k < dim; k += skip )
        for ( j = 2; j < dim; j += skip )
            for ( i = 2; i < dim; i += skip )
                volume->density[ ( k * dim + j ) * dim + i ] = 255;

    return 1;
}

static struct {
	FILE   *f;
#ifdef USE_ZLIB
	gzFile  gz;
#endif
} xfile;

static int
xopen( const char *name )
{
	FILE   *f;
#ifdef USE_ZLIB
	gzFile  gz;
	static const char *myVersion = ZLIB_VERSION;
#endif
	unsigned char magic[4];

	f        = NULL;
	xfile.f  = NULL;
#ifdef USE_ZLIB
	gz       = NULL;
	xfile.gz = NULL;
#endif

    f = fopen( name, "rb" );
    if ( f == NULL ) {
        perror( name );
        return -1;
    }

    if ( fread( magic, 1, 4, f ) != 4 ) {
        perror( name );
        fclose( f );
        return -1;
    }

    if ( magic[0] == 0x1f && magic[1] == 0x8b &&
         magic[2] == 0x08 && magic[3] == 0x08 ) {

        /* looks like a gzip file */
        fclose( f );

#ifdef USE_ZLIB
		if ( zlibVersion()[0] != myVersion[0] ) {
			fprintf( stderr, "incompatible zlib version (expected \"%s\", "
					 "got \"%s\")\n", myVersion, zlibVersion( ) );
			return -1;
		}
    
		if ( strcmp( zlibVersion(), myVersion ) ) {
			fprintf(stderr, "warning: different zlib version (expected "
					"\"%s\", got \"%s\")\n", myVersion, zlibVersion( ) );
		}
		
		gz = gzopen( name, "r" );
		if ( gz == NULL ) {
			fprintf( stderr, "gzopen( \"%s\" ) error\n", name );
			return -1;
		}

		xfile.gz = gz;
#else
		fprintf( stderr, "Compressed file \"%s\" not supported\n", name );
		return -1;
#endif
    } else {
        rewind( f );
		xfile.f = f;
    }

	return 0;
}

static int
xread( int n, void *buf )
{
	int bytes_read;

	if ( xfile.f ) {
		bytes_read = fread( buf, 1, n, xfile.f );
		if ( bytes_read != n ) {
			fprintf( stderr, "expected to read %d bytes, got %d: %s!\n",
					 n, bytes_read, strerror( errno ) );
			return -1;
		}
		return 0;
	}
	
#ifdef USE_ZLIB
	if ( xfile.gz ) {
		bytes_read = gzread( xfile.gz, buf, n );
		if ( bytes_read != n ) {
			fprintf( stderr, "expected to read %d bytes, got %d!\n",
					 n, bytes_read );
			return -1;
		}
		return 0;
	}
#endif

	fprintf( stderr, "xread: no open file?\n" );
	return -1;
}

static void
xclose( void )
{
	if ( xfile.f ) {
		fclose( xfile.f );
		xfile.f = NULL;
	}
#ifdef USE_ZLIB
	if ( xfile.gz ) {
		gzclose( xfile.gz );
		xfile.gz = NULL;
	}
#endif
}

int
Volume_ReadRaw( const char *name, u32 w, u32 h, u32 d, Volume *volume )
{
    u32   n;

    n = w * h *d;
    volume->density = (u8 *) xmalloc( n );
    assert( volume->density );

    if ( xopen( name ) )
        return 0;

    if ( xread( n, volume->density ) ) {
        xclose( );
        return 0;
    }

    xclose( );

    volume->width  = w;
    volume->height = h;
    volume->depth  = d;
    volume->i_chunks = 0;
    volume->j_chunks = 0;
    volume->k_chunks = 0;
    volume->chunk    = NULL;

    return 1;
}

int
Volume_ReadDen( const char *name, Volume *volume )
{
    DenHeader  hdr;
    u32        len, n, w, h, d;

    if ( xopen( name ) )
        return 0;

    if ( xread( sizeof(hdr), &hdr ) ) {
        fprintf( stderr, "couldn't read header\n" );
        xclose( );
        return 0;
    }

    if ( xread( sizeof(len), &len ) ) {
        fprintf( stderr, "couldn't read header\n" );
        xclose( );
        return 0;
    }

    if ( hdr.version == DEN_VERSION_SWAB ) {
        SwapHeader( &hdr );
        len = SwapWord( len );
    }

    if ( hdr.version != DEN_VERSION ) {
        fprintf( stderr, "\n\"%s\" doesn't look like a den file\n", name );
		xclose( );
        return 0;
    }

    if ( (u32)hdr.mapLen[0] * (u32)hdr.mapLen[1] * (u32)hdr.mapLen[2] != len ) {
        fprintf( stderr, "\"%s\" doesn't look like a den file\n", name );
        xclose( );
        return 0;
    }

    w = hdr.mapLen[0];
    h = hdr.mapLen[1];
    d = hdr.mapLen[2];

    n = w * h * d;
    volume->density = (u8 *) xmalloc( n );
    assert( volume->density );

    if ( xread( n, volume->density ) ) {
        xclose( );
        return 0;
    }

	xclose( );

    volume->width    = w;
    volume->height   = h;
    volume->depth    = d;
    volume->i_chunks = 0;
    volume->j_chunks = 0;
    volume->k_chunks = 0;
    volume->chunk    = NULL;

    return 1;
}
