/* track.c
 */

#include <math.h>

#include "track.h"

#ifndef M_SQRT1_2
#define M_SQRT1_2       0.70710678118654752440
#endif

#ifndef M_SQRT2
#define M_SQRT2         1.41421356237309504880
#endif

#ifdef WINDOWS
#define SquareRoot(x)   sqrt(x)
#else
#define SquareRoot(x)   sqrtf(x)
#endif

V3
vzero( void )
{
    V3 v;

    v.i = v.j = v.k = 0.0;

    return v;
}

V3
vset( float i, float j, float k )
{
    V3 v;

    v.i = i;
    v.j = j;
    v.k = k;

    return v;
}

V3
vsub( V3 a, V3 b )
{
    V3 c;

    c.i = a.i - b.i;
    c.j = a.j - b.j;
    c.k = a.k - b.k;

    return c;
}

V3
vcross( V3 a, V3 b )
{
    V3 c;

    c.i = ( a.j * b.k ) - ( a.k * b.j );
    c.j = ( a.k * b.i ) - ( a.i * b.k );
    c.k = ( a.i * b.j ) - ( a.j * b.i );

    return c;
}

float
vdot( V3 a, V3 b )
{
    return a.i * b.i + a.j * b.j + a.k * b.k;
}

float
vlength( V3 v )
{
    return (float) SquareRoot( vdot( v, v ) );
}

V3
vscale( V3 v, float m )
{
    v.i *= m;
    v.j *= m;
    v.k *= m;

    return v;
}

V3
vnormal( V3 v )
{
    return vscale( v, 1.0f / vlength(v) );
}

V3
vadd( V3 a, V3 b )
{
    V3 c;

    c.i = a.i + b.i;
    c.j = a.j + b.j;
    c.k = a.k + b.k;

    return c;
}

V3
rotVector( float m[4][4], V3 a )
{
    V3 b;

    b.i = m[0][0] * a.i + m[0][1] * a.j + m[0][2] * a.k;
    b.j = m[1][0] * a.i + m[1][1] * a.j + m[1][2] * a.k;
    b.k = m[2][0] * a.i + m[2][1] * a.j + m[2][2] * a.k;

    return b;
}

/*
 *  Given an axis and angle, compute quaternion.
 */
Quat
axisToQuat( V3 axis, float phi )
{
    Quat quat;
    float s;

    axis = vnormal( axis );

    s = (float) sin( phi / 2.0f );

    quat.q[0] = axis.i * s;
    quat.q[1] = axis.j * s;
    quat.q[2] = axis.k * s;
    quat.q[3] = (float) cos( phi / 2.0f );

    return quat;
}

/*
 * Project an x,y pair onto a sphere of radius r OR a hyperbolic sheet
 * if we are away from the center of the sphere.
 */
static float
tb_project_to_sphere(float r, float x, float y)
{
    float d, t, z;

    d = (float) SquareRoot( x*x + y*y );
    if ( d < r * M_SQRT1_2 ) {    /* Inside sphere */
        z = (float) SquareRoot( r*r - d*d );
    } else {           /* On hyperbola */
        t = r / (float) M_SQRT2;
        z = t*t / d;
    }
    return z;
}

Quat
trackball( float p1x, float p1y, float p2x, float p2y, float radius )
{
    float phi;  /* how much to rotate about axis */
    V3 p1, p2, axis, d;
    float t;

    if ( p1x == p2x && p1y == p2y ) {
        Quat quat;
        quat.q[0] = quat.q[1] = quat.q[2] = 0.0;
        quat.q[3] = 1.0;
        return quat;
    }

    /*
     * First, figure out z-coordinates for projection of P1 and P2 to
     * deformed sphere
     */
    p1 = vset( p1x, p1y, tb_project_to_sphere( radius, p1x, p1y ) );
    p2 = vset( p2x, p2y, tb_project_to_sphere( radius, p2x, p2y ) );

    /*
     *  Now, we want the cross product of P1 and P2
     */
    axis = vcross( p2, p1 );

    /*
     *  Figure out how much to rotate around that axis.
     */
    d = vsub( p1, p2 );
    t = vlength( d ) / ( 2.0f * radius );

    /*
     * Avoid problems with out-of-control values...
     */
    if ( t > 1.0 ) t = 1.0;
    if ( t < -1.0 ) t = -1.0;
    phi = 2.0f * (float) asin( t );

    return axisToQuat( axis, phi );
}

Quat
addQuats( Quat q1, Quat q2 )
{
    Quat  t1, t2, t3, tf;
    float mag;

    t1.v = vscale( q1.v, q2.q[3] );
    t2.v = vscale( q2.v, q1.q[3] );
    t3.v = vcross( q2.v, q1.v );

    tf.v = vadd( t1.v, t2.v );
    tf.v = vadd( t3.v, tf.v );

    tf.q[3] = q1.q[3] * q2.q[3] - vdot( q1.v, q2.v );

    mag = (float) SquareRoot( tf.q[0] * tf.q[0] + tf.q[1] * tf.q[1] +
							  tf.q[2] * tf.q[2] + tf.q[3] * tf.q[3] );

    tf.q[0] /= mag;
    tf.q[1] /= mag;
    tf.q[2] /= mag;
    tf.q[3] /= mag;

    return tf;
}

void
quatToMatrix( Quat quat, float m[4][4] )
{
    m[0][0] = 1.0f - 2.0f * (quat.q[1] * quat.q[1] + quat.q[2] * quat.q[2]);
    m[0][1] = 2.0f * (quat.q[0] * quat.q[1] - quat.q[2] * quat.q[3]);
    m[0][2] = 2.0f * (quat.q[2] * quat.q[0] + quat.q[1] * quat.q[3]);
    m[0][3] = 0.0f;

    m[1][0] = 2.0f * (quat.q[0] * quat.q[1] + quat.q[2] * quat.q[3]);
    m[1][1] = 1.0f - 2.0f * (quat.q[2] * quat.q[2] + quat.q[0] * quat.q[0]);
    m[1][2] = 2.0f * (quat.q[1] * quat.q[2] - quat.q[0] * quat.q[3]);
    m[1][3] = 0.0f;

    m[2][0] = 2.0f * (quat.q[2] * quat.q[0] - quat.q[1] * quat.q[3]);
    m[2][1] = 2.0f * (quat.q[1] * quat.q[2] + quat.q[0] * quat.q[3]);
    m[2][2] = 1.0f - 2.0f * (quat.q[1] * quat.q[1] + quat.q[0] * quat.q[0]);
    m[2][3] = 0.0f;

    m[3][0] = 0.0f;
    m[3][1] = 0.0f;
    m[3][2] = 0.0f;
    m[3][3] = 1.0f;
}
