/* gui.c */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#else
#include <unistd.h>
#endif

#include "misc.h"
#include "globals.h"
#include "track.h"
#include "draw.h"
#include "gui.h"

typedef struct {
    float x, y;
    int   button;
    int   state;
} Mouse;

static Mouse m = { 0, 0, GLUT_LEFT_BUTTON, GLUT_UP };

void
resize( int w, int h )
{
    globals.win.w = w;
    globals.win.h = h;
}

void
idleRotate( void )
{
    Quat quat = trackball( 0.5, 0.5, 0.1, 0.5, 1.0 );
    globals.camera.quat = addQuats( quat, globals.camera.quat );
    glutPostRedisplay( );
}

void
keyboard( unsigned char c, int x, int y )
{
    x = x;
    y = y;

    switch ( c ) {

      case 27:
      case 'q':
      case 'Q':
		printf( "exiting\n" );
		globals.please_exit = 1;
		glutPostRedisplay( );
        break;

	  case 'w':
		globals.wireframe = !globals.wireframe;
		printf( "wireframe: %s\n", ( globals.wireframe ? "on" : "off" ) );
		glutPostRedisplay( );
		break;

      case 'x':
        globals.idle_rotate = !globals.idle_rotate;
        if ( globals.idle_rotate )
            glutIdleFunc( idleRotate );
        else
            glutIdleFunc( NULL );
        break;

      case 'r':
        globals.rainbow = !globals.rainbow;
        printf( "rainbow: %s\n", ( globals.rainbow ? "on" : "off" ) );
        glutPostRedisplay( );
        break;

      case 'p':
        printf( "-camera %f %f %f %f -radius %f "
                "-opacity %f -thresh %d\n",
                globals.camera.quat.q[0], globals.camera.quat.q[1],
                globals.camera.quat.q[2], globals.camera.quat.q[3],
                globals.camera.radius,
                globals.opacity, globals.threshold );
        break;

      case 'c':
        globals.cull_enable = !globals.cull_enable;
        printf( "cull = %s\n", ( globals.cull_enable ? "yes" : "no" ) );
        glutPostRedisplay( );
        break;

      case 'h':
      default:
        printf( "q : quit\n"
                "p : print current view for command line\n"
                "r : toggle rainbow mode\n"
                "s : toggle chunk sequencing\n"
                "x : rotate\n"
                "P : toggle static/work-queue partition\n"
				"w : toggle wireframe\n"
                );
        break;
    }
}

void
special( int key, int x, int y )
{
    int mods;
    u8  step;

    x = x;
    y = y;

    mods = glutGetModifiers( );
    step = ( mods & GLUT_ACTIVE_SHIFT ) ? 5 : 1;

    switch ( key ) {

      case GLUT_KEY_LEFT:
        if ( globals.threshold > step )
            globals.threshold -= step;
        else
            globals.threshold = 1;
        glutPostRedisplay( );
        break;

      case GLUT_KEY_RIGHT:
        if ( globals.threshold < 254 - step )
            globals.threshold += step;
        else
            globals.threshold = 254;
        glutPostRedisplay( );
        break;

      case GLUT_KEY_PAGE_UP:
        globals.opacity -= 0.05f;
        if ( globals.opacity < 0.05 )
            globals.opacity = 0.05f;
        printf( "Opacity: %.2f\n", globals.opacity );
        glutPostRedisplay( );
        break;

      case GLUT_KEY_PAGE_DOWN:
        globals.opacity += 0.05f;
        if ( globals.opacity > 1.00 )
            globals.opacity = 1.00f;
        printf( "Opacity: %.2f\n", globals.opacity );
        glutPostRedisplay( );
        break;

      default:
        break;
    }
}

void
mouse( int button, int state, int x, int y )
{
    m.x = (float) ( 2 * x - globals.win.w ) / globals.win.w;
    m.y = (float) ( 2 * y - globals.win.h ) / globals.win.h;

    m.button = button;
    m.state  = state;
}

void
motion( int x, int y )
{
    float new_x, new_y;

    if ( m.state == GLUT_UP )
        return;

    new_x = (float) ( 2 * x - globals.win.w ) / globals.win.w;
    new_y = (float) ( 2 * y - globals.win.h ) / globals.win.h;

    if ( m.button == GLUT_LEFT_BUTTON ) {

        Quat quat;

        quat = trackball( m.x, m.y, new_x, new_y, 1.0 );
        globals.camera.quat = addQuats( quat, globals.camera.quat );

    } else if ( m.button == GLUT_MIDDLE_BUTTON ) {

        globals.camera.radius += globals.camera.radius * ( new_y - m.y );
        if ( globals.camera.radius < 0.1f ) globals.camera.radius = 0.1f;
    }

    m.x = new_x;
    m.y = new_y;

    glutPostRedisplay( );
}

void
GuiInit( int *argcp, char **argv )
{
    unsigned int mode;

    glutInitWindowSize( globals.win.w, globals.win.h );
    glutInitWindowPosition( 100, 100 );
    glutInit( argcp, argv );

    mode = GLUT_RGBA;
    if ( !globals.single_buffer )
        mode |= GLUT_DOUBLE;
    glutInitDisplayMode( mode );

    glutCreateWindow( "Marching Cubes" );
}

void
GuiGo( void (GLUTCALLBACK *display)(void) )
{
    glutDisplayFunc( display );
    glutReshapeFunc( resize );
    glutKeyboardFunc( keyboard );
    glutSpecialFunc( special );
    glutMouseFunc( mouse );
    glutMotionFunc( motion );

    glutMainLoop( );
}
