/* draw.c */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <assert.h>

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#include <GL/gl.h>
#include <GL/glu.h>
#include <wiregl/include/wiregl_papi.h>

#include "globals.h"
#include "draw.h"
#include "rainbow.h"

#include "edge.h"
#include "cases.h"

#include "voxel.h"

typedef struct tiny_order_t {
	u8 i, j, k, pad;
} tiny_order_t;

tiny_order_t Order2x2x2[8][8] = {
    /* order = 0 */
    {
      { 0, 0, 0, 0 },
      { 0, 0, 1, 0 },
      { 0, 1, 0, 0 },
      { 1, 0, 0, 0 },
      { 0, 1, 1, 0 },
      { 1, 0, 1, 0 },
      { 1, 1, 0, 0 },
      { 1, 1, 1, 0 },
    },
    /* order = 1 */
    {
      { 1, 0, 0, 0 },
      { 1, 0, 1, 0 },
      { 1, 1, 0, 0 },
      { 0, 0, 0, 0 },
      { 1, 1, 1, 0 },
      { 0, 0, 1, 0 },
      { 0, 1, 0, 0 },
      { 0, 1, 1, 0 },
    },
    /* order = 2 */
    {
      { 0, 1, 0, 0 },
      { 0, 1, 1, 0 },
      { 0, 0, 0, 0 },
      { 1, 1, 0, 0 },
      { 0, 0, 1, 0 },
      { 1, 1, 1, 0 },
      { 1, 0, 0, 0 },
      { 1, 0, 1, 0 },
    },
    /* order = 3 */
    {
      { 1, 1, 0, 0 },
      { 1, 1, 1, 0 },
      { 1, 0, 0, 0 },
      { 0, 1, 0, 0 },
      { 1, 0, 1, 0 },
      { 0, 1, 1, 0 },
      { 0, 0, 0, 0 },
      { 0, 0, 1, 0 },
    },
    /* order = 4 */
    {
      { 0, 0, 1, 0 },
      { 0, 0, 0, 0 },
      { 0, 1, 1, 0 },
      { 1, 0, 1, 0 },
      { 0, 1, 0, 0 },
      { 1, 0, 0, 0 },
      { 1, 1, 1, 0 },
      { 1, 1, 0, 0 },
    },
    /* order = 5 */
    {
      { 1, 0, 1, 0 },
      { 1, 0, 0, 0 },
      { 1, 1, 1, 0 },
      { 0, 0, 1, 0 },
      { 1, 1, 0, 0 },
      { 0, 0, 0, 0 },
      { 0, 1, 1, 0 },
      { 0, 1, 0, 0 },
    },
    /* order = 6 */
    {
      { 0, 1, 1, 0 },
      { 0, 1, 0, 0 },
      { 0, 0, 1, 0 },
      { 1, 1, 1, 0 },
      { 0, 0, 0, 0 },
      { 1, 1, 0, 0 },
      { 1, 0, 1, 0 },
      { 1, 0, 0, 0 },
    },
    /* order = 7 */
    {
      { 1, 1, 1, 0 },
      { 1, 1, 0, 0 },
      { 1, 0, 1, 0 },
      { 0, 1, 1, 0 },
      { 1, 0, 0, 0 },
      { 0, 1, 0, 0 },
      { 0, 0, 1, 0 },
      { 0, 0, 0, 0 },
    },
};

#define ACCEL_EMPTY(x,threshold) ( (x).min >  (threshold) || \
                                   (x).max <= (threshold) )
#define CHUNK_EMPTY(chunk,threshold) ( (chunk).min_density >  (threshold) || \
                                       (chunk).max_density <= (threshold) )


static int
DrawChunk( VolumeChunk *chunk, int order )
{
    int i, j, k, width, height, depth, area, n, a;
    VolumeMinMax *minmax;
    tiny_order_t *foo = Order2x2x2[order];
    VolumeTraversalOrder *list;
    int triangles = 0;

	if ( CHUNK_EMPTY( *chunk, globals.threshold ) )
		return 0;

	if ( globals.bbox_hint ) {

		GLfloat bounds[8];
			
		bounds[0] = (GLfloat) chunk->i0;
		bounds[1] = (GLfloat) chunk->j0;
		bounds[2] = (GLfloat) chunk->k0;
		bounds[3] = (GLfloat) 1.0f;

		bounds[4] = (GLfloat) ( chunk->i0 + chunk->w );
		bounds[5] = (GLfloat) ( chunk->j0 + chunk->h );
		bounds[6] = (GLfloat) ( chunk->k0 + chunk->d );
		bounds[7] = (GLfloat) 1.0f;

		glHint( WIREGL_OBJECT_BBOX_HINT, (GLenum) bounds );
	}

	glBegin( GL_TRIANGLES );

    width  = chunk->w >> 1;
    height = chunk->h >> 1;
    depth  = chunk->d >> 1;
    area   = width * height;

    n      = area * depth;
    minmax = chunk->minmax;
    list   = chunk->accelOrder[order];
    while ( n-- ) {

        if ( !ACCEL_EMPTY(minmax[list->offset],globals.threshold) ) {

            i = list->i << 1;
            j = list->j << 1;
            k = list->k << 1;

            for ( a = 0; a < 8; a++ ) {
                triangles += DoVoxel( chunk,
                                      i + foo[a].i,
                                      j + foo[a].j,
                                      k + foo[a].k );
            }
        }

        list++;
    }

	glEnd( );

    return triangles;
}

static void
UpdateViewer( Viewer *v, V3 *pos, V3 *up )
{
    float m[4][4];

    quatToMatrix( v->quat, m );

    *pos = rotVector( m, v->look );
    *pos = vscale( *pos, v->radius );

    *up  = rotVector( m, v->up );
}

static int
DrawOrder( Quat q )
{
    float m[4][4];
    int   order;

    quatToMatrix( q, m );

    order = 0;
    if ( m[0][2] > 0 )
        order |= 1;
    if ( m[1][2] > 0 )
        order |= 2;
    if ( m[2][2] > 0 )
        order |= 4;

    return order;
}

static int
get_next_chunk( void )
{
	int value = globals.chunk_index;
	globals.chunk_index += globals.num_threads;
	return value;
}

static void
WaitForChunk( Volume *volume, int x, int y, int z )
{
    GLuint id;

    assert ( x >= 0 && x < volume->i_chunks &&
             y >= 0 && y < volume->j_chunks &&
             z >= 0 && z < volume->k_chunks );

    id = ( z * volume->j_chunks + y ) * volume->i_chunks + x;
    glSemaphoreP( id + 1 );
}

static void
ReleaseChunk( Volume *volume, int x, int y, int z )
{
    GLuint id;

    assert ( x >= 0 && x < volume->i_chunks &&
             y >= 0 && y < volume->j_chunks &&
             z >= 0 && z < volume->k_chunks );

    id = ( z * volume->j_chunks + y ) * volume->i_chunks + x;
    glSemaphoreV( id + 1 );
}

static void
WaitForNeighbors( Volume *volume, int order, int x, int y, int z )
{
    order = ~order;

    if ( order & 0x1 ) {
        if ( x > 0 )
            WaitForChunk( volume, x - 1, y, z );
    } else {
        if ( x + 1 < volume->i_chunks )
            WaitForChunk( volume, x + 1, y, z );
    }

    if ( order & 0x2 ) {
        if ( y > 0 )
            WaitForChunk( volume, x, y - 1, z );
    } else {
        if ( y + 1 < volume->j_chunks )
            WaitForChunk( volume, x, y + 1, z );
    }

    if ( order & 0x4 ) {
        if ( z > 0 )
            WaitForChunk( volume, x, y, z - 1 );
    } else {
        if ( z + 1 < volume->k_chunks )
            WaitForChunk( volume, x, y, z + 1 );
    }
}

static void
ReleaseNeighbors( Volume *volume, int order, int x, int y, int z )
{
    order = ~order;

    if ( order & 0x1 ) {
        if ( x + 1 < volume->i_chunks )
            ReleaseChunk( volume, x, y, z );
    } else {
        if ( x > 0 )
            ReleaseChunk( volume, x, y, z );
    }

    if ( order & 0x2 ) {
        if ( y + 1 < volume->j_chunks )
            ReleaseChunk( volume, x, y, z );
    } else {
        if ( y > 0 )
            ReleaseChunk( volume, x, y, z );
    }

    if ( order & 0x4 ) {
        if ( z + 1 < volume->k_chunks )
            ReleaseChunk( volume, x, y, z );
    } else {
        if ( z > 0 )
            ReleaseChunk( volume, x, y, z );
    }
}

static int
DrawFrame_Serial( Volume *volume, int order )
{
    int                   my_chunk;
	int                   triangles = 0;
    VolumeTraversalOrder *list;
    VolumeChunk          *chunk;

    list = volume->chunkOrder[order];

	my_chunk = 0;

    while ( my_chunk < volume->num_chunks ) {

		chunk = volume->chunk + list[my_chunk].offset;
		triangles += DrawChunk( chunk, order );

		my_chunk++;
	}

	return triangles;
}

static int
DrawFrame_NoOrder( Volume *volume, int order )
{
    int                   my_chunk;
	int                   triangles = 0;
    VolumeTraversalOrder *list;
    VolumeChunk          *chunk;

    list = volume->chunkOrder[order];
	my_chunk = get_next_chunk( );

    while ( my_chunk < volume->num_chunks ) {

		chunk = volume->chunk + list[my_chunk].offset;
		triangles += DrawChunk( chunk, order );

		my_chunk = get_next_chunk( );
	}

	return triangles;
}

static int
DrawFrame_Partial( Volume *volume, int order )
{
    int                   my_chunk;
	int                   triangles = 0;
    VolumeTraversalOrder *list;
    VolumeChunk          *chunk;

    list = volume->chunkOrder[order];

	my_chunk = get_next_chunk( );

    while ( my_chunk < volume->num_chunks ) {

		chunk = volume->chunk + list[my_chunk].offset;

        WaitForNeighbors( volume, order,
                          chunk->i_idx, chunk->j_idx, chunk->k_idx );

		triangles += DrawChunk( chunk, order );

        ReleaseNeighbors( volume, order,
                          chunk->i_idx, chunk->j_idx, chunk->k_idx );

		my_chunk = get_next_chunk( );
	}

	return triangles;
}

static int
DrawFrame_Total( Volume *volume, int order )
{
    int                   my_chunk;
	int                   triangles = 0;
    VolumeTraversalOrder *list;
    VolumeChunk          *chunk;

    list = volume->chunkOrder[order];

	my_chunk = get_next_chunk( );

    while ( my_chunk < volume->num_chunks ) {

        if ( my_chunk > 0 )
            glSemaphoreP( my_chunk );

		chunk = volume->chunk + list[my_chunk].offset;
		triangles += DrawChunk( chunk, order );

        if ( my_chunk + 1 < volume->num_chunks )
            glSemaphoreV( my_chunk + 1 );

		my_chunk = get_next_chunk( );
	}

	return triangles;
}

static int
DrawFrame_Barrier( Volume *volume, int order )
{
    int                   my_chunk, step;
	int                   triangles;
    VolumeTraversalOrder *list;
    VolumeChunk          *chunk;

    list = volume->chunkOrder[order];

	my_chunk = get_next_chunk( );

	step = 0;
	triangles = 0;

    while ( my_chunk < volume->num_chunks ) {

		while ( step < list[ my_chunk ].step ) {
			glBarrierExec( 1 );
			step++;
		}

		chunk = volume->chunk + list[my_chunk].offset;
		triangles += DrawChunk( chunk, order );

		my_chunk = get_next_chunk( );
	}

	while ( step < list[ volume->num_chunks - 1 ].step ) {
		glBarrierExec( 1 );
		step++;
	}

	return triangles;
}

void
ContextInit( void )
{
    GLfloat ambient[]  = { 0.3f, 0.3f, 0.3f, 1.0f };
    GLfloat diffuse[]  = { 0.8f, 0.7f, 0.8f, 1.0f };
    GLfloat specular[] = { 0.0f, 0.0f, 0.0f, 1.0f };

    glClearColor( 0, 0, 0, 0 );

    glDisable( GL_CULL_FACE );

    glLightModeli( GL_LIGHT_MODEL_TWO_SIDE, 1 );
    glShadeModel( GL_SMOOTH );

    glEnable( GL_LIGHT0 );
    glLightfv( GL_LIGHT0, GL_AMBIENT,  ambient  );
    glLightfv( GL_LIGHT0, GL_SPECULAR, specular );
    glLightfv( GL_LIGHT0, GL_DIFFUSE,  diffuse  );

    glEnable( GL_LIGHT1 );
    glLightfv( GL_LIGHT1, GL_AMBIENT,  ambient  );
    glLightfv( GL_LIGHT1, GL_SPECULAR, specular );
    glLightfv( GL_LIGHT1, GL_DIFFUSE,  diffuse  );

    glDisable( GL_LIGHT1 );
    glEnable( GL_LIGHTING );
    glEnable( GL_NORMALIZE );

#if 0
    {
        GLfloat whitish[]  = { 0.8f, 1.0f, 0.9f, 1.0f };
        GLfloat bluish[]   = { 0.3f, 0.4f, 1.0f, 1.0f };

        glMaterialfv( GL_FRONT, GL_AMBIENT_AND_DIFFUSE, whitish );
        glMaterialfv( GL_BACK,  GL_AMBIENT_AND_DIFFUSE, bluish  );
        glDisable( GL_COLOR_MATERIAL );
    }
#else
    glColorMaterial( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE );
    glEnable( GL_COLOR_MATERIAL );
#endif

	if ( globals.depth_buffer ) {
		glEnable( GL_DEPTH_TEST );
		glDepthMask( 1 );
	} else {
		glDisable( GL_DEPTH_TEST );
		glDepthMask( 0 );
	}

    glDrawBuffer( globals.single_buffer ? GL_FRONT : GL_BACK );
}

void
FrameInit( void )
{
    float   lightDir0[4] = {  0.5f, 1.0f, 1.0f, 0.0f };
    float   lightDir1[4] = { -0.5f, 1.0f, 1.0f, 0.0f };
    float   dim;
    V3      pos, up;
    Volume *v = &globals.volume;
	GLfloat color[4];

#if 0
	/* don't upset the poor pipeserver */
	glViewport( 0, 0, globals.win.w, globals.win.h );
#endif

    glMatrixMode( GL_PROJECTION );
    glLoadIdentity( );

    dim = globals.camera.radius;
    glOrtho( -dim, dim, -dim, dim, -400, 400 );

    glMatrixMode( GL_MODELVIEW );

    glLoadIdentity( );

    glLightfv( GL_LIGHT0, GL_POSITION, lightDir0 );
    glLightfv( GL_LIGHT1, GL_POSITION, lightDir1 );

    UpdateViewer( &globals.camera, &pos, &up );
    gluLookAt( pos.i, pos.j, pos.k,
               0, 0, 0,
               up.i, up.j, up.k );

    glTranslatef( -1.0f, -1.0f, -1.0f );
    glScalef( 2.0f / (float) ( v->width  - 1 ),
              2.0f / (float) ( v->height - 1 ),
              2.0f / (float) ( v->depth  - 1 ) );

    glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
    if ( globals.opacity < 0.999 )
        glEnable( GL_BLEND );
    else
        glDisable( GL_BLEND );

    if ( globals.cull_enable )
        glEnable( GL_CULL_FACE );
    else
        glDisable( GL_CULL_FACE );

    if ( globals.wireframe )
        glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );
    else
        glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

	color[0] = 0.8f;
	color[1] = 0.8f;
	color[2] = 1.0f;
	color[3] = globals.opacity;
    if ( globals.rainbow ) {
        int idx = globals.thread_id & 0x3f;
        color[0] = rainbow[idx].r;
        color[1] = rainbow[idx].g;
        color[2] = rainbow[idx].b;
	}

	glColor4fv( color );
}

int
DrawFrame( void )
{
    int order, triangles, i;
	int (*func)( Volume *, int );

	if ( globals.num_threads == 1 )
	{
		func = DrawFrame_Serial;
	}
	else
	{
		switch ( globals.sequence_chunks ) {

		  case OrderNone:    func = DrawFrame_NoOrder; break;
		  case OrderPartial: func = DrawFrame_Partial; break;
		  case OrderTotal:   func = DrawFrame_Total;   break;
		  case OrderBarrier: func = DrawFrame_Barrier; break;
		  default:
			abort( );
		}
	}

	order = DrawOrder( globals.camera.quat );

    FrameInit( );

	triangles = 0;
	for ( i = 0; i < globals.passes; i++ )
	{
		globals.chunk_index = globals.thread_id;
		triangles += func( &globals.volume, order );
	}

	if ( globals.bbox_hint ) {
		glHint( WIREGL_DEFAULT_BBOX_HINT, 1 );
	}

	return triangles;
}
