/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <io.h> /* for read() */
#endif
#include <GL/gl.h>
#include <errno.h>

#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_pipe.h"
#include "wiregl/include/wiregl_util.h"
#include "wiregl/glcontext/glcontext.h"

int calcTexSize(int size) {
  int texSize = 32;
  while (texSize < size)
    texSize *= 2;
  return texSize;
}

int safename(char *filename) {
  char *p;
  int atdir;

  if(!wiregl_pipe.distrib_textures) {
	static int complained = 0;
	if(!complained)
		wireGLWarning( WIREGL_WARN_IMPORTANT, "Distributed textures not enabled -- no -dtex server option" );
	complained++;
	return 0;
  }
	
  if (filename[0] == '/' || filename[0] == '\\'
		|| strchr(filename, ':') != NULL) {
	return 0;	/* No absolute paths */
  }
  atdir = 1;
  for(p = filename; *p; p++) {
    if(*p == '.' && p[1] == '.' &&
		(p[2] == '/' || p[2] == '\\')) {
	if(atdir) return 0;	/* No ..'s */
    }
    atdir = (*p == '/' || *p == '\\');
  }
  return 1;
}

static int plaintno = 0;

static int complainnow(void) {
  /* Emit one complaint every 2**n errors */
    int complain = ( (plaintno & (plaintno-1)) == 0 );
    plaintno++;
    return complain;
}

void writePPM(char *filename, GLvoid *data, int w, int h) {
  FILE *fp;
  static int plaintno = 0;

  if(!safename(filename)) {
    /* emit one complaint every 2**n errors */
    if(complainnow())
	wireGLWarning( WIREGL_WARN_IMPORTANT,
	    "Not safe to write distributed texture to %.80s", filename);
    return;
  }
  fp = fopen(filename, "w");
  if (!fp) {				 /* No permission? Missing directory? */
    if(complainnow())
	wireGLWarning( WIREGL_WARN_NOTICE,
	    "Can't create distributed texture file %.80s: %s",
		filename, strerror(errno));
    return;
  }

  fprintf(fp, "P6\n%d %d\n255\n", w, h);
  fwrite(data, w*h*3, 1, fp);

  if(fclose(fp) != 0) {			 /* Disk full? */
    if(complainnow())
	wireGLWarning( WIREGL_WARN_NOTICE,
	    "Error writing distributed texture %.80s: %s",
		filename, strerror(errno));
    return;
  }

  plaintno = 0;		/* success -- reset complaint counter */
}

GLvoid *readPPM(char *filename) {
  GLvoid *data;
  char input[80];
  int w, h, want, got;
  FILE *fp;
  int fpos;

  fp = fopen(filename, "r");
  if (!fp) {
    if(complainnow())
	wireGLWarning( WIREGL_WARN_NOTICE,
		"Can't open distributed texture file %.80s: %s",
		filename, strerror(errno) );
    return NULL;
  }

  fscanf(fp, "%s %d %d %s%c", input, &w, &h, input, input);

  data = (GLubyte *) malloc(w*calcTexSize(h)*3);

  /* fread(data, w*h*3, 1, fp); */
  /* Try doing read() instead for speed */
  fpos = ftell(fp);
  lseek(fileno(fp), fpos, 0);
  want = w*h*3;
  got = read(fileno(fp), data, want);
  fclose(fp);
  if(got != want) {
    if(complainnow())
	wireGLWarning( WIREGL_WARN_NOTICE,
		"Got only %d of %d bytes expected from %.80s",
		got, want, filename);
  } else {
    plaintno = 0;
  }
  return data;
}

/* copy chunk into nearest texture size */
GLubyte *padTex(GLubyte *img, int w, int h) {
  GLubyte *tex;
  int r, texW, texH;

  texW = calcTexSize(w);
  texH = calcTexSize(h);
  tex = (GLubyte *) malloc(texW*texH*3);
  for (r=0; r<h; r++)
    memcpy((void *) &tex[r*texW*3], (void *) &img[r*w*3], w*3);

  return tex;
}

void __decodeTexImage2D( void )
{
	GLenum target = READ_DATA( sizeof( int ) + 0, GLenum );
	GLint level = READ_DATA( sizeof( int ) + 4, GLint );
	GLint internalformat = READ_DATA( sizeof( int ) + 8, GLint );
	GLsizei width = READ_DATA( sizeof( int ) + 12, GLsizei );
	GLsizei height = READ_DATA( sizeof( int ) + 16, GLsizei );
	GLint border = READ_DATA( sizeof( int ) + 20, GLint );
	GLenum format = READ_DATA( sizeof( int ) + 24, GLenum );
	GLenum type = READ_DATA( sizeof( int ) + 28, GLenum );
	int is_null = READ_DATA( sizeof( int ) + 32, int );
	GLvoid *pixels;
	int allocated = 0;
	int distributed = (type == GL_FALSE || type == GL_TRUE);

	if ( is_null )
		pixels = NULL;
	else 
		pixels = DATA_POINTER( sizeof( int ) + 36, GLvoid );

	if (distributed) {
	    int texSizeX = calcTexSize(width);
	    int texSizeY = calcTexSize(height);
	    if (type == GL_FALSE) {
		pixels = readPPM((char *) pixels);
		allocated = 1;
	    } else {
		char *filename = (char *) pixels;
		pixels = filename + strlen(filename)+1;
		writePPM(filename, pixels, width, height);
	    }
	    if (width != texSizeX) {
		GLubyte *tex = padTex(pixels, width, height);
		if (allocated) free(pixels);
		pixels = tex;
		allocated = 1;
	    }
	    width = texSizeX;
	    height = texSizeY;
	    type = GL_UNSIGNED_BYTE;
	}

	__gltrack_TexImage2D( target, level, internalformat, width, height,
						  border, format, type, pixels );
	INCR_VAR_PTR();
	if (allocated) free(pixels);
}

void __decodeTexImage1D( void )
{
	GLenum target = READ_DATA( sizeof( int ) + 0, GLenum );
	GLint level = READ_DATA( sizeof( int ) + 4, GLint );
	GLint internalformat = READ_DATA( sizeof( int ) + 8, GLint );
	GLsizei width = READ_DATA( sizeof( int ) + 12, GLsizei );
	GLint border = READ_DATA( sizeof( int ) + 16, GLint );
	GLenum format = READ_DATA( sizeof( int ) + 20, GLenum );
	GLenum type = READ_DATA( sizeof( int ) + 24, GLenum );
	int is_null = READ_DATA( sizeof( int ) + 28, int );
	GLvoid *pixels;

	if ( is_null )
		pixels = NULL;
	else 
		pixels = DATA_POINTER( sizeof( int ) + 32, GLvoid );

	__gltrack_TexImage1D( target, level, internalformat, width, border,
						  format, type, pixels );
	INCR_VAR_PTR();
}

void __decodeDeleteTextures( void )
{
	GLsizei n = READ_DATA( sizeof( int ) + 0, GLsizei );
	GLuint *textures = DATA_POINTER( sizeof( int ) + 4, GLuint );

	__gltrack_DeleteTextures( n, textures );
	INCR_VAR_PTR();
}


void __decodePrioritizeTextures( void )
{
	GLsizei n = READ_DATA( sizeof( int ) + 0, GLsizei );
	GLuint *textures = DATA_POINTER( sizeof( int ) + 4, GLuint );
	GLclampf *priorities = DATA_POINTER( sizeof( int ) + 4 + n*sizeof( GLuint ), GLclampf );

	__gltrack_PrioritizeTextures( n, textures, priorities );
	INCR_VAR_PTR();
}

void __decodeBindTexture( void )
{
	GLenum target = READ_DATA(0, GLenum);
	GLuint textureid = READ_DATA(4, GLuint);
	
	__gltrack_BindTexture (target, textureid);

	INCR_DATA_PTR( 8 );
}

void __decodeTexParameterfv( void )
{
	GLenum target = READ_DATA( sizeof( int ) + 0, GLenum );
	GLenum pname = READ_DATA( sizeof( int ) + 4, GLenum );
	GLfloat *params = DATA_POINTER( sizeof( int ) + 8, GLfloat );

	__gltrack_TexParameterfv( target, pname, params );
	INCR_VAR_PTR();
}

void __decodeTexParameteriv( void )
{
	GLenum target = READ_DATA( sizeof( int ) + 0, GLenum );
	GLenum pname = READ_DATA( sizeof( int ) + 4, GLenum );
	GLint *params = DATA_POINTER( sizeof( int ) + 8, GLint );

	__gltrack_TexParameteriv( target, pname, params );
	INCR_VAR_PTR();
}

void __decodeTexParameterf( void )
{
	GLenum target = READ_DATA( 0, GLenum );
	GLenum pname = READ_DATA( 4, GLenum );
	GLfloat param = READ_DATA( 8, GLfloat );

	__gltrack_TexParameterf( target, pname, param );
	INCR_DATA_PTR( 12 );
}
void __decodeTexParameteri( void )
{
	GLenum target = READ_DATA( 0, GLenum );
	GLenum pname = READ_DATA( 4, GLenum );
	GLint param = READ_DATA( 8, GLint );
	__gltrack_TexParameteri( target, pname, param );
	INCR_DATA_PTR( 12 );
}

void __decodeTexSubImage2D( void )
{
	GLenum target = READ_DATA( sizeof( int ) + 0, GLenum );
	GLint level = READ_DATA( sizeof( int ) + 4, GLint );
	GLint xoffset = READ_DATA( sizeof( int ) + 8, GLint );
	GLint yoffset = READ_DATA( sizeof( int ) + 12, GLint );
	GLsizei width = READ_DATA( sizeof( int ) + 16, GLsizei );
	GLsizei height = READ_DATA( sizeof( int ) + 20, GLsizei );
	GLenum format = READ_DATA( sizeof( int ) + 24, GLenum );
	GLenum type = READ_DATA( sizeof( int ) + 28, GLenum );
	GLvoid *pixels = DATA_POINTER( sizeof( int ) + 32, GLvoid );
	
	glPixelStorei( GL_UNPACK_ROW_LENGTH, 0 );
	glPixelStorei( GL_UNPACK_SKIP_PIXELS, 0 );
	glPixelStorei( GL_UNPACK_SKIP_ROWS, 0 );
	glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );

	__gltrack_TexSubImage2D( target, level, xoffset, yoffset, width, height,
							 format, type, pixels );
	INCR_VAR_PTR();
}

void __decodeTexSubImage1D( void )
{
	GLenum target = READ_DATA( sizeof( int ) + 0, GLenum );
	GLint level = READ_DATA( sizeof( int ) + 4, GLint );
	GLint xoffset =	READ_DATA( sizeof( int ) + 8, GLint );
	GLsizei width = READ_DATA( sizeof( int ) + 12, GLsizei );
	GLenum format = READ_DATA( sizeof( int ) + 16, GLenum );
	GLenum type = READ_DATA( sizeof( int ) + 20, GLenum );
	GLvoid *pixels = DATA_POINTER( sizeof( int ) + 24, GLvoid );

	glPixelStorei( GL_UNPACK_ROW_LENGTH, 0 );
	glPixelStorei( GL_UNPACK_SKIP_PIXELS, 0 );
	glPixelStorei( GL_UNPACK_SKIP_ROWS, 0 );
	glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
	
	__gltrack_TexSubImage1D( target, level, xoffset, width, 
							 format, type, pixels );
	INCR_VAR_PTR();
}


void __decodeTexEnvfv( void )
{
	GLenum target = READ_DATA( sizeof( int ) + 0, GLenum );
	GLenum pname = READ_DATA( sizeof( int ) + 4, GLenum );
	GLfloat *params = DATA_POINTER( sizeof( int ) + 8, GLfloat );

	__gltrack_TexEnvfv( target, pname, params );
	INCR_VAR_PTR();
}

void __decodeTexEnviv( void )
{
	GLenum target = READ_DATA( sizeof( int ) + 0, GLenum );
	GLenum pname = READ_DATA( sizeof( int ) + 4, GLenum );
	GLint *params = DATA_POINTER( sizeof( int ) + 8, GLint );

	__gltrack_TexEnviv( target, pname, params );
	INCR_VAR_PTR();
}

#define DATA_POINTER_DOUBLE( offset )

void __decodeTexGendv( void )
{
	GLenum coord = READ_DATA( sizeof( int ) + 0, GLenum );
	GLenum pname = READ_DATA( sizeof( int ) + 4, GLenum );
#ifdef WIREGL_UNALIGNED_ACCESS_OKAY
	GLdouble *params = DATA_POINTER( sizeof(int) + 8, GLdouble );
#else
	GLdouble params[4];
	int n_param = READ_DATA( 0, int ) - ( sizeof(int) + 8 );

	if ( n_param > sizeof(params) )
		wireGLError( "__decodeTexGendv: n_param=%d, expected <= %d\n", n_param, sizeof(params) );
	memcpy( params, DATA_POINTER( sizeof( int ) + 8, GLdouble ), n_param );
#endif

	__gltrack_TexGendv( coord, pname, params );
	INCR_VAR_PTR();
}

void __decodeTexGenfv( void )
{
	GLenum coord = READ_DATA( sizeof( int ) + 0, GLenum );
	GLenum pname = READ_DATA( sizeof( int ) + 4, GLenum );
	GLfloat *params = DATA_POINTER( sizeof( int ) + 8, GLfloat );

	__gltrack_TexGenfv( coord, pname, params );
	INCR_VAR_PTR();
}

void __decodeTexGeniv( void )
{
	GLenum coord = READ_DATA( sizeof( int ) + 0, GLenum );
	GLenum pname = READ_DATA( sizeof( int ) + 4, GLenum );
	GLint *params = DATA_POINTER( sizeof( int ) + 8, GLint );

	__gltrack_TexGeniv( coord, pname, params );
	INCR_VAR_PTR();
}
