/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif

#include <GL/gl.h>
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_pipe.h"
#include "wiregl/include/wiregl_lightning2.h"

#if defined (IRIX) || defined( IRIX64 ) || defined( Linux)
#include <X11/Xlib.h>

typedef XID GLXDrawable;
extern void glXSwapBuffers (Display *dpy, GLXDrawable drawable);
#endif

#define WIREGL_VIS_SHOW_PIPES 0

#if WIREGL_VIS_SHOW_PIPES

static struct {
    float r, g, b;
} rainbow[] = {
  { 1.00, 1.00, 1.00 },
  { 1.00, 1.00, 0.50 },
  { 1.00, 0.50, 1.00 },
  { 1.00, 0.50, 0.50 },
  { 0.50, 1.00, 1.00 },
  { 0.50, 1.00, 0.50 },
  { 0.50, 0.50, 1.00 },
  { 0.50, 0.50, 0.50 },
  { 1.00, 1.00, 0.75 },
  { 1.00, 0.50, 0.75 },
  { 1.00, 0.75, 1.00 },
  { 1.00, 0.75, 0.50 },
  { 1.00, 0.75, 0.75 },
  { 0.50, 1.00, 0.75 },
  { 0.50, 0.50, 0.75 },
  { 0.50, 0.75, 1.00 },
  { 0.50, 0.75, 0.50 },
  { 0.50, 0.75, 0.75 },
  { 0.75, 1.00, 1.00 },
  { 0.75, 1.00, 0.50 },
  { 0.75, 1.00, 0.75 },
  { 0.75, 0.50, 1.00 },
  { 0.75, 0.50, 0.50 },
  { 0.75, 0.50, 0.75 },
  { 0.75, 0.75, 1.00 },
  { 0.75, 0.75, 0.50 },
  { 0.75, 0.75, 0.75 },
  { 1.00, 1.00, 0.62 },
  { 1.00, 0.50, 0.62 },
  { 1.00, 0.75, 0.62 },
  { 1.00, 0.62, 1.00 },
  { 1.00, 0.62, 0.50 },
  { 1.00, 0.62, 0.75 },
  { 1.00, 0.62, 0.62 },
  { 0.50, 1.00, 0.62 },
  { 0.50, 0.50, 0.62 },
  { 0.50, 0.75, 0.62 },
  { 0.50, 0.62, 1.00 },
  { 0.50, 0.62, 0.50 },
  { 0.50, 0.62, 0.75 },
  { 0.50, 0.62, 0.62 },
  { 0.75, 1.00, 0.62 },
  { 0.75, 0.50, 0.62 },
  { 0.75, 0.75, 0.62 },
  { 0.75, 0.62, 1.00 },
  { 0.75, 0.62, 0.50 },
  { 0.75, 0.62, 0.75 },
  { 0.75, 0.62, 0.62 },
  { 0.62, 1.00, 1.00 },
  { 0.62, 1.00, 0.50 },
  { 0.62, 1.00, 0.75 },
  { 0.62, 1.00, 0.62 },
  { 0.62, 0.50, 1.00 },
  { 0.62, 0.50, 0.50 },
  { 0.62, 0.50, 0.75 },
  { 0.62, 0.50, 0.62 },
  { 0.62, 0.75, 1.00 },
  { 0.62, 0.75, 0.50 },
  { 0.62, 0.75, 0.75 },
  { 0.62, 0.75, 0.62 },
  { 0.62, 0.62, 1.00 },
  { 0.62, 0.62, 0.50 },
  { 0.62, 0.62, 0.75 },
  { 0.62, 0.62, 0.62 }
};

static void
VisShowPipes( void )
{
	int idx;
	float r, g, b, a;
	glPushAttrib( GL_ENABLE_BIT       | 
				  GL_CURRENT_BIT      | 
				  GL_TRANSFORM_BIT    |
				  GL_DEPTH_BUFFER_BIT |
				  GL_VIEWPORT_BIT );

	glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
	glEnable( GL_BLEND );

	glDisable( GL_DEPTH_TEST );
	glDepthMask( 0 );

	glDisable( GL_SCISSOR_TEST );
	glDisable( GL_TEXTURE_1D );
	glDisable( GL_TEXTURE_2D );
	glDisable( GL_LIGHTING );
	glDisable( GL_CULL_FACE );

	glViewport( 0, 0, wiregl_pipe.actual_window_width,
				wiregl_pipe.actual_window_height );

	glMatrixMode( GL_MODELVIEW );
	glPushMatrix( );
	glLoadIdentity( );
	
	glMatrixMode( GL_PROJECTION );
	glPushMatrix( );
	glLoadIdentity( );
	glOrtho( 0.0, 1.0, 0.0, 1.0, -1.0, 1.0 );

	idx = wiregl_pipe.pipe_num % ( sizeof(rainbow) / sizeof(rainbow[0]) );
	r = rainbow[idx].r;
	g = rainbow[idx].g;
	b = rainbow[idx].b;
	a = 0.5f;

#if 0
	wireGLWarning( WIREGL_WARN_CRITICAL, "pipe=%d idx=%d r=%.3f g=%.3f "
				   "b=%.3f a=%.3f", wiregl_pipe.pipe_num, idx, r, g, b, a );
#endif

	glColor4f( r, g, b, a );
	glRectf( 0.0f, 0.0f, 1.0f, 1.0f );

	glPopMatrix( );

	glMatrixMode( GL_MODELVIEW );
	glPopMatrix( );

	glPopAttrib( );
}

#endif

static void 
SwapL2( void ) 
{
#if WIREGL_VIS_SHOW_PIPES

	/* MWE- Does NVIDIA drop the ball here?  If I don't do a Finish
	 * here we sometimes get some blank tiles -- my guess is some
	 * piece of state ordering isn't correctly handled. */
	glFinish( );

	VisShowPipes( );
#endif

	if ( wiregl_pipe.L2.single_buffer )
	{
		wireGLL2DrawFrame( &wiregl_pipe.L2.frame[0] );
	}
	else
	{
		int idx = wiregl_pipe.L2.frame_num & 0x1;
		wireGLL2DrawFrame( &wiregl_pipe.L2.frame[idx] );

		/* wait for readback */
		wireGLL2Wait( );

		wiregl_pipe.L2.frame_num = !wiregl_pipe.L2.frame_num;
	}
}

static unsigned char *vis_buffer_pack;
static unsigned int   vis_buffer_len;
static unsigned char *vis_data_current;
static unsigned char *vis_opcode_current;
static unsigned char *vis_data_start;
static unsigned char *vis_opcode_start;
static unsigned char *vis_data_end;

#define GET_BUFFERED_POINTER( len )      \
  data_ptr = vis_data_current;           \
  if ( data_ptr + (len) > vis_data_end ) \
  {                                      \
	  GrowBuffer( len );                 \
      data_ptr = vis_data_current;       \
  }                                      \
  vis_data_current += (len)

#define WRITE_DATA( offset, type, data ) \
  *( (type *) (data_ptr + (offset))) = (data)

#define WRITE_OPCODE( opcode ) \
  *(vis_opcode_current--) = (unsigned char) opcode

#define GET_BUFFERED_POINTER_NO_ARGS( ) \
  GET_BUFFERED_POINTER( 4 );  \
  WRITE_DATA( 0, GLuint, 0xdeadbeef )

static void
SetupBuffer( void )
{
	unsigned int num_opcodes;

	if ( vis_buffer_len == 0 )
	{
		vis_buffer_len  = 8192;
		vis_buffer_pack = wireGLAlloc( vis_buffer_len );
	}

	/* from opengl_stub/buffer.c */
	num_opcodes = ( vis_buffer_len - sizeof(WireGLMessageOpcodes) ) / 5;
	num_opcodes = ( num_opcodes + 0x3 ) & ~0x3;

	vis_data_start     = (unsigned char *) vis_buffer_pack + 
		num_opcodes + sizeof(WireGLMessageOpcodes);

	vis_data_end       = vis_buffer_pack + vis_buffer_len;
	vis_opcode_start   = vis_data_start - 1;

	vis_data_current   = vis_data_start;
	vis_opcode_current = vis_opcode_start;
}

static void
GrowBuffer( unsigned int delta )
{
	unsigned char *old_buffer = vis_buffer_pack;
	unsigned int   num_opcodes = vis_opcode_start - vis_opcode_current;
	unsigned char *start = vis_opcode_current + 1;
	unsigned int   len   = vis_data_current - start;

	vis_buffer_len  += 2 * delta;
	vis_buffer_pack  = wireGLAlloc( vis_buffer_len );
	SetupBuffer( );

	vis_opcode_current -= num_opcodes;
	vis_data_current   += len - num_opcodes;
	memcpy( vis_opcode_current+1, start, len );

	wireGLFree( old_buffer );
}

static void
SendBuffer( WireGLConnection *conn )
{
	int len;
	int num_opcodes = vis_opcode_start - vis_opcode_current;
	WireGLMessageOpcodes *hdr = (WireGLMessageOpcodes *) 
		( vis_data_start - ( ( num_opcodes + 3 ) & ~0x3 ) - sizeof(*hdr) );

	hdr->type       = WIREGL_MESSAGE_OPCODES;
	hdr->senderId   = conn->sender_id;
	hdr->numOpcodes = num_opcodes;

	len = vis_data_current - (unsigned char *) hdr;

	wireGLNetSend( conn, NULL, hdr, len );
}

static void 
glRasterPos2f_pack( GLfloat x, GLfloat y )
{
	unsigned char * data_ptr;
	GET_BUFFERED_POINTER (sizeof (x) + sizeof (y));
	WRITE_DATA( 0, GLfloat, x);
	WRITE_DATA( 4, GLfloat, y);
	WRITE_OPCODE(WIREGL_RASTERPOS2F_OPCODE);
}

#define VIS_BARRIER 0x1212

static void *
glDrawPixels_pack( GLsizei width, GLsizei height, GLenum format, GLenum type )
{
	unsigned char *data_ptr;
	int packet_length;
	int datasize;

	packet_length = 
		sizeof( width ) + 
		sizeof( height ) +
		sizeof( format ) +
		sizeof( type );

	/* ONLY SUPPORT RGB unsigned byte! */

	datasize = width*height*3;
	packet_length += datasize + sizeof (int);

	GET_BUFFERED_POINTER(packet_length);

	WRITE_DATA( 0, int, packet_length);
	WRITE_DATA( sizeof (int) + 0, GLsizei, width );
	WRITE_DATA( sizeof (int) + 4, GLsizei, height );
	WRITE_DATA( sizeof (int) + 8, GLenum, format );
	WRITE_DATA( sizeof (int) + 12, GLenum, type );

	WRITE_OPCODE(WIREGL_DRAWPIXELS_OPCODE);

	return data_ptr + sizeof(int) + 16;
}

static void 
glBarrierCreate_pack( GLuint name, GLuint count )
{
	unsigned char *data_ptr;
	GET_BUFFERED_POINTER(8);
	WRITE_DATA( 0, GLuint, name);
	WRITE_DATA( 4, GLuint, count);
	WRITE_OPCODE(WIREGL_BARRIERCREATE_OPCODE);
}

static void
glBarrierExec_pack( GLuint name )
{
	unsigned char *data_ptr;
	GET_BUFFERED_POINTER(4);
	WRITE_DATA( 0, GLuint, name);
	WRITE_OPCODE(WIREGL_BARRIEREXEC_OPCODE);
}

static void
wireGLSwapBuffers_pack( void )
{
	unsigned char *data_ptr;
	GET_BUFFERED_POINTER_NO_ARGS( );
	WRITE_OPCODE( WIREGL_SWAPBUFFERS_OPCODE );
}

static void
wireGLWriteback_pack( void )
{
	unsigned char *data_ptr;
	GET_BUFFERED_POINTER_NO_ARGS( );
	WRITE_OPCODE( WIREGL_WRITEBACK_OPCODE );
}

static void 
SwapVis( void )
{
	static int barrier_created = 0;
	int i, d, img_w, img_h;
	WireGLWorkQueue *q = __currentQueue;
	unsigned char *pixels;

#if WIREGL_VIS_SHOW_PIPES

	/* MWE- Does NVIDIA drop the ball here?  If I don't do a Finish
	 * here we sometimes get some blank tiles -- my guess is some
	 * piece of state ordering isn't correctly handled. */
	glFinish( );

	VisShowPipes( );
#endif

	/* wait for everybody to be done with the previous frame */
	for ( d = 0; d < wiregl_pipe.num_displays; d++ )
	{
		WireGLConnection *conn = wiregl_pipe.display[d].conn;
		while ( conn->pending_writebacks > 0 )
		{
			wireGLNetRecv( );
		}
	}

	for ( d = 0; d < wiregl_pipe.num_displays; d++ )
	{
		int x = wiregl_pipe.display[d].x;
		int y = wiregl_pipe.display[d].y;
		int w = wiregl_pipe.display[d].w;
		int h = wiregl_pipe.display[d].h;
		SetupBuffer( );

		if ( !barrier_created )
		{
			glBarrierCreate_pack( VIS_BARRIER, wiregl_pipe.num_pipes );
		}
		
		for ( i = 0; i < q->num_extents; i++ ) 
		{
			WireGLWorkQueueExtent *extent = &q->extent[i];

			if ( d != extent->display )
			{
				continue;
			}

			img_w = extent->outputwindow.x2 - extent->outputwindow.x1;
			img_h = extent->outputwindow.y2 - extent->outputwindow.y1;	

			wireGLAssert( img_w < wiregl_pipe.actual_window_width );
			wireGLAssert( img_h < wiregl_pipe.actual_window_height );

			glRasterPos2f_pack( (extent->imagewindow.x1 - x)*2.0f/w - 1.0f,
								(extent->imagewindow.y1 - y)*2.0f/h - 1.0f );
			pixels = (unsigned char *) 
				glDrawPixels_pack( img_w, img_h, GL_RGB, GL_UNSIGNED_BYTE );

			glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
			glPixelStorei( GL_PACK_ALIGNMENT, 1 );
			glReadPixels( extent->outputwindow.x1, extent->outputwindow.y1,
						  img_w, img_h, GL_RGB, GL_UNSIGNED_BYTE, pixels );
		}

		glBarrierExec_pack( VIS_BARRIER );

		SendBuffer( wiregl_pipe.display[d].conn );
	}

	barrier_created = 1;

	if ( wiregl_pipe.vis_send_swaps )
	{
		for ( i = 0; i < wiregl_pipe.num_displays; i++ ) 
		{
			WireGLConnection *conn = wiregl_pipe.display[i].conn;

			SetupBuffer( );

			wireGLSwapBuffers_pack( );

			if ( wiregl_pipe.vis_sync )
			{
				conn->pending_writebacks++;
				wireGLWriteback_pack( );
			}

			SendBuffer( conn );
		}
	}
}

#if WIREGL_TRACK_DEPTH_COMPLEXITY
static void
__wireGLExtractDepthComplexityFromStencil( void )
{
	static int frame = 0;
	static FILE *dc_log = NULL;

	if ( frame > 1400 )
	{
		fclose( dc_log );
		dc_log = NULL;
	}
	else 
	{
		GLsizei        width    = wiregl_pipe.window_width;
		GLsizei        height   = wiregl_pipe.window_height;
		unsigned int   n_pixels = width * height;
		unsigned char *stencil;
		unsigned int i, accum;
		float depth_complexity;

		if ( dc_log == NULL )
		{
			dc_log = fopen( "/tmp/dc.log", "w" );
			if ( dc_log == NULL )
			{
				wireGLSimpleError( "Couldn't open depth log" );
			}
		}

		stencil = (unsigned char *) wireGLAlloc( n_pixels );
		glReadPixels( 0, 0, width, height,
					  GL_STENCIL_INDEX, GL_UNSIGNED_BYTE, stencil );

		accum = 0;
		for ( i = 0; i < n_pixels; i++ )
		{
			accum += stencil[i];
		}

		depth_complexity = (float) accum / (float) n_pixels;

		fprintf( dc_log, "frame=%6d depth_complexity=%8.3f\n",
				 frame, depth_complexity );

		glClear( GL_STENCIL_BUFFER_BIT );

		frame++;
	}
}
#endif

static void 
RotateColor( void )
{
	static unsigned char *buffer = NULL;
	static int size = 0;

	WireGLWorkQueue *q = __currentQueue;
	unsigned char *p;
	int i, j;

	if ( !wiregl_pipe.num_displays && !wiregl_pipe.rotate_color )
	{
		return;
	}

	glPushAttrib(GL_CURRENT_BIT | GL_ENABLE_BIT);
	glDisable(GL_TEXTURE_2D);
	glDisable(GL_TEXTURE_1D);
	glDisable(GL_LIGHTING);
	glDisable(GL_BLEND);
	glDisable(GL_ALPHA_TEST);
	glDisable(GL_DEPTH_TEST);
	glDisable(GL_FOG);
	glDisable(GL_STENCIL_TEST);
	glDisable(GL_SCISSOR_TEST);
	glDisable(GL_LOGIC_OP);

	glViewport( 0, 0, wiregl_pipe.actual_window_width,
				wiregl_pipe.actual_window_height );
	glDrawBuffer( GL_BACK );
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix( );
	glLoadIdentity( );
	
	glMatrixMode( GL_PROJECTION );
	glPushMatrix( );
	glLoadIdentity( );
	glOrtho( 0.0, wiregl_pipe.actual_window_width, 
			 0.0, wiregl_pipe.actual_window_height,
			 -1.0, 1.0 );
	glTranslatef( 0.375f, 0.375f, 0.0f );
	
	glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
	glPixelStorei( GL_PACK_ALIGNMENT, 1 );
	
	for ( i = 0; i < q->num_extents; i++ ) 
	{
		WireGLWorkQueueExtent *extent = &q->extent[i];

		float m00 = wiregl_pipe.colormatrix[0];
		float m01 = wiregl_pipe.colormatrix[1];
		float m02 = wiregl_pipe.colormatrix[2];
		float m03 = wiregl_pipe.colormatrix[3];
		float m10 = wiregl_pipe.colormatrix[4];
		float m11 = wiregl_pipe.colormatrix[5];
		float m12 = wiregl_pipe.colormatrix[6];
		float m13 = wiregl_pipe.colormatrix[7];
		float m20 = wiregl_pipe.colormatrix[8];
		float m21 = wiregl_pipe.colormatrix[9];
		float m22 = wiregl_pipe.colormatrix[10];
		float m23 = wiregl_pipe.colormatrix[11];

		int x = extent->outputwindow.x1;
		int y = extent->outputwindow.y1;
		int w = extent->outputwindow.x2 - extent->outputwindow.x1;
		int h = extent->outputwindow.y2 - extent->outputwindow.y1;
		
		if ( wiregl_pipe.num_displays ) 
		{
			int d = extent->display;
		
			if ( d != -1 && wiregl_pipe.display[d].rotate_color )
			{
				m00 = wiregl_pipe.display[d].colormatrix[0];
				m01 = wiregl_pipe.display[d].colormatrix[1];
				m02 = wiregl_pipe.display[d].colormatrix[2];
				m03 = wiregl_pipe.display[d].colormatrix[3];
				m10 = wiregl_pipe.display[d].colormatrix[4];
				m11 = wiregl_pipe.display[d].colormatrix[5];
				m12 = wiregl_pipe.display[d].colormatrix[6];
				m13 = wiregl_pipe.display[d].colormatrix[7];
				m20 = wiregl_pipe.display[d].colormatrix[8];
				m21 = wiregl_pipe.display[d].colormatrix[9];
				m22 = wiregl_pipe.display[d].colormatrix[10];
				m23 = wiregl_pipe.display[d].colormatrix[11];
			} 
			else if ( !wiregl_pipe.rotate_color )
			{
				continue;
			}
		}

		if ( buffer == NULL || size < w * h * 3 )
		{
			size =  w * h * 3; 
			if ( buffer ) wireGLFree( buffer );
			buffer = wireGLAlloc( size );
		}

		glReadPixels( x, y, w, h, GL_RGB, GL_UNSIGNED_BYTE, buffer );

		p = buffer;
		for ( j = 0; j < w * h; j++ )
		{
			const float r = p[0];
			const float g = p[1];
			const float b = p[2];
			float r_dst;
			float g_dst;
			float b_dst;

			r_dst = (r*m00 + g*m01 + b*m02 + m03);
			g_dst = (r*m10 + g*m11 + b*m12 + m13);
			b_dst = (r*m20 + g*m21 + b*m22 + m23);

			if (r_dst < 0.0f)   r_dst = 0.0f;
			if (r_dst > 255.0f) r_dst = 255.0f;
			if (g_dst < 0.0f)   g_dst = 0.0f;
			if (g_dst > 255.0f) g_dst = 255.0f;
			if (b_dst < 0.0f)   b_dst = 0.0f;
			if (b_dst > 255.0f) b_dst = 255.0f;

			*p++ = (unsigned char) r_dst;
			*p++ = (unsigned char) g_dst;
			*p++ = (unsigned char) b_dst;
		}
		glRasterPos2i( x, y );
		glDrawPixels( w, h, GL_RGB, GL_UNSIGNED_BYTE, buffer );
	}

	glPopMatrix( );
	glMatrixMode( GL_MODELVIEW );
	glPopMatrix( );
	
	
	glPopAttrib( );
}

static void
wireGLHandleWindowMessages( void )
{
#if defined(WINDOWS)
	MSG msg;
	while ( PeekMessage( &msg, wiregl_pipe.hWnd, 0, 0xffffffff, PM_REMOVE ) )
	{
		TranslateMessage( &msg );    
		DispatchMessage( &msg );
	}
	
	BringWindowToTop( wiregl_pipe.hWnd );
#endif
}
	
void
wireGLDeviceSwapBuffers( void )
{
#if defined(WINDOWS)
	SwapBuffers( wiregl_pipe.device_context );
#else
	glXSwapBuffers( wiregl_pipe.dpy, wiregl_pipe.window );
#endif
}

void 
__decodeSwapBuffers( void )
{
	static int feather_init = 0;
	
	INCR_DATA_PTR_NO_ARGS( );

#if WIREGL_TRACK_DEPTH_COMPLEXITY
	__wireGLExtractDepthComplexityFromStencil( );
#endif
		
	if ( !feather_init ) 
	{
		wireGLConfigureFeathering ( wiregl_pipe.window_width,
			wiregl_pipe.window_height,
			&wiregl_pipe.feather );
		feather_init = 1;
	}
	
	wireGLDrawFeatheringStrips( );

	if ( wiregl_pipe.rotate_color_flag ) 
	{
		RotateColor( );
	}
		
	if ( wiregl_pipe.use_L2 ) 
	{
		SwapL2( );
	} 
	else if ( wiregl_pipe.use_vis ) 
	{
		SwapVis( );
	}
	
	wireGLHandleWindowMessages( );

	wireGLDeviceSwapBuffers( );
}

void
__decodeSyncL2( void )
{
	int phase = READ_DATA( 0, int );
	INCR_DATA_PTR( 4 );

	if ( wiregl_pipe.L2.single_buffer )
	{
		if ( phase == 0 )
		{
			wireGLWarning( WIREGL_WARN_DEBUG, "SyncL2: skipping in single "
						   "buffer mode" );
		}
	}
	else
	{
		wireGLL2Sync( phase );
	}
}
