/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif
#include <GL/gl.h>

#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include "wiregl/include/wiregl_opcodes.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_pipe.h"
#include "wiregl/include/wiregl_util.h"
#include "wiregl/glcontext/glcontext.h"

#define FLOW_CONTROL 1
#define MAX_OUTSTANDING_BYTES ( 1 << 20 )

unsigned char *unpacker_data_ptr = NULL;

WireGLConnection *current_conn = NULL;
WireGLReceiveBuffer *receive_buffers = NULL;

static WireGLHashTable  *work_queues = NULL;
static WireGLWorkQueue  *work_ring   = NULL;
static struct {
	WireGLReceiveBuffer *head;
	WireGLReceiveBuffer *tail;
} pending = { NULL, NULL };

/* Are these two the same? */
WireGLWorkQueue *__currentWorkQueue( void ) { return work_ring; }
WireGLWorkQueue *__currentQueue;

typedef struct BucketRegion *BucketRegion_ptr;
typedef struct BucketRegion {
	int		         id;
	WireGLPipeExtent extents;
	BucketRegion_ptr right;
	BucketRegion_ptr up;
} BucketRegion;

#define HASHRANGE 256
BucketRegion *rhash[HASHRANGE][HASHRANGE];
BucketRegion *rlist;
int rlist_alloc = 0;

#define BKT_DOWNHASH(a, range) ((a)*HASHRANGE/(range))
#define BKT_UPHASH(a, range) ((a)*HASHRANGE/(range) + ((a)*HASHRANGE%(range)?1:0))

void 
buildhash (void) {
	int i, j, k, m;
	int r_len=0;
	int xinc, yinc;

	BucketRegion *rlist;
	BucketRegion *rptr;

	/* Allocate rlist */
	rlist_alloc = GLCONFIG_MAX_PROJECTORS*GLCONFIG_MAX_EXTENTS;
	rlist = (BucketRegion *) wireGLAlloc( rlist_alloc * sizeof(*rlist) );

	for ( i = 0; i < HASHRANGE; i++ )
	{
		for ( j = 0; j < HASHRANGE; j++ )
		{
			rhash[i][j] = NULL;
		}
	}

	/* Fill the rlist */
	xinc = wiregl_pipe.extent[0].x2 - wiregl_pipe.extent[0].x1;
	yinc = wiregl_pipe.extent[0].y2 - wiregl_pipe.extent[0].y1;

	rptr = rlist;
	for (i=0; i<wiregl_pipe.mural_w; i+=xinc) {
		for (j=0; j<wiregl_pipe.mural_h; j+=yinc) {
			for (k=0; k<wiregl_pipe.num_extents; k++) {
				if (wiregl_pipe.extent[k].x1 == i &&
					wiregl_pipe.extent[k].y1 == j) {
					rptr->extents = wiregl_pipe.extent[k];
					rptr->id = k;
					break;
				}
			} 
			if (k == wiregl_pipe.num_extents) {
				rptr->extents.x1 = i;
				rptr->extents.y1 = j;
				rptr->extents.x2 = i+xinc;
				rptr->extents.y2 = j+yinc;
				rptr->id = -1;
			}
			rptr++;
		}
	}
	r_len = rptr - rlist;

	/* Fill hash table */
	for ( i = 0; i <r_len; i++ )
	{
		BucketRegion *r = &rlist[i];
		
		for (k=BKT_DOWNHASH(r->extents.x1, wiregl_pipe.mural_w);
			 k<=BKT_UPHASH(r->extents.x2, wiregl_pipe.mural_w) && 
				 k < HASHRANGE;
			 k++) {
			for (m=BKT_DOWNHASH(r->extents.y1, wiregl_pipe.mural_h);
				 m<=BKT_UPHASH(r->extents.y2, wiregl_pipe.mural_h) &&
					 m < HASHRANGE;
				 m++) {
				if ( rhash[m][k] == NULL ||
					(rhash[m][k]->extents.x1 > r->extents.x1 &&
					 rhash[m][k]->extents.y1 > r->extents.y1)) {
					 rhash[m][k] = r;
				}
			}
		}
	}

	/* Initialize links */
	for (i=0; i<r_len; i++) {
		BucketRegion *r = &rlist[i];
		r->right = NULL;
		r->up    = NULL;
	}

	/* Build links */
	for (i=0; i<r_len; i++) {
		BucketRegion *r = &rlist[i];
		for (j=0; j<r_len; j++) {
			BucketRegion *q = &rlist[j];
			if (r==q) continue;

			/* Right Edge */
			if (r->extents.x2 == q->extents.x1 &&
				r->extents.y1 == q->extents.y1 &&
				r->extents.y2 == q->extents.y2) {
				r->right = q;
			}

			/* Upper Edge */
			if (r->extents.y2 == q->extents.y1 &&
				r->extents.x1 == q->extents.x1 &&
				r->extents.x2 == q->extents.x2) {
				r->up = q;
			}
		}
	}
}


static int
find_output_display( GLrecti rect )
{
	int i;

	if ( wiregl_pipe.num_displays == 0 )
	{
		/* we aren't using L2 or the VIS server, so this doesn't matter */
		return -1;
	}

	for ( i = 0; i < wiregl_pipe.num_displays; i++ )
	{
		if ( rect.x1 >= wiregl_pipe.display[i].x &&
			 rect.x2 <= wiregl_pipe.display[i].x + wiregl_pipe.display[i].w &&
			 rect.y1 >= wiregl_pipe.display[i].y &&
			 rect.y2 <= wiregl_pipe.display[i].y + wiregl_pipe.display[i].h )
		{
			return i;
		}
	}

	wireGLWarning( WIREGL_WARN_CRITICAL, "Cannot find display for extent "
				   "(%d, %d) -> (%d, %d)", 
				   rect.x1, rect.y1, rect.x2, rect.y2 );

	for ( i = 0; i < wiregl_pipe.num_displays; i++ ) 
	{
		wireGLWarning( WIREGL_WARN_CRITICAL, "\tDisplay %d"
					  " (%d, %d) -> (%d, %d)", i, 
					   wiregl_pipe.display[i].x, wiregl_pipe.display[i].y,
					   wiregl_pipe.display[i].x + wiregl_pipe.display[i].w,
					   wiregl_pipe.display[i].y + wiregl_pipe.display[i].h);
	}
	wireGLSimpleError("Quitting....");

	/* unreached */
	return -1;
}

static WireGLWorkQueue *
CreateWorkQueue( WireGLConnection *conn, WireGLConnection *peer_conn )
{
	WireGLWorkQueue *q;
	int i;
	int x, y, w, h, y_max;

	q = (WireGLWorkQueue *) wireGLAlloc( sizeof(*q) );
	q->context = __glCreateContext( );
	__glApplyViewTranform( q->context, 
						   (GLboolean) wiregl_pipe.apply_viewtransform );

#if 0
	/* Hack! Fix for viewport */
	q->context->viewport.v_x = 0;
	q->context->viewport.v_y = 0;
	q->context->viewport.v_h = wiregl_pipe.actual_window_height;
	q->context->viewport.v_w = wiregl_pipe.actual_window_width;
	q->context->viewport.v_valid = 1;
#endif

	x = wiregl_pipe.use_L2 ? 2 : 0;
	y = 0;
	y_max = 0;

	q->imagespace.x1 = 0;
	q->imagespace.y1 = 0;
	q->imagespace.x2 = wiregl_pipe.mural_w;
	q->imagespace.y2 = wiregl_pipe.mural_h;

	q->apply_viewtransform = wiregl_pipe.apply_viewtransform;
	q->num_extents = wiregl_pipe.num_extents;

	wireGLWarning( WIREGL_WARN_DEBUG, "%d extents", q->num_extents );
	for ( i = 0; i < q->num_extents; i++ ) 
	{
		WireGLWorkQueueExtent *extent = &q->extent[i];

		extent->imagewindow.x1 = wiregl_pipe.extent[i].x1;
		extent->imagewindow.y1 = wiregl_pipe.extent[i].y1;
		extent->imagewindow.x2 = wiregl_pipe.extent[i].x2;
		extent->imagewindow.y2 = wiregl_pipe.extent[i].y2;

		extent->display = find_output_display( extent->imagewindow );
		
		extent->bounds.x1 = ( (GLfloat) (2*extent->imagewindow.x1) / 
							  wiregl_pipe.mural_w - 1.0f );
		extent->bounds.y1 = ( (GLfloat) (2*extent->imagewindow.y1) / 
							  wiregl_pipe.mural_h - 1.0f );
		extent->bounds.x2 = ( (GLfloat) (2*extent->imagewindow.x2) / 
							  wiregl_pipe.mural_w - 1.0f );
		extent->bounds.y2 = ( (GLfloat) (2*extent->imagewindow.y2) /
							  wiregl_pipe.mural_h - 1.0f );

		w = wiregl_pipe.extent[i].x2 - wiregl_pipe.extent[i].x1;
		h = wiregl_pipe.extent[i].y2 - wiregl_pipe.extent[i].y1;

		if ( x + w > wiregl_pipe.actual_window_width ) 
		{
			y += y_max;
			x = ( wiregl_pipe.use_L2 ) ? 2 : 0;
			y_max = 0;
		}

		extent->outputwindow.x1 = x;
		extent->outputwindow.y1 = ( wiregl_pipe.actual_window_height - 
									wiregl_pipe.window_height - y );
		extent->outputwindow.x2 = x + w;
		extent->outputwindow.y2 = ( wiregl_pipe.actual_window_height - 
									wiregl_pipe.window_height - y + h );
	
		wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "q->bounds[%d]: "
					   "(%.3f %.3f) -> (%.3f %.3f)", i,
					   extent->bounds.x1, extent->bounds.y1,
					   extent->bounds.x2, extent->bounds.y2 ); 

		wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "q->imagewindow[%d]: "
					   "(%d %d) -> (%d %d)", i,
					   extent->imagewindow.x1, extent->imagewindow.y1,
					   extent->imagewindow.x2, extent->imagewindow.y2 );
 
		wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "q->outputwindow[%d]: "
					   "(%d %d) -> (%d %d)", i, 
					   extent->outputwindow.x1, extent->outputwindow.y1,
					   extent->outputwindow.x2, extent->outputwindow.y2 ); 

		if ( y_max < h )
		{
			y_max = h;
		}

		x += w + ( wiregl_pipe.use_L2 ? 2 : 0 );
	}

	q->head = q->tail = NULL;
	q->blocked = 0;
	q->delete_when_done = 0;
	q->conn = conn;
	q->peer_conn = peer_conn;
	q->next = q->prev = NULL;

	return q;
}

static void DestroyWorkQueue( WireGLWorkQueue *q )
{
	wireGLAssert( q->head == NULL && q->tail == NULL );
	__glDestroyContext( q->context );
	wireGLFree( q );
}

void
handle_first_connection( WireGLConnectionRequest *request )
{
	int i, j;

	/*
	** Extract the displays
	*/
	wiregl_pipe.num_displays  	 = request->num_displays;
	wiregl_pipe.num_pipes     	 = request->num_pipes;
	wiregl_pipe.pipe_num      	 = request->pipe_num;

	wiregl_pipe.L2.single_buffer = request->L2_single_buffer;

	for ( i = 0; i <request->num_displays; i++ )
	{
		char *p;
		char server[WIREGL_MAX_DISPLAY_NAME_LEN];

		/* Copy out the name */
		strcpy( wiregl_pipe.display[i].name, request->display[i].name );
		strcpy( server, request->display[i].name );

		wireGLWarning( WIREGL_WARN_CRITICAL, "Display: %s", server );

		/* parse the name */
		for (p = server; *p && *p != ':'; p++)
			*p = (char) tolower(*p);
		*p = '\0';
		
		if ( !strcmp( server, "lightning2" ) )
		{	

			/* Set l2 flag */
			wiregl_pipe.display[i].L2 = 1;
			wiregl_pipe.use_L2 = 1;

			/*
			** Get the port,channel for the L2 case, 
			*/
			wiregl_pipe.display[i].port = 0;
			wiregl_pipe.display[i].channel = 0;

			p = wiregl_pipe.display[i].name + strlen(server);
			
			if ( *p == ':' && *(p+1) == '/' && *(p+2) == '/' )
			{
				char *port;
				char *channel;
				char *q;
				p +=3;
				channel = p;

				for (q=channel; *q >= '0' && *q <= '9'; q++)
					;
				*q++ = '\0';
				for (port = q; *q >= '0' && *q <= '9'; q++)
					;
				*q = '\0';
				
				if ( !*port )
				{
					wireGLWarning( WIREGL_WARN_CRITICAL, 
								   "No Lighting2 Port found" );
				}
				else 
				{
					wiregl_pipe.display[i].port = atoi( port );
				}

				if ( !*channel )
				{
					wireGLWarning( WIREGL_WARN_CRITICAL,
								   "No Lighting2 Channel found" );
				}
				else 
				{
					wiregl_pipe.display[i].channel = atoi( channel );
				}
			} 
			else 
			{
				wireGLWarning( WIREGL_WARN_CRITICAL, 
							   "No Lighting2 Port/Channel found: %s",
							   request->display[i].name );
			}

			wireGLWarning( WIREGL_WARN_CRITICAL, "L2 Port %d: %d",
						   i, wiregl_pipe.display[i].port );
			wireGLWarning( WIREGL_WARN_CRITICAL, "L2 Channel %d: %d",
						   i, wiregl_pipe.display[i].channel );
		}

		/*
		Here we are connecting to a vis server.
		*/
		else 
		
		{
			WireGLConnectionRequest req;
			
			wiregl_pipe.use_vis = 1;

			memset( &req, 0, sizeof(req) );

			req.max_send     = request->max_send;
			req.window_x     = 0;
			req.window_y     = 0;
			req.feather      = request->feather;
			req.depth_bits   = request->depth_bits;
			req.stencil_bits = request->stencil_bits;

			req.num_extents  = 1;
			req.extent[0].x1 = 0;
			req.extent[0].y1 = 0;
			req.extent[0].x2 = request->display[i].w;
			req.extent[0].y2 = request->display[i].h;

			req.mural_w      = request->mural_w;
			req.mural_h      = request->mural_h;

			/* do not do transforms */
			req.apply_viewtransform = 0;
	
			req.peer[0]      = '\0';

			req.num_displays = 0;

			wiregl_pipe.display[i].conn = 
				wireGLConnectToServer( wiregl_pipe.display[i].name,
										WIREGL_TCPIP_DEFAULT_PORT, &req);

			if ( !wiregl_pipe.display[i].conn )
			{
				wireGLSimpleError( "Cannot connect to vis server: %s", 
								   server );
			}
		}

		/* Copy out the position data */
		wiregl_pipe.display[i].x = request->display[i].x;
		wiregl_pipe.display[i].y = request->display[i].y;
		wiregl_pipe.display[i].w = request->display[i].w;
		wiregl_pipe.display[i].h = request->display[i].h;

		wiregl_pipe.display[i].rotate_color = request->display[i].rotate_color;
		for ( j = 0; j < 12; j++ )
		{
			wiregl_pipe.display[i].colormatrix[j] = 
				request->display[i].colormatrix[j];
		}
		if ( wiregl_pipe.display[i].rotate_color )
		{
			wiregl_pipe.rotate_color_flag = 1;
		}
	}

	wiregl_pipe.window_x      = request->window_x;
	wiregl_pipe.window_y      = request->window_y;
	wiregl_pipe.depth_bits    = request->depth_bits;
	wiregl_pipe.stencil_bits  = request->stencil_bits;
	wiregl_pipe.feather       = request->feather;

	wiregl_pipe.num_extents   = request->num_extents;

	wiregl_pipe.window_height = 0;
	wiregl_pipe.window_width  = 0;

	wiregl_pipe.mural_w       = request->mural_w;
	wiregl_pipe.mural_h       = request->mural_h;

	wiregl_pipe.rotate_color  = request->rotate_color;
	for ( i = 0; i < 12; i++ )
	{
		wiregl_pipe.colormatrix[i] = request->colormatrix[i];
	}
	
	if ( wiregl_pipe.rotate_color )
	{
		wiregl_pipe.rotate_color_flag = 1;
	}
	
	wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "request->extents: %d",
				   request->num_extents);

	for ( i = 0; i < request->num_extents; i++ ) 
	{
		wiregl_pipe.extent[i].x1 = request->extent[i].x1;
		wiregl_pipe.extent[i].y1 = request->extent[i].y1;
		wiregl_pipe.extent[i].x2 = request->extent[i].x2;
		wiregl_pipe.extent[i].y2 = request->extent[i].y2;
	
		wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "extent[%d]: (%d, %d, "
					   "%d, %d)", i,
					   request->extent[i].x1, request->extent[i].x2, 
					   request->extent[i].y1, request->extent[i].y2 );
		
		wiregl_pipe.window_width += 
			wiregl_pipe.extent[i].x2 - wiregl_pipe.extent[i].x1;
		if ( wiregl_pipe.window_height < 
			 wiregl_pipe.extent[i].y2 - wiregl_pipe.extent[i].y1 )
		{
			wiregl_pipe.window_height = 
				wiregl_pipe.extent[i].y2 - wiregl_pipe.extent[i].y1;
		}
	}

	wireGLWarning( WIREGL_WARN_VERBOSE_DEBUG, "window size: %dx%d",
				   wiregl_pipe.window_width, wiregl_pipe.window_height);

	wiregl_pipe.apply_viewtransform = request->apply_viewtransform;
	wiregl_pipe.vis_send_swaps = request->vis_send_swaps;

	wiregl_pipe.optimize_bucket = request->optimize_bucket;
	if (wiregl_pipe.optimize_bucket) 
		buildhash();
}

void 
wireGLPipeserverConnect( WireGLConnection *conn, 
						 WireGLConnectionRequest *request )
{
	static int first_connection = 1;
	WireGLWorkQueue  *q;
	WireGLConnection *peer;

	wireGLWarning( WIREGL_WARN_NOTICE, "new client, sender_id=%u",
				   conn->sender_id );

	if ( first_connection ) 
	{
		handle_first_connection( request );
	}

	peer = NULL;
	if ( request->peer[0] ) 
	{
		peer = wireGLConnectToServer( request->peer, 
									  WIREGL_TCPIP_DEFAULT_PORT,
									  request + 1 );
	}

	if ( work_queues == NULL )
	{
		work_queues = wireGLAllocHashtable( );
		if ( !wiregl_pipe.no_graphics )
		{
			wireGLCreateWindow( );
		}
	}

	q = CreateWorkQueue( conn, peer );

	if ( wiregl_pipe.use_L2 && first_connection && !wiregl_pipe.no_graphics )
	{
		if ( wiregl_pipe.L2.single_buffer )
		{
			wireGLL2MakeFrame( &wiregl_pipe.L2.frame[0], 
							   WIREGL_L2_SINGLE_BUFFER, q );
		}
		else
		{
			wireGLL2MakeFrame( &wiregl_pipe.L2.frame[0], 
							   WIREGL_L2_DOUBLE_BUFFER_A, q );
			wireGLL2MakeFrame( &wiregl_pipe.L2.frame[1],
							   WIREGL_L2_DOUBLE_BUFFER_B, q );
		}
	}

	wireGLHashtableAdd( work_queues, conn->sender_id, q );

	if ( !work_ring )
	{
		work_ring = q;
		q->next = q;
		q->prev = q;
	}
	else
	{
		q->next = work_ring->next;
		work_ring->next->prev = q;

		q->prev = work_ring;
		work_ring->next = q;
	}

	first_connection = 0;
}

void 
wireGLPipeserverDisconnect( unsigned int sender_id )
{
	WireGLWorkQueue *q;
	wireGLWarning( WIREGL_WARN_NOTICE, "disconnecting sender_id=%u",
				   sender_id );

	q = (WireGLWorkQueue *) wireGLHashtableSearch( work_queues, sender_id );
	if ( q ) 
	{
		/* don't call delete, it will free() the value! */
		/* wireGLHashtableDelete( work_queues, sender_id ); */
		q->delete_when_done = 1;
	}
	else
	{
		wireGLWarning( WIREGL_WARN_CRITICAL, "wireGLDisconnect: sender_id=%u,"
					   " no such queue?", sender_id );
	}
}

static WireGLReceiveBuffer *
AllocReceiveBuffer( void )
{
    WireGLReceiveBuffer *recv_buf;

    if ( !receive_buffers )
    {
        recv_buf = (WireGLReceiveBuffer *) wireGLAlloc( sizeof(*recv_buf) );
    }
    else
    {
        recv_buf = receive_buffers;
        receive_buffers = receive_buffers->next;
    }

    recv_buf->next = NULL;
    return recv_buf;
}

static void
FreeReceiveBuffer( WireGLReceiveBuffer *recv_buf )
{
    recv_buf->next = receive_buffers;
    receive_buffers = recv_buf;
}

static void 
PutReceiveBufferOnQueue( WireGLWorkQueue *q, WireGLReceiveBuffer *buf )
{
	buf->next = NULL;

	if ( q->tail )
		q->tail->next = buf;
	else
		q->head = buf;
	q->tail = buf;
}

void 
wireGLPipeserverReceive( WireGLConnection *conn, void *buf, unsigned int len )
{
	WireGLMessage *msg = (WireGLMessage *) buf;

	switch ( msg->type )
	{
	  case WIREGL_MESSAGE_OPCODES:
		{
			WireGLReceiveBuffer *recv_buf = AllocReceiveBuffer( );

			recv_buf->conn     = conn;
			recv_buf->length   = len;
			recv_buf->buffer   = buf;

			if ( pending.tail ) 
				pending.tail->next = recv_buf;
			else
				pending.head = recv_buf;
			pending.tail = recv_buf;

		}
		break;

	  case WIREGL_MESSAGE_WRITEBACK:
		{
			conn->pending_writebacks--;
			wireGLNetFree( conn, buf );
		}
		break;

	  default:
		wireGLError( "wireGLPipeserverReceive: msg->type=0x%x?", msg->type );
		break;
	}
}

static int
wireGLPipeGetMoreWork( void )
{
	int work = wireGLNetRecv( );

	while ( pending.head )
	{
		WireGLReceiveBuffer *recv_buf;
		WireGLMessage       *msg;
		WireGLWorkQueue     *q;

		recv_buf = pending.head;

		pending.head = pending.head->next;
		if ( pending.head == NULL )
			pending.tail = NULL;

		msg = (WireGLMessage *) recv_buf->buffer;

		wireGLAssert( work_queues );
	
		q = (WireGLWorkQueue *) 
			wireGLHashtableSearch( work_queues, msg->opcodes.senderId );
		if ( q == NULL )
		{
			wireGLError( "No queue for sender=%u", msg->opcodes.senderId );
		}

		if ( q->peer_conn ) 
		{
			wireGLNetSend( q->peer_conn, NULL, msg, recv_buf->length );
		}

		PutReceiveBufferOnQueue( q, recv_buf );
	}

	return work;
}


static WireGLWorkQueue *
GetNextQueue( void )
{
	for ( ; ; )
	{
		WireGLWorkQueue *q;

		if ( work_ring && work_ring->delete_when_done )
		{
			/* only delete this queue when all its buffers are empty
			 * (presumably we got the delete notification in order,
			 * and once we've emptied all the current buffers we won't
			 * get any more) */

			q = work_ring;
			if ( q->head == NULL )
			{
				if ( q == q->next )
				{
					work_ring = NULL;
				}
				else
				{
					work_ring     = q->next;
					q->prev->next = q->next;
					q->next->prev = q->prev;
				}

				/* additionally, if the queue is blocked then there is
				 * a pointer in a parallel api structure somewhere to
				 * it, so it will get unblocked eventually.  We should
				 * wait for it to get unblocked.  Unfortunately, if
				 * the client that will unblock this queue exits, then
				 * the semaphoreV/barrier/etc. will never arrive.
				 * Gargh.  Should we be reference couting these?
				 * Uggggly. */
				if ( !q->blocked )
				{
					wireGLWarning( WIREGL_WARN_NOTICE, "destroying q=0x%p "
								   "sender_id=%u", q, q->conn->sender_id );
					DestroyWorkQueue( q );
				}
				else
				{
					wireGLWarning( WIREGL_WARN_CRITICAL, "it's time to "
								   "release the resources held by "
								   "sender_id=%u, but it tagged as blocked, "
								   "which means there is an outstanding "
								   "pointer to them.", q->conn->sender_id );
				}
			}
		}

		/* now look for something to do */
		if ( work_ring )
		{
			int all_blocked = 1;
			WireGLWorkQueue *start = work_ring;
			while ( work_ring->next != start ) 
			{
				if ( !work_ring->blocked )
					all_blocked = 0;

				/* work on this guy if he isn't blocked and he has at
				 * least 1 buffer of work to do */
				if ( !work_ring->blocked && work_ring->head )
					return work_ring;

				work_ring = work_ring->next;
			}

			if ( !work_ring->blocked )
				all_blocked = 0;

			if ( !work_ring->blocked && work_ring->head )
				return work_ring;

			if ( all_blocked )
			{
				wireGLWarning( WIREGL_WARN_CRITICAL, "all the available "
							   "clients are blocked" );
			}
		}

		wireGLPipeGetMoreWork( );
	}

	/* UNREACHABLE */
	/* return NULL; */
}


static WireGLReceiveBuffer *
GetNextBufferFromQueue( WireGLWorkQueue *q )
{
	WireGLReceiveBuffer *buf;

	buf = q->head;
	if ( buf ) 
	{
		q->head = q->head->next;
		if ( q->head == NULL )
			q->tail = NULL;
	}

	return buf;
}

void wireGLCheckGLErrors( void )
{
	GLenum err;

	err = glGetError( );
	while ( err != GL_NO_ERROR ) {

#define X(ex)   case ex: msg = # ex;

		char *msg, buf[64];
		switch ( err )
		{
			X(GL_INVALID_ENUM);
			X(GL_INVALID_VALUE);
			X(GL_INVALID_OPERATION);
			X(GL_STACK_OVERFLOW);
			X(GL_STACK_UNDERFLOW);
			X(GL_OUT_OF_MEMORY);
		  default:
			sprintf( buf, "UNKNOWN ERROR (0x%x)", err );
			msg = buf;
		}
		wireGLWarning( WIREGL_WARN_CRITICAL, "glGetError -> %s", msg );

		err = glGetError( );
#undef X
	}
}

static unsigned char
RenderCommands( unsigned char *buffer, int num_opcodes, int length )
{
	int i;
	unsigned char *save_unpacker_data_ptr, *opcode_ptr;
	unsigned char op = WIREGL_NOP_OPCODE;

	/* make sure we are word aligned */
	wireGLAssert( ((int) buffer & 0x3 ) == 0 );

	save_unpacker_data_ptr = unpacker_data_ptr;

	unpacker_data_ptr = buffer + (( num_opcodes + 3 ) & ~0x3 );
	opcode_ptr        = unpacker_data_ptr - 1;

	wireGLAssert( unpacker_data_ptr <= buffer + length );

	for ( i = 0; i < num_opcodes; i++ )
	{
		op = *opcode_ptr--;
#if 0
		printf( "%d/%d: op=%3u ptr=0x%p ", i, num_opcodes,
			op, unpacker_data_ptr );
		wireGLPrintFunc( (WireGLOpcode) op, 
						 (unsigned char *) unpacker_data_ptr );

		if ( op == WIREGL_SWAPBUFFERS_OPCODE )
			printf("--- swap ---------------------\n");
#endif

#if 0
		switch ( op )
		{
		  case WIREGL_LOADMATRIXF_OPCODE:
			INCR_DATA_PTR( 16 * 4 );
			break;

		  case WIREGL_BOUNDSINFO_OPCODE:
			INCR_VAR_PTR( );
			break;

		  case WIREGL_MATRIXMODE_OPCODE:
		  case WIREGL_PUSHMATRIX_OPCODE:
		  case WIREGL_LOADIDENTITY_OPCODE:
		  case WIREGL_POPMATRIX_OPCODE:
		  case WIREGL_DISABLE_OPCODE:
		  case WIREGL_ENABLE_OPCODE:
		  case WIREGL_CLEAR_OPCODE:
			INCR_DATA_PTR( 4 );
			break;

		  case WIREGL_MATERIALFV_OPCODE:
			INCR_VAR_PTR( );
			break;

		  default:
			printf( "%d/%d: op=%3u ptr=0x%p ", i, num_opcodes,
					op, unpacker_data_ptr );
			wireGLPrintFunc( (WireGLOpcode) op, 
							 (unsigned char *) unpacker_data_ptr );

			if ( op == WIREGL_SWAPBUFFERS_OPCODE )
				printf("--- swap ---------------------\n");

			wireGLAssert( op <= WIREGL_NOP_OPCODE );
			wiregl_decode_functions[op]( );
		}

#else		

		wireGLAssert( op <= WIREGL_NOP_OPCODE );
		wiregl_decode_functions[op]( );
#endif
	}

	/* we should have decoded exactly all of the data */
	wireGLAssert( unpacker_data_ptr == buffer + length );

	/* when we're in release mode we don't actually look at the length */
	WIREGL_UNUSED(length);

	unpacker_data_ptr = save_unpacker_data_ptr;

	/* return the last opcode */
	return op;
}

void
__decodeBoundsInfo( void ) 
{
	int i, len;
	GLrecti bounds;
	int num_opcodes;
	unsigned char *payload;

	WireGLWorkQueue *q = __currentQueue;

	/* 
	INCR_VAR_PTR( );
	return;
	*/

	len         = READ_DATA(  0, GLint   );
	bounds.x1   = READ_DATA(  4, GLint   );
	bounds.y1   = READ_DATA(  8, GLint   );
	bounds.x2   = READ_DATA( 12, GLint   );
	bounds.y2   = READ_DATA( 16, GLint   );
	num_opcodes = READ_DATA( 20, GLint   );
	payload     = DATA_POINTER( 24, unsigned char );

#if 0
	printf( "bounds: (%d, %d) -> (%d, %d)\n", 
			bounds.x1, bounds.y1, bounds.x2, bounds.y2 );
#endif

	if ( wiregl_pipe.optimize_bucket )
	{
		BucketRegion *r;
		BucketRegion *p;
		
		for (r = rhash[BKT_DOWNHASH(bounds.y1, wiregl_pipe.mural_h)][BKT_DOWNHASH(bounds.x1, wiregl_pipe.mural_w)];
			 r && bounds.y2 > r->extents.y1;
			 r = r->up)
		{
			for (p=r; p && bounds.x2 > p->extents.x1; p = p->right)
			{
				if ( p->id != -1 && 
					 bounds.x1 < p->extents.x2  &&
					 /* ibounds.x2 > p->extents.x1 && */
					 bounds.y1 < p->extents.y2 && 
					 bounds.y2 > p->extents.y1 )
				{
					if ( q->apply_viewtransform )
					{
						__glSetOutputBounds( q->context,
											 &q->extent[p->id].outputwindow, 
											 &q->imagespace,
											 &q->extent[p->id].imagewindow );
					}
					
					RenderCommands( payload, num_opcodes, len - 24 );
				}
			}
		}

	} else {

		for ( i = 0; i < q->num_extents; i++ ) 
		{
			WireGLWorkQueueExtent *extent = &q->extent[i];
		
	#if 0
			printf( "q->imagewindow: (%d, %d) -> (%d, %d)\n", 
					extent->imagewindow.x1, extent->imagewindow.y1,
					extent->imagewindow.x2, extent->imagewindow.y2 );
	#endif
			if ( !( extent->imagewindow.x2 > bounds.x1 &&
					extent->imagewindow.x1 < bounds.x2 &&
					extent->imagewindow.y2 > bounds.y1 &&
					extent->imagewindow.y1 < bounds.y2 ) ) 
				continue;

			if ( q->apply_viewtransform )
			{
				__glSetOutputBounds( q->context, &extent->outputwindow, 
									 &q->imagespace, &extent->imagewindow );
			}
			
			RenderCommands( payload, num_opcodes, len - 24 );
		}
	}

	INCR_VAR_PTR( );
}

static void
wireGLHandleWritebacksWhenDroppingBuffers( WireGLReceiveBuffer *buf )
{
	/* This is a total hack, based on knowing that the only
	 * opcodes that require us to do a writeback are
	 * READ_PIXELS and WRITEBACK, that they'll always be the
	 * last opcode in a block, and that the addr of the
	 * writeback is the last argument.  This isn't integrated
	 * with ring mode, so that mode will still be broken for
	 * writebacks when the graphics are being pitched */
	WireGLMessageOpcodes *msg  = (WireGLMessageOpcodes *) buf->buffer;
	unsigned char        *data = (unsigned char *) msg + sizeof(*msg);
	unsigned char        *opcode;

	opcode = data + 3 - (( msg->numOpcodes - 1 ) & 0x3 );

	if ( *opcode == WIREGL_READPIXELS_OPCODE ||
		 *opcode == WIREGL_WRITEBACK_OPCODE )
	{
		WireGLMessageWriteback *reply = 
			(WireGLMessageWriteback *) wireGLNetAlloc( buf->conn );
				
		reply->type = WIREGL_MESSAGE_WRITEBACK;

		wireGLNetSend( buf->conn, (void **) &reply, reply, sizeof(*reply) );
	}
}

void 
wireGLRenderRemoteStreams( void )
{
	if ( wiregl_pipe.no_graphics )
	{
		for ( ; ; )
		{
			WireGLWorkQueue     *q   = GetNextQueue( );
			WireGLReceiveBuffer *buf = GetNextBufferFromQueue( q );

			if ( buf )
			{
				wireGLHandleWritebacksWhenDroppingBuffers( buf );

				wireGLNetFree( buf->conn, buf->buffer );
				FreeReceiveBuffer( buf );
			}

			work_ring = work_ring->next;
		}
	}


	for ( ; ; )
	{
		int switch_contexts = 0;
		WireGLReceiveBuffer *buf;

		WireGLWorkQueue *q = GetNextQueue( );
		__currentQueue = q;

		__glMakeCurrent( q->context );

		buf = GetNextBufferFromQueue( q );
		while ( buf )
		{
			WireGLMessageOpcodes *hdr = (WireGLMessageOpcodes *) buf->buffer;
			unsigned char last_op;
			current_conn = q->conn;

			last_op = 
				RenderCommands( (unsigned char *) buf->buffer + sizeof(*hdr),
								hdr->numOpcodes, buf->length - sizeof(*hdr) );

			if ( wiregl_pipe.debug )
			{
				wireGLCheckGLErrors( );
			}

			/* we've stopped rendering, lets grab the current state */
			__glExtractCurrentState( );

			switch( last_op )
			{
				case WIREGL_BARRIEREXEC_OPCODE:
				case WIREGL_SEMAPHOREP_OPCODE:
				case WIREGL_SEMAPHOREV_OPCODE:
					switch_contexts = 1;
					break;
			}

			/* a round-about way of saying "if we are blocked, it had
			 * better be because the last opcode in the block (the one
			 * we'll test to set switch_contexts) caused it */
			wireGLAssert( ( q->blocked && switch_contexts ) || !q->blocked );

			wireGLNetFree( buf->conn, buf->buffer );
			FreeReceiveBuffer( buf );

			if ( switch_contexts )
				break;

			buf = GetNextBufferFromQueue( q );
		}

		work_ring = work_ring->next;
	}
}
