/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#endif
#include <GL/gl.h>

#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_pipe.h"
#include "wiregl/include/wiregl_common.h"

/* Structures
 * ==========
 *
 * CompressedNormalT
 * -----------------
 * This structure stores the compressed data that is produced by the
 * compression algorithm. The symmetry field contains information
 * regarding in which octant and sextant the normal lies, and the left
 * and right values contain a compressed version of nx/nz and ny/nz.
 *
 * NormalDataT
 * -----------
 * This type is used to force the compiler to write the
 * CompressedNormalT to the data buffer as a specific number of
 * bytes. Normally, the CompressedNormalT would take up four bytes
 * even if only the first two are being used. So, we declare a
 * separate type that takes up the amount of room we want to use in
 * the data buffer and use a pointer cast to write only that much
 * data.
 *
 * COMPRESSION_SHIFT
 * -----------------
 * This value dictates how many bits should be removed from the
 * mantissa of the floating point value in the quantization step.
 */

typedef struct
{
	unsigned symmetry : 6;
	unsigned leftValue : 5;
	unsigned rightValue : 5;
} CompressedNormalT;

typedef unsigned short NormalDataT;

#define QUANTIZATION_LEVEL 5

#include <math.h>
#include "../util/floatlib.c"

// keep the data word-aligned
#define NORMAL_SIZE ((sizeof(NormalDataT) + 3) & ~0x3)

#define NEGATIVE_X 0x0001
#define NEGATIVE_Y 0x0002
#define NEGATIVE_Z 0x0004
#define SEXTANT_XY 0x0008
#define SEXTANT_XZ 0x0010
#define SEXTANT_YZ 0x0020

static GLfloat compressionDiv = (1 << (QUANTIZATION_LEVEL)) - 1;

GLfloat *normalLookup;

#ifdef WINDOWS
#pragma warning( disable : 4725 ) 
#define RETRIEVE_VALUES() __asm		\
	{								\
		__asm mov  esi, outValues	\
		__asm add  esi, 8			\
									\
		__asm fild leftMantissa		\
		__asm fdiv compressionDiv	\
		__asm fst  nxnz				\
		__asm fmul st(0),st(0)		\
									\
		__asm fild rightMantissa	\
		__asm fdiv compressionDiv	\
		__asm fst  nynz				\
		__asm fmul st(0),st(0)		\
									\
		__asm fld1					\
		__asm fadd					\
		__asm fadd					\
		__asm fsqrt					\
		__asm fld1					\
		__asm fdivr					\
		__asm fst  dword ptr [esi]	\
		__asm fld  st(0)			\
									\
		__asm sub  esi,4			\
		__asm fld  nynz				\
		__asm fmul					\
		__asm fstp dword ptr [esi]	\
									\
		__asm sub  esi,4			\
		__asm fld  nxnz				\
		__asm fmul					\
		__asm fstp dword ptr[esi]	\
	}

#else	/* !WINDOWS */

#define RETRIEVE_VALUES()	nxnz = leftMantissa / compressionDiv; \
	nynz = rightMantissa / compressionDiv; \
	outValues[2] = 1 / sqrtf(nxnz * nxnz + nynz * nynz + 1); \
	outValues[1] = nynz * outValues[2]; \
	outValues[0] = nxnz * outValues[2]; \

#endif	/* !WINDOWS */

/* swap
 * ----
 * This function swaps two floating point values.
 */

static __inline void swap(GLfloat *in1, GLfloat *in2)
{
	GLfloat temp;
	temp = *in1;
	*in1 = *in2;
	*in2 = temp;
}

/* __retrieveValues
 * ----------------
 * This function decompresses the values sent by the WireGL
 * client. The data still needs to be sorted and have its signs
 * restored after this function.  
 */

static __inline void __retrieveValues ( unsigned int leftMantissa, unsigned int rightMantissa, GLfloat * outValues )
{
	GLfloat nxnz, nynz;

	RETRIEVE_VALUES();
}

/* __assignOctantAndSextant
 * ------------------------
 * This function restores the order and signs of the values given the
 * symmetry data passed to it by the WireGL client.  
 */

static __inline void __assignOctantAndSextant ( unsigned char symmetry, GLfloat * outValues )
{
	if(symmetry & SEXTANT_XY)							// x < y
	{
		if(symmetry & SEXTANT_XZ)						// x < y && x < z
		{
			if(!(symmetry & SEXTANT_YZ))				// x < z < y
				swap( &outValues[1], &outValues[2] );
		}
		else											// z < x < y
			swap( &outValues[1], &outValues[2] );
	}
	else												// y < x
	{
		if(symmetry & SEXTANT_YZ)						// y < x && y < z
		{
			if(!(symmetry & SEXTANT_XZ))				// y < z < x
				swap( &outValues[0], &outValues[2] );
		}
		else											// z < y < x
			swap( &outValues[0], &outValues[2] );
	}

	if(symmetry & NEGATIVE_X)
		outValues[0] = -outValues[0];
	if(symmetry & NEGATIVE_Y)
		outValues[1] = -outValues[1];
	if(symmetry & NEGATIVE_Z)
		outValues[2] = -outValues[2];
}

void wireGLInitNormalLookupTable( void )
{
	int i;
	CompressedNormalT *normal;
	GLfloat *cp;
	
	cp = normalLookup = (GLfloat*)wireGLAlloc( (1 << (sizeof(NormalDataT) * 8)) * 4 * sizeof(GLfloat) );
	
	for(i = 0; i < (1 << (sizeof(NormalDataT) * 8)); i++)
	{
		normal = (CompressedNormalT*)&i;
		__retrieveValues(normal->leftValue, normal->rightValue, cp);
		__assignOctantAndSextant((unsigned char)normal->symmetry, cp);
		cp += 4;
	}
}

void __decodeNormal3Compressed( void )
{
	NormalDataT inData;
	//	CompressedNormalT *inNormal = (CompressedNormalT*)&inData;

	inData = READ_DATA(0, NormalDataT);

	/* Retrieve the values, restore the symmetry and call glNormal. */
	
	//__retrieveValues(inNormal->leftValue, inNormal->rightValue, v);
	//__assignOctantAndSextant(inNormal->symmetry, v);
	//glNormal3fv( ( GLfloat* ) v );

	glNormal3fv( normalLookup + ((unsigned long)inData << 2) );

	INCR_DATA_PTR( NORMAL_SIZE );
}
