/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/
#ifdef WINDOWS
#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#else
#include <sys/time.h>
#endif
#include <GL/gl.h>

#include <ctype.h>

#include "wiregl/include/wiregl_util.h"
#include "wiregl/include/wiregl_serial.h"
#include "wiregl/include/wiregl_lightning2.h"
#include "wiregl/include/wiregl_pipe.h"

#define USE_DRAWPIXELS 0

/* type and macro definitions for manipulating strip headers */

#define LGH_NULL_COLUMN (0xf)

/* first pixel */
#define LGH_STRIP_PARITY_POS	55
#define LGH_STRIP_PARITY_LEN	1
#define LGH_STRIP_ADDRESS_POS	32
#define LGH_STRIP_ADDRESS_LEN	23

/* second pixel */
#define LGH_STRIP_BUFFER_POS	23
#define LGH_STRIP_BUFFER_LEN	1
#define LGH_STRIP_COLUMN_POS	19
#define LGH_STRIP_COLUMN_LEN	4
#define LGH_STRIP_WIDTH_POS		8
#define LGH_STRIP_WIDTH_LEN		11
#define LGH_STRIP_OPCODE_POS	0
#define LGH_STRIP_OPCODE_LEN	8

#define LGH_STRIP_SET_FIELD(strip,x,pos,len)					        \
{															            \
		WireGLL2StripHeader mask = ((((WireGLL2StripHeader)1)<<(len))-1) << (pos); \
		strip = ((strip) & ~mask) | ((WireGLL2StripHeader)(x) << (pos));	        \
}

#define LGH_STRIP_SET(strip,x,field) \
	LGH_STRIP_SET_FIELD(strip,x,LGH_STRIP_##field##_POS,LGH_STRIP_##field##_LEN)

#define LGH_STRIP_GET_FIELD(strip,pos,len) \
    (((strip) >> (pos)) & ((1 << (len)) - 1))

#define LGH_STRIP_GET(strip,field) \
    LGH_STRIP_GET_FIELD(strip,LGH_STRIP_##field##_POS,LGH_STRIP_##field##_LEN)

/*
--                        Number of displays selected
  --             +-------+-------+-------+-------+-------+-------+
  --             |   2   |   3   |   4   |   5   |   6   |   8   |
  --             +-------+-------+-------+-------+-------+-------+
  --             | Start | Start | Start | Start | Start | Start |
  --   Display   |  row  |  row  |  row  |  row  |  row  |  row  |
  --   -------   +-------+-------+-------+-------+-------+-------+
  --      0      |     0 |     0 |     0 |     0 |     0 |     0 |
  --      1      |  1024 |   640 |   512 |   384 |   320 |   256 |
  --      2      |   N/A |  1280 |  1024 |   768 |   640 |   512 |
  --      3      |   N/A |   N/A |  1536 |  1152 |   960 |   768 |
  --      4      |   N/A |   N/A |   N/A |  1536 |  1280 |  1024 |
  --      5      |   N/A |   N/A |   N/A |   N/A |  1600 |  1280 |
  --      6      |   N/A |   N/A |   N/A |   N/A |   N/A |  1536 |
  --      7      |   N/A |   N/A |   N/A |   N/A |   N/A |  1792 |
*/
#if 0
static int baseAddress2Displays[2]= {0, 1024};
static int baseAddress3Displays[3]= {0, 640, 1280};
static int baseAddress4Displays[4]= {0, 512, 1024, 1536};
static int baseAddress5Displays[5]= {0, 384, 768, 1152, 1536};
static int baseAddress6Displays[6]= {0, 320, 640, 960, 1280, 1600};
static int baseAddress8Displays[8]= {0, 256, 512, 768, 1024, 1280, 1536, 1792};
#endif

static int baseAddress4Displays[4]= {0, 512, 1024, 1536};

static int * 
computeBaseAddress( void )
{
	/* return baseAddress2Displays; */
	return baseAddress4Displays;
}

/*
  --             Start bank,   Start bank,   Start bank,   Start bank,
  --   Display   2 displays    5 displays    6 displays    all others
  --   -------   -----------   -----------   -----------   -----------
  --      0           0             0             0             0
  --      1           2             1             1             1
  --      2          N/A            2             2             2
  --      3          N/A            3             3             3
  --      4          N/A            2             0             0
  --      5          N/A           N/A            2             1
  --      6          N/A           N/A           N/A            2
  --      7          N/A           N/A           N/A            3
*/

#if 0
static int startingBank2Displays[2]= {0, 2};
static int startingBank3Displays[3]= {0, 1, 2};
static int startingBank4Displays[4]= {0, 1, 2, 3};
static int startingBank5Displays[5]= {0, 1, 2, 3, 2};
static int startingBank6Displays[6]= {0, 1, 2, 3, 0, 2};
static int startingBank8Displays[8]= {0, 1, 2, 3, 0, 1, 2, 3};
#endif 

static int startingBank4Displays[4]= {0, 1, 2, 3};

static int * 
computeStartingBank( void)
{
	/* return startingBank2Displays; */
	return startingBank4Displays;
}

static int 
getXResolution(void)
{
	/* return 1280; */
	return 1024;
}

/* computing this address assuming that we know the baseAddress, the
 * number of displays, refresh and resolution */

static int 
computeStartingAddress(int display) 
{
	/* XXX - stuff we need to make faster */

	/* assume know base address at this point */
	int *baseAddress = computeBaseAddress();

	/* assume we know the starting bank at this point */
	int *startingBank = computeStartingBank();

	/* initialize startingaddress to memory address of pixel 0,0 */
	return  (	((baseAddress[display]<<12) & 0x007FF000) |
				((startingBank[display]<<10) & 0x00000C00) );
}


static int
lghComputeAddress( int x, int y, int display )
{	
	int xRes = getXResolution();
	int baseAddress = computeStartingAddress(display);
	int offset = y * xRes + x;
	return baseAddress + offset;
}


static int
lghComputeParity( WireGLL2StripHeader strip )
{
	struct i64 {unsigned long a, b;} x64;

	unsigned long x;

	//initially set parity bit to 0
	LGH_STRIP_SET(strip,0,PARITY);

	x64 = *((struct i64 *) &strip);

	x = x64.a ^ x64.b;

	x = x ^ (x >> 16);
	x = x ^ (x >>  8);
	x = x ^ (x >>  4);
	x = x ^ (x >>  2);
	x = x ^ (x >>  1);

	return (int)( x & 1 );
}

static WireGLL2StripHeader
wireGLL2StripHeaderSwap( WireGLL2StripHeader x )
{
	WireGLL2StripHeader temp = 0;
	unsigned char *dst = (unsigned char *) &temp;
	unsigned char *src = (unsigned char *) &x;

	dst[0] = src[6];
	dst[1] = src[5];
	dst[2] = src[4];

	dst[6] = src[0];
	dst[5] = src[1];
	dst[4] = src[2];

	return temp;
}

static WireGLL2StripHeader
wireGLL2StripHeaderCreate( int column,
						   int buffer,
						   int display,
						   int x,
						   int y,
						   int width,
						   int opcode )
{
	WireGLL2StripHeader strip;
	int address;
	int parity;

	wireGLAssert( column >= 0 && column < 15 );
	wireGLAssert( buffer == 0 || buffer == 1 );
	wireGLAssert( display >= 0 && display < 8 );
	wireGLAssert( x >= 0 && x < 1024 );
	wireGLAssert( y >= 0 && y < 768 );
	wireGLAssert( width > 0 && x + width <= 1024 );
	wireGLAssert( opcode >= 0 && opcode < 256 );

	/* GWS */
	wireGLAssert( column == 0 );

	strip = 0;
	LGH_STRIP_SET(strip,column,COLUMN);
	LGH_STRIP_SET(strip,buffer,BUFFER);
	LGH_STRIP_SET(strip,width,WIDTH);
	LGH_STRIP_SET(strip,opcode,OPCODE);

	address = lghComputeAddress(x, y, display);
	LGH_STRIP_SET(strip,address,ADDRESS);

	parity = lghComputeParity(strip);
	LGH_STRIP_SET(strip,parity,PARITY);

	return strip;
}

static WireGLL2StripHeader
wireGLL2StripHeaderCreateNull( int buffer, int width )
{
	WireGLL2StripHeader strip = 0;
	int parity;

	LGH_STRIP_SET(strip,LGH_NULL_COLUMN,COLUMN);
	LGH_STRIP_SET(strip,buffer,BUFFER);
	LGH_STRIP_SET(strip,width,WIDTH);

	parity = lghComputeParity(strip);
	LGH_STRIP_SET(strip,parity,PARITY);

	return strip;
}

char *
wireGLL2StripHeaderToString( char *string, WireGLL2StripHeader strip )
{
	int par, addr, buf, col, width, op, d, x, y;

	par   = (int) LGH_STRIP_GET(strip,PARITY);
	addr  = (int) LGH_STRIP_GET(strip,ADDRESS);
	buf   = (int) LGH_STRIP_GET(strip,BUFFER);
	col   = (int) LGH_STRIP_GET(strip,COLUMN);
	width = (int) LGH_STRIP_GET(strip,WIDTH);
	op    = (int) LGH_STRIP_GET(strip,OPCODE);

	d = 0;
	x = addr % 1024;
	y = addr / 1024;

	sprintf( string, "par=%d addr=%06x[d=%d x=%4d y=%3d] buf=%d col=%x "
			 "width=%03x op=%02x", par, addr, d, x, y, buf, col, width, op );

	return string;
}

#define LGH_FRAME_A  0x0
#define LGH_FRAME_B  0x1
#define LGH_FRAME_AB 0x2

static void 
DrawFrame( int buffer )
{
	WireGLL2StripHeader strip_a;
	WireGLL2StripHeader strip_b;
	int i;
	int width = wiregl_pipe.actual_window_width;
	int height = wiregl_pipe.actual_window_height;

	strip_a = wireGLL2StripHeaderCreateNull( 0x0, 2047 );
	strip_b = wireGLL2StripHeaderCreateNull( 0x1, 2047 );

	strip_a = wireGLL2StripHeaderSwap( strip_a );
	strip_b = wireGLL2StripHeaderSwap( strip_b );

	glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT );
	glDisable( GL_TEXTURE_2D );
	glDisable( GL_TEXTURE_1D );
	glDisable( GL_LIGHTING );
	glDisable( GL_BLEND );
	glDisable( GL_ALPHA_TEST );
	glDisable( GL_DEPTH_TEST );
	glDisable( GL_FOG );
	glDisable( GL_STENCIL_TEST );
	glDisable( GL_SCISSOR_TEST );
	glDisable( GL_LOGIC_OP );
	
	glViewport( 0, 0, width, height );
	glDrawBuffer( GL_BACK );
	glMatrixMode( GL_MODELVIEW );
	glPushMatrix( );
	glLoadIdentity( );
	
	glMatrixMode( GL_PROJECTION );
	glPushMatrix( );
	glLoadIdentity( );
	glOrtho( 0.0, width, 0.0, height, 0.0, 100.0 );
	glTranslatef( 0.375f, 0.375f, 0.0f );

	glClear(GL_COLOR_BUFFER_BIT);

	for ( i = 0; i < height; i++ )
	{
		WireGLL2StripHeader *strip;
		glRasterPos2i( 0, i ); 
		switch ( buffer ) 
		{
		  case 0:
			strip = &strip_a;
			break;
		  case 1:
			strip = &strip_b;
			break;
		  default:
			strip = ( i & 0x1 ) ? &strip_a : &strip_b;
			break;
		}
		glDrawPixels( 2, 1, GL_RGBA, GL_UNSIGNED_BYTE, strip );
	}

	glPopMatrix( );
	glMatrixMode( GL_MODELVIEW );
	glPopMatrix( );
	
	glPopAttrib( );		
}

static char
wireGLL2BackChannelDrain( void )
{
	int  n;
	char dummy, last;

	dummy = '\0';
	do {
		last = dummy;
		n = wireGLSerialRead( wiregl_pipe.L2.back_channel.port, &dummy, 1 );
	} while ( n != 0 );

	return last;
}

static double
wireGLGetTime( void )
{
#if defined(WINDOWS)
	int ticks = GetTickCount( );
	return 0.001 * (double) ticks;
#else
	struct timeval now;
	gettimeofday( &now, NULL );
	return (double) now.tv_sec + 1e-6 * (double) now.tv_usec;
#endif
}

static char
wireGLL2BackChannelReadChar( void )
{
	int n;
    char dummy = '\0';
	double start, elapsed;

	start = wireGLGetTime( );
	elapsed = 0.0;
	while ( elapsed < 1.0 )
	{
		n = wireGLSerialRead( wiregl_pipe.L2.back_channel.port, &dummy, 1 );
		if ( n > 0 )
		{
			return dummy;
		}
		elapsed = wireGLGetTime( ) - start;
    }

	wireGLWarning( WIREGL_WARN_NOTICE, "L2: Waiting for back channel" );

	do
	{
		n = wireGLSerialRead( wiregl_pipe.L2.back_channel.port, &dummy, 1 );
	}
	while ( n == 0 );

	return dummy;
}

static char
wireGLL2BackChannelReadCharWithTimeout( double max_seconds )
{
	int n;
    char dummy = '\0';
	double start, elapsed;

	start = wireGLGetTime( );

	elapsed = 0.0;
	while ( elapsed < max_seconds )
	{
		n = wireGLSerialRead( wiregl_pipe.L2.back_channel.port, &dummy, 1 );
		if ( n > 0 )
		{
			return dummy;
		}
		elapsed = wireGLGetTime( ) - start;
    }

	wireGLSimpleError( "L2: waited %.2f seconds for something on the "
					   "backchannel, but nothing arrived", elapsed );
	return dummy;
}

void
wireGLL2Wait( void )
{
	(void) wireGLL2BackChannelReadChar( );
}

void 
wireGLL2Sync( int phase )
{
	static int phase_prev = -1;
    static char last_char_read;
	int  dummy;

	wireGLWarning( WIREGL_WARN_DEBUG, "SyncL2: phase=%d", phase );
	
	if ( phase != phase_prev + 1 )
	{
		wireGLWarning( WIREGL_WARN_CRITICAL, "SyncL2: expected phase %d, not "
					   "phase %d", phase_prev + 1, phase );
		return;
	}

	phase_prev = phase;
		
	switch ( phase )
	{
	  case 0:
		/* open the back channel serial port */
		wiregl_pipe.L2.back_channel.port = 
			wireGLSerialOpen( wiregl_pipe.L2.back_channel.name,
							  WIREGL_SERIAL_READ,
							  wiregl_pipe.L2.back_channel.baud );

        /* Purge any characters from the local back-channel serial
         * port buffer. */
		(void) wireGLL2BackChannelDrain( );

        /* Draw a frame with alternating A & B strip headers into the
         * back buffer. */
        DrawFrame( LGH_FRAME_AB );

	    /* Swap the buffers. */
		wireGLDeviceSwapBuffers( );

        /* Draw a frame with only B strip headers into the back
		 * buffer. */
        DrawFrame( LGH_FRAME_B );

        /* Spin waiting for a character to arrive over the
         * back-channel. */
		last_char_read = wireGLL2BackChannelReadCharWithTimeout( 5.0 );

        /* Synchronize with all other rendering nodes in the same swap
         * set.  This ensures that the system will continue to swap
         * frantically (due to the alternating A & B strip headers)
         * until every node has received at least one character on the
         * back-channel. */
		break;

	  case 1:

		/* Swap the buffers. */
		wireGLDeviceSwapBuffers( );

        /* Synchronize with all other rendering nodes in the same swap
		 * set.  This ensures that every node is now displaying a
		 * frame with only * B strip headers. */
		break;

	  case 2:

        /* Draw a frame with only A strip headers into the back
         * buffer. */
        DrawFrame(LGH_FRAME_A);

        /* Wait until enough time has passed for Lightning-2 to have
         * done anything it's going to do with what it's currently
         * seeing (all B strip headers) and send back any back-channel
         * pings it's going to send.  Something like two
         * refresh-cycle-times should be plenty for this. */
        wireGLSleep( 500 );

        /* Make sure that last_char_read holds the last character to
         * arrive over the back-channel. */
		dummy = wireGLL2BackChannelDrain( );
		if ( dummy != '\0' )
		{
			last_char_read = (char) dummy;
		}

		wireGLWarning( WIREGL_WARN_DEBUG, "SyncL2: back channel says '%c' "
					   "(0x%02x)", 
					   ( isalpha(last_char_read) ? last_char_read : '.' ),
					   last_char_read );

        /* stupid gordon tricks */
        if ( last_char_read == 'B' )
		{
			wireGLDeviceSwapBuffers( );
		}
		break;

	  case 3:
        if ( last_char_read == 'A' )
		{
			wireGLDeviceSwapBuffers( );
			(void) wireGLL2BackChannelReadChar( );
        }
        break;

	  case 4:
        DrawFrame( LGH_FRAME_B );
		wireGLDeviceSwapBuffers( );

        /* At this point, the nodes should be synced up and ready for
         * the user application to do its thing.  The user application
         * should draw an 'A' frame into the back buffer, call the
         * 'lghWaitForNotification' call, swap the buffers, draw a 'B'
         * frame, wait, swap, etc... */
		wiregl_pipe.L2.frame_num = 0;
		break;
	}

	glFinish( );
}

static void
wireGLL2FrameInit( WireGLL2Frame *frame )
{
    frame->tex_id      = 0;
    frame->tex_w       = 0;
    frame->tex_h       = 0;
    frame->tex_data    = NULL;

    frame->num_extents = 0;
    frame->max_extents = 16;
    frame->extent      = wireGLAlloc( sizeof(frame->extent[0]) *
                                      frame->max_extents );

    frame->num_headers = 0;
    frame->max_headers = 16;
    frame->header      = wireGLAlloc( sizeof(frame->header[0]) *
                                      frame->max_headers );
}

static void
wireGLL2FrameAddHeader( WireGLL2Frame *frame, int x, int y, 
						WireGLL2StripHeader strip )
{
    if ( frame->num_headers == frame->max_headers )
	{
        frame->max_headers <<= 1;
        wireGLRealloc( (void **) &frame->header,
                       sizeof(frame->header[0]) * frame->max_headers );
    }

	frame->header[frame->num_headers].win_x = x;
	frame->header[frame->num_headers].win_y = y;
    frame->header[frame->num_headers].strip = strip;
    frame->num_headers++;
}

static int
wireGLL2CompareHeaders( const void *a, const void *b )
{
    const WireGLL2Header *hdr_a = (const WireGLL2Header *) a;
    const WireGLL2Header *hdr_b = (const WireGLL2Header *) b;

    if ( hdr_a->win_x < hdr_b->win_x )
        return -1;
    if ( hdr_a->win_x > hdr_b->win_x )
        return 1;
    if ( hdr_a->win_y < hdr_b->win_y )
        return -1;
    if ( hdr_a->win_y > hdr_b->win_y )
        return 1;

    wireGLError( "wireGLL2CompareHeaders: headers at same x,y?" );
    return 0;
}

/* XXX bad hack */
static int bogus_tex_id = 2047;

static void
wireGLL2FramePack( WireGLL2Frame *frame )
{
	unsigned int i, tex_w, tex_h, tex_x, tex_y;
	WireGLL2Extent *extent;

    /* sort the headers */
    qsort( frame->header, frame->num_headers, sizeof(frame->header[0]),
           wireGLL2CompareHeaders );

    /* how big does the texture need to be? */
    tex_w = 1;
    while ( tex_w * tex_w < frame->num_headers * 2 )
	{
        tex_w <<= 1;
    }
    tex_h = 1;
    while ( tex_w * tex_h < frame->num_headers * 2 )
	{
        tex_h <<= 1;
    }

    frame->tex_w    = tex_w;
    frame->tex_h    = tex_h;
    frame->tex_data = (unsigned char *) wireGLAlloc( tex_w * tex_h * 3 );

    /* layout the headers */
    tex_x  = 0;
    tex_y  = 0;
    extent = NULL;
    for ( i = 0; i < frame->num_headers; i++ ) 
	{
        WireGLL2Header *header = &frame->header[i];
		unsigned char *src, *dst;

        if ( tex_y == tex_h ) 
		{
            tex_y = 0;
            tex_x += 2;
            wireGLAssert( tex_x < tex_w );
            extent = NULL;
        }
        else if ( extent == NULL ||
                  header->win_x != extent->win_x1 ||
                  header->win_y != extent->win_y2 ) 
		{
            extent = NULL;
        }

        if ( extent == NULL ) 
		{
            if ( frame->num_extents == frame->max_extents ) 
			{
                frame->max_extents <<= 1;
                wireGLRealloc( (void **) &frame->extent,
                               sizeof(frame->extent[0]) * frame->max_extents );
            }
            extent = &frame->extent[ frame->num_extents ];
            frame->num_extents++;

            extent->win_x1 = header->win_x;
            extent->win_y1 = header->win_y;
            extent->win_x2 = extent->win_x1 + 2;
            extent->win_y2 = extent->win_y1;
            extent->tex_x1 = (float) tex_x / (float) tex_w;
            extent->tex_y1 = (float) tex_y / (float) tex_h;
        }

        /* jam in the data here */
		src = (unsigned char *) &header->strip;
		dst = frame->tex_data + 3 * ( ( tex_y * tex_w ) + tex_x );
		dst[0] = src[0];
		dst[1] = src[1];
		dst[2] = src[2];
		dst[3] = src[4];
		dst[4] = src[5];
		dst[5] = src[6];

        extent->win_y2 += 1;
        tex_y++;
    }

    for ( i = 0; i < frame->num_extents; i++ )
	{
        extent = &frame->extent[i];

        extent->tex_x2 = extent->tex_x1 + 2.0f / (float) tex_w;
        extent->tex_y2 = extent->tex_y1 +
            (float) ( extent->win_y2 - extent->win_y1 ) / (float) tex_h;
    }

    wireGLWarning( WIREGL_WARN_DEBUG, "L2: frame texture is %ux%u, %u extents",
                   tex_w, tex_h, frame->num_extents );


    /* bind the texture */
	glPushAttrib( GL_TEXTURE_BIT );
    frame->tex_id = bogus_tex_id++;
    glBindTexture( GL_TEXTURE_2D, frame->tex_id );
    glTexImage2D( GL_TEXTURE_2D, 0, GL_RGB8, frame->tex_w, frame->tex_h,
                  0, GL_RGB, GL_UNSIGNED_BYTE, frame->tex_data );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
	glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
	glPopAttrib( );

#if 0
    /* pitch these, we don't need them anymore */
    wireGLFree( frame->header );
    frame->header = NULL;

    wireGLFree( frame->tex_data );
    frame->tex_data = NULL;
#endif
}

typedef struct WireGLL2Run {
	int x;
	int len;
	struct WireGLL2Run *next;
} WireGLL2Run;

static void
wireGLL2RunRemove( WireGLL2Run **head, int x, int len )
{
	WireGLL2Run *run;

	run  = *head;
	while ( run && ( run->x + run->len <= x ) )
	{
		head = &run->next;
		run = *head;
	}

	if ( run == NULL || x < run->x || x + len > run->x + run->len )

	{
		wireGLError( "wireGLRunRemove( x=%d, len=%d ): no run intersection",
					 x, len );
	}

	if ( x == run->x )
	{
		/* the starts abut, so move the start up */
		run->x += len;
		run->len -= len;

		if ( len == 0 )
		{
			*head = run->next;
		}
	}
	else if ( x + len == run->x + run->len )
	{
		/* the ends abut, so move the end down */
		run->len = len;
	}
	else
	{
		/* the piece we are removing is in the middle of this run */
		WireGLL2Run *temp = (WireGLL2Run *) wireGLAlloc( sizeof(*temp) );

		/* the part after the piece we are removing */
		temp->x    = x + len;
		temp->len  = ( run->x + run->len ) - temp->x;
		temp->next = run->next;

		/* the part before the piece we are removing */
		run->len  = x - run->x;
		run->next = temp;
	}
}

static WireGLL2Run **
wireGLL2BuildRunArray( int width, int height )
{
	WireGLL2Run **runs = (WireGLL2Run **) 
		wireGLAlloc( height * sizeof(*runs) );
	int y;

	for ( y = 0; y < height; y++ )
	{
		WireGLL2Run *temp = (WireGLL2Run *) wireGLAlloc( sizeof(*temp) );
		temp->x    = 0;
		temp->len  = width;
		temp->next = NULL;
		runs[y] = temp;
	}

	return runs;
}

void
wireGLL2MakeFrame( WireGLL2Frame *frame, int l2_buffer, 
					struct WireGLWorkQueue *q )
{
	WireGLL2Run **runs;
	int num_extents, i, y;
	WireGLWorkQueueExtent *extent;
	WireGLL2StripHeader strip;
	int save_buffer = l2_buffer;
	
	l2_buffer &= 0x1;

	wireGLL2FrameInit( frame );

	runs = wireGLL2BuildRunArray( wiregl_pipe.actual_window_width,
								  wiregl_pipe.actual_window_height );

	num_extents = q->num_extents;
	extent = q->extent;
	for ( i = 0; i < num_extents; i++, extent++ )
	{
		WireGLPipeDisplay *display = &wiregl_pipe.display[ extent->display ];

		int extent_h = extent->imagewindow.y2 - extent->imagewindow.y1;
		int extent_w = extent->imagewindow.x2 - extent->imagewindow.x1;
		for ( y = 0; y < extent_h; y++ )
		{
			int l2_x, l2_y, win_x, win_y;

			l2_x = extent->imagewindow.x1 - display->x;
			l2_y = display->h-1 - ( extent->imagewindow.y1 + y - display->y );

			strip = wireGLL2StripHeaderCreate( display->channel, l2_buffer,
											   display->port, 
											   l2_x, l2_y, extent_w, 0 );

			win_x = extent->outputwindow.x1 - 2;
			win_y = extent->outputwindow.y1 + y;

			wireGLAssert( win_x >= 0 && 
						  win_x < wiregl_pipe.actual_window_width );
			wireGLAssert( win_y >= 0 && 
						  win_y < wiregl_pipe.actual_window_height );
				
			wireGLL2FrameAddHeader( frame, win_x, win_y, strip );

			wireGLL2RunRemove( &runs[win_y], win_x, extent_w + 2 );

		}
	}

	for ( y = 0; y < wiregl_pipe.actual_window_height; y++ )
	{
		WireGLL2Run *run;
		for ( run = runs[y]; run != NULL; run = run->next )
		{
			wireGLAssert( run->len >= 2 );
			strip = wireGLL2StripHeaderCreateNull( l2_buffer, run->len - 2 );
#if 0
			/* HACK */
			strip = wireGLL2StripHeaderCreateNull( l2_buffer, 2047 );
#endif
			wireGLL2FrameAddHeader( frame, run->x, y, strip );
		}
	}

	if ( save_buffer == WIREGL_L2_SINGLE_BUFFER )
	{
		WireGLL2Header *header = frame->header;
		wireGLWarning( WIREGL_WARN_DEBUG, "L2: single buffer" );
		for ( i = 0; i < (int) frame->num_headers; i++, header++ )
		{
			int a_or_b, parity;

			a_or_b = ( header->win_y ) & 0x1;
			LGH_STRIP_SET( header->strip, a_or_b, BUFFER );

			parity = lghComputeParity( header->strip );
			LGH_STRIP_SET( header->strip, parity, PARITY );
		}
	}

	{
		WireGLL2Header *header = frame->header;
		for ( i = 0; i < (int) frame->num_headers; i++, header++ )
		{
			header->strip = wireGLL2StripHeaderSwap( header->strip );
		}
	}

	wireGLL2FramePack( frame );
}

#if USE_DRAWPIXELS

void
wireGLL2DrawFrame( WireGLL2Frame *frame ) 
{
	int width = wiregl_pipe.actual_window_width;
	int height = wiregl_pipe.actual_window_height;
	WireGLL2Header *header;
	unsigned int num_headers, i;

	glPushAttrib( GL_CURRENT_BIT | GL_ENABLE_BIT );
	glDisable( GL_TEXTURE_2D );
	glDisable( GL_TEXTURE_1D );
	glDisable( GL_LIGHTING );
	glDisable( GL_BLEND );
	glDisable( GL_ALPHA_TEST );
	glDisable( GL_DEPTH_TEST );
	glDisable( GL_FOG );
	glDisable( GL_STENCIL_TEST );
	glDisable( GL_SCISSOR_TEST );
	glDisable( GL_LOGIC_OP );

	/* HACK */
	glDisable( GL_CLIP_PLANE0 );
	glDisable( GL_CLIP_PLANE1 );
	glDisable( GL_CLIP_PLANE2 );
	glDisable( GL_CLIP_PLANE3 );
	glDisable( GL_CLIP_PLANE4 );
	glDisable( GL_CLIP_PLANE5 );
	glDisable( GL_COLOR_LOGIC_OP );
	
	glViewport( 0, 0, width, height );
	glDrawBuffer( GL_BACK );

	glMatrixMode( GL_MODELVIEW );
	glPushMatrix( );
	glLoadIdentity( );
	
	glMatrixMode( GL_PROJECTION );
	glPushMatrix( );
	glLoadIdentity( );
	glOrtho( 0.0, width, 0.0, height, -1.0, 1.0 );
	glTranslatef( 0.375f, 0.375f, 0.0f );
	
	header = frame->header;
	num_headers = frame->num_headers;
	for ( i = 0; i < num_headers; i++, header++ ) 
	{
		glRasterPos2i( header->win_x, header->win_y );
		glDrawPixels( 2, 1, GL_RGBA, GL_UNSIGNED_BYTE, &header->strip );
	}

	glPopMatrix( );

	glMatrixMode( GL_MODELVIEW );
	glPopMatrix( );
	
	glPopAttrib( );		
}

#else

void
wireGLL2DrawFrame( WireGLL2Frame *frame )
{
	unsigned int i, num_extents;
	WireGLL2Extent *extent;
	int width = wiregl_pipe.actual_window_width;
	int height = wiregl_pipe.actual_window_height;

    glPushAttrib( GL_CURRENT_BIT | GL_TEXTURE_BIT | GL_ENABLE_BIT );
    glDisable( GL_TEXTURE_1D   );
    glDisable( GL_LIGHTING     );
    glDisable( GL_BLEND        );
    glDisable( GL_ALPHA_TEST   );
    glDisable( GL_DEPTH_TEST   );
    glDisable( GL_FOG          );
    glDisable( GL_STENCIL_TEST );
    glDisable( GL_SCISSOR_TEST );
    glDisable( GL_LOGIC_OP     );

	/* HACK */
	glDisable( GL_CLIP_PLANE0 );
	glDisable( GL_CLIP_PLANE1 );
	glDisable( GL_CLIP_PLANE2 );
	glDisable( GL_CLIP_PLANE3 );
	glDisable( GL_CLIP_PLANE4 );
	glDisable( GL_CLIP_PLANE5 );
	glDisable( GL_COLOR_LOGIC_OP );
	glDisable( GL_CULL_FACE );
	glDisable( GL_DITHER );
	glDisable( GL_POLYGON_SMOOTH );
	glDisable( GL_POLYGON_STIPPLE );
	glDisable( GL_TEXTURE_GEN_Q );
	glDisable( GL_TEXTURE_GEN_R );
	glDisable( GL_TEXTURE_GEN_S );
	glDisable( GL_TEXTURE_GEN_T );

    glBindTexture( GL_TEXTURE_2D, frame->tex_id );
    glEnable( GL_TEXTURE_2D );
	glTexEnvi( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_DECAL );
	/* NOTE -- other pieces of the texture environment might need to
	 * be set here */

    glViewport( 0, 0, width, height );
    glDrawBuffer( GL_BACK );

    glMatrixMode( GL_MODELVIEW );
    glPushMatrix( );
    glLoadIdentity( );

    glMatrixMode( GL_PROJECTION );
    glPushMatrix( );
    glLoadIdentity( );
    glOrtho( 0.0, width, 0.0, height, -1.0, 1.0 );
    glTranslatef( 0.375f, 0.375f, 0.0f );

    glBegin( GL_QUADS );

    num_extents = frame->num_extents;
    extent = frame->extent;
    for ( i = 0; i < num_extents; i++, extent++ )
	{
        glTexCoord2f( extent->tex_x1, extent->tex_y1 );
        glVertex2i(   extent->win_x1, extent->win_y1 );

        glTexCoord2f( extent->tex_x2, extent->tex_y1 );
        glVertex2i(   extent->win_x2, extent->win_y1 );

        glTexCoord2f( extent->tex_x2, extent->tex_y2 );
        glVertex2i(   extent->win_x2, extent->win_y2 );

        glTexCoord2f( extent->tex_x1, extent->tex_y2 );
        glVertex2i(   extent->win_x1, extent->win_y2 );
    }

    glEnd( );

    glPopMatrix( );

    glMatrixMode( GL_MODELVIEW );
    glPopMatrix( );

    glPopAttrib( );
}

#endif
