/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_util.h"
#include "wiregl/glcontext/glcontext.h"

/* FYI: Including .c files is crap. */
#include "wiregl/glcontext/glconvert_gen.c"
#include "length_table.c"

#define vmin(a,b) ((a)<(b)?(a):(b))

/* Ok.  So these might be major hacks 
** but the alternative is a huge switch statement
** and our protocol generator sorts the opcodes,
** so why not just do greater than less than?
*/
#define IS_VERTEX(a)	((a) >= WIREGL_VERTEX2D_OPCODE && (a) <= WIREGL_VERTEX4S_OPCODE)
#define IS_COLOR(a)		((a) >= WIREGL_COLOR3B_OPCODE && (a) <= WIREGL_COLOR4US_OPCODE)
#define IS_NORMAL(a)	((a) >= WIREGL_NORMAL3B_OPCODE && (a) <= WIREGL_NORMAL3S_OPCODE)
#define IS_INDEX(a)		((a) >= WIREGL_INDEXD_OPCODE && (a) <= WIREGL_INDEXS_OPCODE)
#define IS_TEXCOORD(a)	((a) >= WIREGL_TEXCOORD1D_OPCODE && (a) <= WIREGL_TEXCOORD4S_OPCODE)
#define IS_EDGEFLAG(a)	((a) == WIREGL_EDGEFLAG_OPCODE)

#define ASSERT_BOUNDS(op, data) \
				wireGLAssert(op <= __wiregl_globals.buffer.opcode_start); \
				wireGLAssert(op > __wiregl_globals.buffer.opcode_end); \
				wireGLAssert(data >= __wiregl_globals.buffer.data_start); \
				wireGLAssert(data < __wiregl_globals.buffer.data_end)

static const GLvectorf vdefault = {0.0f, 0.0f, 0.0f, 1.0f};

void
wireGLVpinch (GLcontext *g) {
	GLcurrentstate *c = &(g->current);
	int vtx_count = c->current->vtx_count - c->current->vtx_count_begin;
	int num_restore;
	int loop = 0;
	int wind = 0;
	int i;
	unsigned char * op;
	unsigned char * data;
	unsigned char * vtx_op;
	unsigned char * vtx_data;

	unsigned char *color_ptr = __wiregl_globals.current.color.ptr;
	unsigned char *normal_ptr = __wiregl_globals.current.normal.ptr;
	unsigned char *texcoord_ptr = __wiregl_globals.current.texcoord.ptr;
	unsigned char *edgeflag_ptr = __wiregl_globals.current.edgeflag.ptr;
	// unsigned char *index_ptr = __wiregl_globals.current.index.ptr;

	GLvertex v_current;

	v_current.pos = vdefault;
	v_current.color = c->color;
	v_current.normal = c->normal;
	v_current.texcoord = c->texcoord;
	v_current.edgeflag = c->edgeflag;
	v_current.index = c->index;

	/* First lets figure out how many vertexes
	** we need to recover
	*/
	if (!c->beginend || vtx_count == 0) {
		c->num_restore = 0;
		return;
	}
	switch (c->mode) {
	case GL_POINTS:
		num_restore = 0;
		break;
	case GL_LINES:
		num_restore = vtx_count % 2;
		break;
	case GL_LINE_STRIP:
		/* A seperate flag is used to inidcate
		** that this is really a line loop that 
		** we're sending down as a line strip.
		** If so, we need to recover the first 
		** vertex and the last one.
		*/
		if (c->isloop) {
			num_restore = 2;
			loop = 1;
			break;
		}
		num_restore = 1;
		break;
	case GL_LINE_LOOP:
		num_restore = vmin(vtx_count,2);
		/* Only convert this into a 
		** line strip if we've issued 
		** more than two vertexes.
		*/
		loop = vtx_count>2?1:0;
		break;
	case GL_TRIANGLES:
		num_restore = vtx_count % 3;
		break;
	case GL_TRIANGLE_STRIP:
		num_restore = vmin(vtx_count,2);
		/* Check the winding. If we're issuing 
		** the odd vertex, the winding of the 
		** triangle changes. We need to record
		** this for restoring the strip.
		*/
		wind = (vtx_count > 2)?vtx_count%2:0;
		break;
	/* Note to self: GL_TRIANGLE_FAN <=> GL_POLYGON 
	** if you ignore the vertex params
	*/
	case GL_TRIANGLE_FAN:
	case GL_POLYGON:
		num_restore = vmin(vtx_count,2);
		loop = 1;
		break;
	case GL_QUADS:
		num_restore = vtx_count % 4;
		break;
	case GL_QUAD_STRIP:
		/* Don't care about winding here 
		** since quads consist of even 
		** num of triangles.
		*/
		if (vtx_count < 4)
			num_restore = vtx_count;
		else 
			num_restore = 2 + (vtx_count%2);
		break;
	default:
		wireGLSimpleError( "Unkown mode: %d", c->mode );
		return;
	}

	vtx_op = __wiregl_globals.buffer.opcode_current;
	vtx_data = __wiregl_globals.buffer.data_current;

	for (i=num_restore; i>0 ;i--) {
		GLvertex *vtx = c->vtx + i - 1;
		
		op = vtx_op;
		data= vtx_data;

		/* If we're restoring a line strip
		** that is really a line loop, the 
		** first vertex is still sitting in
		** the first vtx.  Lets quit early
		** and preserve that value.  It will
		** be issued at glEnd.
		*/
		if (i==1 && c->isloop) break;

		/* Search for a vertex command */
		
		/* If we're dealing with a loop (polygon, fan, lineloop),
		** the first vertex comes is found by searching forward
		** from the glBegin.  The other vertexes come from the end
		** moving backward.  Note we fill the restored vtx list
		** starting from the end.
		*/
		if (i==1 && loop) {
			op = __wiregl_globals.current.begin_op;
			data = __wiregl_globals.current.begin_data;

			/* Perform the line loop to line strip 
			** conversion. Set the isloop flag so 
			** we don't forget 
			*/
			if (c->mode == GL_LINE_LOOP) {
				c->isloop = GL_TRUE;
				c->mode = GL_LINE_STRIP;
				*((GLenum *)data) = GL_LINE_STRIP;
			}
				
			do {
				data += __wiregl_packet_length_table[*op];
				op--;
				ASSERT_BOUNDS(op, data);
			} while (!IS_VERTEX(*op));
		} else { 
			do {
				op++;
				data -= __wiregl_packet_length_table[*op];
				ASSERT_BOUNDS(op, data);
			} while (!IS_VERTEX(*op));
		}
		

		/* Found a vertex */
		vtx_op = op;
		vtx_data = data;

		/* Lets search for the parameters */
		/* The following code should be auto-generated but oh well... */
		if (vtx_op < __wiregl_globals.buffer.opcode_start) {
			/* Is the color pointer after my vertex? */
			if (color_ptr > vtx_data) {
				
				/* Perform the search */
				op = vtx_op+1;
				data = vtx_data - __wiregl_packet_length_table[*(vtx_op+1)];
				while (op <= __wiregl_globals.buffer.opcode_start && !IS_COLOR(*op)) {
					op++;
					data -= __wiregl_packet_length_table[*op];
				}

				/* Did I hit the begining of the buffer? */
				if (op > __wiregl_globals.buffer.opcode_start) {
					v_current.color = c->color_pre;
					color_ptr = NULL;
				} else {
					ASSERT_BOUNDS (op, data);
					VPINCH_CONVERT_COLOR (*op, data, v_current.color);
					color_ptr = data;
				}

			}

			/* Is the normal pointer after my vertex? */
			if (normal_ptr > vtx_data) {
				
				/* Perform the search */
				op = vtx_op+1;
				data = vtx_data - __wiregl_packet_length_table[*(vtx_op+1)];
				while (op <= __wiregl_globals.buffer.opcode_start && !IS_NORMAL(*op)) {
					op++;
					data -= __wiregl_packet_length_table[*op];
				}

				/* Did I hit the begining of the buffer? */
				if (op > __wiregl_globals.buffer.opcode_start) {
					v_current.normal = c->normal_pre;
					normal_ptr = NULL;
				} else {
					ASSERT_BOUNDS (op, data);
					VPINCH_CONVERT_NORMAL (*op, data, v_current.normal)
					normal_ptr = data;
				}
			}

			/* Is the texture pointer after my vertex? */
			if (texcoord_ptr > vtx_data) {
				
				/* Perform the search */
				op = vtx_op+1;
				data = vtx_data - __wiregl_packet_length_table[*(vtx_op+1)];
				while (op <= __wiregl_globals.buffer.opcode_start && !IS_TEXCOORD(*op)) {
					op++;
					data -= __wiregl_packet_length_table[*op];
				}

				/* Did I hit the begining of the buffer? */
				if (op > __wiregl_globals.buffer.opcode_start) {
					v_current.texcoord = c->texcoord_pre;
					texcoord_ptr = NULL;
				} else {
					ASSERT_BOUNDS (op, data);
					VPINCH_CONVERT_TEXCOORD (*op, data, v_current.texcoord)
					texcoord_ptr = data;
				}
			}

			/* Is the stupid edgeflag pointer after my vertex? */
			if (edgeflag_ptr > vtx_data) {
				
				/* Perform the search */
				op = vtx_op+1;
				data = vtx_data - __wiregl_packet_length_table[*(vtx_op+1)];
				while (op <= __wiregl_globals.buffer.opcode_start && !IS_EDGEFLAG(*op)) {
					op++;
					data -= __wiregl_packet_length_table[*op];
				}

				/* Did I hit the begining of the buffer? */
				if (op > __wiregl_globals.buffer.opcode_start) {
					v_current.edgeflag = c->edgeflag_pre;
					edgeflag_ptr = NULL;
				} else {
					ASSERT_BOUNDS (op, data);
					VPINCH_CONVERT_EDGEFLAG (*op, data, v_current.edgeflag)
					edgeflag_ptr = data;
				}
			}
		} else {
			v_current.color = c->color_pre;
			v_current.normal = c->normal_pre;
			v_current.texcoord = c->texcoord_pre;
			v_current.edgeflag = c->edgeflag_pre;
		}

		/* Copy current values */
		*vtx = v_current;

		/* Extract the position */
		switch (*vtx_op) {
		case WIREGL_VERTEX2D_OPCODE:
			__convert_d2(&(vtx->pos.x), (GLdouble *) vtx_data);
			break;
		case WIREGL_VERTEX2F_OPCODE:
			__convert_f2(&(vtx->pos.x), (GLfloat *) vtx_data);
			break;
		case WIREGL_VERTEX2I_OPCODE:
			__convert_i2(&(vtx->pos.x), (GLint *) vtx_data);
			break;
		case WIREGL_VERTEX2S_OPCODE:
			__convert_s2(&(vtx->pos.x), (GLshort *) vtx_data);
			break;
		case WIREGL_VERTEX3D_OPCODE:
			__convert_d3(&(vtx->pos.x), (GLdouble *) vtx_data);
			break;
		case WIREGL_VERTEX3F_OPCODE:
			__convert_f3(&(vtx->pos.x), (GLfloat *) vtx_data);
			break;
		case WIREGL_VERTEX3I_OPCODE:
			__convert_i3(&(vtx->pos.x), (GLint *) vtx_data);
			break;
		case WIREGL_VERTEX3S_OPCODE:
			__convert_s3(&(vtx->pos.x), (GLshort *) vtx_data);
			break;
		case WIREGL_VERTEX4D_OPCODE:
			__convert_d4(&(vtx->pos.x), (GLdouble *) vtx_data);
			break;
		case WIREGL_VERTEX4F_OPCODE:
			__convert_f4(&(vtx->pos.x), (GLfloat *) vtx_data);
			break;
		case WIREGL_VERTEX4I_OPCODE:
			__convert_i4(&(vtx->pos.x), (GLint *) vtx_data);
			break;
		case WIREGL_VERTEX4S_OPCODE:
			__convert_s4(&(vtx->pos.x), (GLshort *) vtx_data);
			break;
		default:
			wireGLAbort();
		}


	}

	/* record the number of vtx to restore
	** and the winding info before we quit 
	*/
	c->num_restore = num_restore;
	c->wind = wind;
}
