/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_util.h"
#include "wiregl/glcontext/glcontext.h"
#include "packer_internals.h"

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexImage1D  )(GLenum target, GLint level, 
		GLint internalformat, GLsizei width, GLint border, GLenum format, 
		GLenum type, const GLvoid *pixels)
{
	unsigned char *data_ptr;
	int packet_length;
	int texturesize;

	packet_length = 
		sizeof( target ) +
		sizeof( level ) +
		sizeof( internalformat ) +
		sizeof( width ) + 
		sizeof( border ) +
		sizeof( format ) +
		sizeof( type ) +
		sizeof( int );

	texturesize =__gltexture_getcurrenttexturesize(target, level);
	packet_length += texturesize;

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );
	WRITE_DATA( 0, GLenum, target );
	WRITE_DATA( 4, GLint, level );
	WRITE_DATA( 8, GLint, internalformat );
	WRITE_DATA( 12, GLsizei, width );
	WRITE_DATA( 16, GLint, border );
	WRITE_DATA( 20, GLenum, format );
	WRITE_DATA( 24, GLenum, type );
	WRITE_DATA( 28, int, (pixels == NULL) );

	if (pixels) {
		memcpy((void *)(data_ptr + 32), 
		(const void *) __gltexture_getcurrenttexturedata(target, level),
		texturesize);
	}

	wireGLSendPacket( WIREGL_TEXIMAGE1D_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}


void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexImage2D  )(GLenum target, GLint level, 
		GLint internalformat, GLsizei width, GLsizei height, GLint border, 
		GLenum format, GLenum type, const GLvoid *pixels)
{
	unsigned char *data_ptr;
	int packet_length;
	int texturesize;
	int distributed = (type == GL_FALSE || type == GL_TRUE);

	packet_length = 
		sizeof( target ) +
		sizeof( level ) +
		sizeof( internalformat ) +
		sizeof( width ) +
		sizeof( height ) + 
		sizeof( border ) +
		sizeof( format ) +
		sizeof( type ) +
		sizeof( int );


	if (distributed) {
	    if (type == GL_FALSE)
		texturesize = strlen((char *) pixels) + 1;
	    else /* type == GL_TRUE */
		texturesize = width*height*3 + strlen((char *) pixels) + 1;
	} else {
	    texturesize =__gltexture_getcurrenttexturesize(target, level);
	}


	packet_length += texturesize;

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );
	WRITE_DATA( 0, GLenum, target );
	WRITE_DATA( 4, GLint, level );
	WRITE_DATA( 8, GLint, internalformat );
	WRITE_DATA( 12, GLsizei, width );
	WRITE_DATA( 16, GLsizei, height );
	WRITE_DATA( 20, GLint, border );
	WRITE_DATA( 24, GLenum, format );
	WRITE_DATA( 28, GLenum, type );
	WRITE_DATA( 32, int, (pixels == NULL) );

	if (pixels) {
	    if (distributed)
		memcpy((void *)(data_ptr + 36), pixels, texturesize);
	    else
		memcpy((void *)(data_ptr + 36), 
		    (const void *) __gltexture_getcurrenttexturedata(target, level),
		    texturesize);
	}

	wireGLSendPacket( WIREGL_TEXIMAGE2D_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( DeleteTextures  )( GLsizei n, const GLuint *textures )
{
	unsigned char *data_ptr;
	int i;
	int packet_length = 
		sizeof( n ) + 
		n*sizeof( *textures );

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );

	WRITE_DATA( 0, GLsizei, n );
	memcpy( data_ptr + 4, textures, n*sizeof(*textures) );

	for (i=0; i<n; i++)
		__gltexture_deletetexture(textures[i]);

	wireGLSendPacket( WIREGL_DELETETEXTURES_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}

static void __handleTexEnvData( GLenum target, GLenum pname, const GLfloat *params )
{
	unsigned char *data_ptr;
	int params_length;

	int packet_length = 
		sizeof( int ) + 
		sizeof( target ) + 
		sizeof( pname );

	if ( pname == GL_TEXTURE_ENV_COLOR )
	{
		params_length = 4*sizeof( *params );
	}
	else
	{
		params_length = sizeof( *params );
	}

	packet_length += params_length;

	GET_BUFFERED_POINTER( packet_length );
	WRITE_DATA( 0, int, packet_length );
	WRITE_DATA( sizeof( int ) + 0, GLenum, target );
	WRITE_DATA( sizeof( int ) + 4, GLenum, pname );
	memcpy( data_ptr + sizeof( int ) + 8, params, params_length );
}


void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexEnvfv  )( GLenum target, GLenum pname,
		const GLfloat *params )
{
	__handleTexEnvData( target, pname, params );
	WRITE_OPCODE( WIREGL_TEXENVFV_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexEnviv  )( GLenum target, GLenum pname,
		const GLint *params )
{
	/* floats and ints are the same size, so the packing should be the same */
	__handleTexEnvData( target, pname, (const GLfloat *) params );
	WRITE_OPCODE( WIREGL_TEXENVIV_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( PrioritizeTextures  )( GLsizei n,
		const GLuint *textures, const GLclampf *priorities )
{
	unsigned char *data_ptr;
	int packet_length = 
		sizeof( n ) + 
		n*sizeof( *textures ) + 
		n*sizeof( *priorities );

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );

	WRITE_DATA( 0, GLsizei, n );
	memcpy( data_ptr + 4, textures, n*sizeof( *textures ) );
	memcpy( data_ptr + 4 + n*sizeof( *textures ),
			priorities, n*sizeof( *priorities ) );

	wireGLSendPacket( WIREGL_PRIORITIZETEXTURES_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}

static void __handleTexGenData( GLenum coord, GLenum pname, 
		int sizeof_param, const GLvoid *params )
{
	unsigned char *data_ptr;
	int packet_length = sizeof( int ) + sizeof( coord ) + sizeof( pname ) + sizeof_param;
	int params_length = sizeof_param;
	if (pname == GL_OBJECT_PLANE || pname == GL_EYE_PLANE)
	{
		packet_length += 3*sizeof_param;
		params_length += 3*sizeof_param;
	}
	
	GET_BUFFERED_POINTER( packet_length );
	WRITE_DATA( 0, int, packet_length );
	WRITE_DATA( sizeof( int ) + 0, GLenum, coord );
	WRITE_DATA( sizeof( int ) + 4, GLenum, pname );
	memcpy( data_ptr + sizeof( int ) + 8, params, params_length );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexGendv  )( GLenum coord, GLenum pname,
		const GLdouble *params )
{
	__handleTexGenData( coord, pname, sizeof( *params ), params );
	WRITE_OPCODE( WIREGL_TEXGENDV_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexGenfv  )( GLenum coord, GLenum pname,
		const GLfloat *params )
{
	__handleTexGenData( coord, pname, sizeof( *params ), params );
	WRITE_OPCODE( WIREGL_TEXGENFV_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexGeniv  )( GLenum coord, GLenum pname,
		const GLint *params )
{
	__handleTexGenData( coord, pname, sizeof( *params ), params );
	WRITE_OPCODE( WIREGL_TEXGENIV_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( GenTextures   )(GLsizei n, GLuint *textures)
{
	WIREGL_UNUSED(n);
	WIREGL_UNUSED(textures);
}

void __handleTexParameterData( GLenum target, GLenum pname, const GLfloat *params )
{
	unsigned char *data_ptr;
	int packet_length = sizeof( int ) + sizeof( target ) + sizeof( pname );
	int params_length = 0;

	switch( pname )
	{
		case GL_TEXTURE_MIN_FILTER:
		case GL_TEXTURE_MAG_FILTER:
		case GL_TEXTURE_WRAP_S:
		case GL_TEXTURE_WRAP_T:
		case GL_TEXTURE_PRIORITY:
			params_length = sizeof( *params );
			break;
		case GL_TEXTURE_BORDER_COLOR:
			params_length = 4* sizeof( *params );
			break;
		default:
			wireGLSimpleError( "Unnown pname: %d", pname );
			break;
	}
	packet_length += params_length;

	GET_BUFFERED_POINTER( packet_length );
	WRITE_DATA( 0, int, packet_length );
	WRITE_DATA( sizeof( int ) + 0, GLenum, target );
	WRITE_DATA( sizeof( int ) + 4, GLenum, pname );
	memcpy( data_ptr + sizeof( int ) + 8, params, params_length );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexParameterfv  )( GLenum target, GLenum pname, 
		const GLfloat *params )
{
	__handleTexParameterData( target, pname, params );
	WRITE_OPCODE( WIREGL_TEXPARAMETERFV_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexParameteriv  )( GLenum target, GLenum pname, 
		const GLint *params )
{
	__handleTexParameterData( target, pname, (GLfloat *) params );
	WRITE_OPCODE( WIREGL_TEXPARAMETERIV_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexSubImage2D   )(GLenum target, GLint level, 
		GLint xoffset, GLint yoffset, GLsizei width, GLsizei height, 
		GLenum format, GLenum type, const GLvoid *pixels)
{
	unsigned char *data_ptr;
	int packet_length;

	packet_length = 
		sizeof( target ) +
		sizeof( level ) +
		sizeof( xoffset ) +
		sizeof( yoffset ) +
		sizeof( width ) +
		sizeof( height ) +
		sizeof( format ) +
		sizeof( type ) +
		width*height*__gl_PixelSizeInBytes( format, type );

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );
	WRITE_DATA( 0, GLenum, target );
	WRITE_DATA( 4, GLint, level );
	WRITE_DATA( 8, GLint, xoffset );
	WRITE_DATA( 12, GLint, yoffset );
	WRITE_DATA( 16, GLsizei, width );
	WRITE_DATA( 20, GLsizei, height );
	WRITE_DATA( 24, GLenum, format );
	WRITE_DATA( 28, GLenum, type );

	__glpixel_getdata2D((GLvoid *) (data_ptr + 32),
		width, height, format, type, pixels );

	wireGLSendPacket( WIREGL_TEXSUBIMAGE2D_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( TexSubImage1D   )(GLenum target, GLint level, 
		GLint xoffset, GLsizei width, GLenum format, GLenum type, const GLvoid *pixels)
{
	unsigned char *data_ptr;
	int packet_length;

	packet_length = 
		sizeof( target ) +
		sizeof( level ) +
		sizeof( xoffset ) +
		sizeof( width ) +
		sizeof( format ) +
		sizeof( type ) +
		width*__gl_PixelSizeInBytes( format, type );

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );
	WRITE_DATA( 0, GLenum, target );
	WRITE_DATA( 4, GLint, level );
	WRITE_DATA( 8, GLint, xoffset );
	WRITE_DATA( 12, GLsizei, width );
	WRITE_DATA( 16, GLenum, format );
	WRITE_DATA( 20, GLenum, type );

	__glpixel_getdata2D((GLvoid *) (data_ptr + 24),
		width, 1, format, type, pixels );

	wireGLSendPacket( WIREGL_TEXSUBIMAGE1D_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}

