/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_util.h"

int
wireGLPipeIntersectRect( WireGLPipeServer *pipe, int *rect, int *isect )
{
	/* rect[4] = { x1, y1, x2, y2 } */
	isect[0] = pipe->mural_x[0];
	isect[1] = pipe->mural_y[0];
	isect[2] = pipe->mural_x[0] + pipe->mural_w[0];
	isect[3] = pipe->mural_y[0] + pipe->mural_h[0];

	if ( isect[0] < rect[0] ) isect[0] = rect[0];
	if ( isect[1] < rect[1] ) isect[1] = rect[1];
	if ( isect[2] > rect[2] ) isect[2] = rect[2];
	if ( isect[3] > rect[3] ) isect[3] = rect[3];

	return ( isect[2] > isect[0] && isect[3] > isect[1] );
}

void OPENGL_APIENTRY
WIREGL_PACK_FUNCTION( ReadPixels )( GLint x, GLint y, 
									GLsizei width, GLsizei height,
									GLenum format, GLenum type,
									GLvoid *pixels )
{
	WireGLGlobals *g = &__wiregl_globals;

	int i, rect[4], stride;
	int bytes_per_pixel = 0;

	wireGLWarning( WIREGL_WARN_DEBUG, "ReadPixels( x=%d, y=%d, width=%d, "
				   "height=%d, format=0x%x, type=0x%x, pixels=0x%p )", 
				   x, y, width, height, format, type, pixels );

	wireGLFlushBuffer( g->context );

	switch ( type )
	{
	  case GL_UNSIGNED_BYTE:
	  case GL_BYTE:
		bytes_per_pixel = 1;
		break;

	  case GL_UNSIGNED_SHORT:
	  case GL_SHORT:
		bytes_per_pixel = 2;
		break;

	  case GL_UNSIGNED_INT:
	  case GL_INT:
	  case GL_FLOAT:
		bytes_per_pixel = 4;
		break;

	  default:
		wireGLError( "ReadPixels: type=0x%x", type );
	}

	switch ( format )
	{
	  case GL_COLOR_INDEX:
	  case GL_STENCIL_INDEX:
	  case GL_DEPTH_COMPONENT:
	  case GL_RED:
	  case GL_GREEN:
	  case GL_BLUE:
	  case GL_ALPHA:
	  case GL_LUMINANCE:
		break;

	  case GL_LUMINANCE_ALPHA:
		bytes_per_pixel *= 2;
		break;

	  case GL_RGB:
		bytes_per_pixel *= 3;
		break;

	  case GL_RGBA:
		bytes_per_pixel *= 4;
		break;

	  default:
		wireGLError( "ReadPixels: format=0x%x", format );
	}

	rect[0] = x;
	rect[1] = y;
	rect[2] = x + width;
	rect[3] = y + height;
	stride  = width * bytes_per_pixel;

	for ( i = 0; i < g->num_servers; i++ )
	{
		WireGLPipeServer *pipe = &g->servers[i];
		int isect[4];
		int offset, pipe_width, pipe_height, pipe_x, pipe_y, bytes_per_row;
		unsigned char *data_ptr;

		if ( !wireGLPipeIntersectRect( pipe, rect, isect ) )
			continue;

		offset = ( width * isect[1] + isect[0] ) * bytes_per_pixel;

		pipe->conn->pending_writebacks++;

		pipe_width  = isect[2] - isect[0];
		pipe_height = isect[3] - isect[1];
		pipe_x      = isect[0] - pipe->mural_x[0];
		pipe_y      = isect[1] - pipe->mural_y[0];
		bytes_per_row = pipe_width * bytes_per_pixel;

		wireGLWarning( WIREGL_WARN_DEBUG, "ReadPixels: asking pipe=%d for "
					   "%dx%d pixels @%d,%d (%d,%d)", i, pipe_width, 
					   pipe_height, pipe_x, pipe_y, isect[0], isect[1] );

		data_ptr = GET_BUFFERED_POINTER( 32 + sizeof(WireGLNetworkPointer) );
		WRITE_DATA( 0,  GLint,  pipe_x );
		WRITE_DATA( 4,  GLint,  pipe_y );
		WRITE_DATA( 8,  GLint,  pipe_width );
		WRITE_DATA( 12, GLint,  pipe_height );
		WRITE_DATA( 16, GLenum, format );
		WRITE_DATA( 20, GLenum, type );
		WRITE_DATA( 24, GLint,  stride );
		WRITE_DATA( 28, GLint,  bytes_per_row );
		WRITE_NETWORK_POINTER( 32, (char *) pixels + offset );
		WRITE_OPCODE( WIREGL_READPIXELS_OPCODE );

		wireGLPipeServerBufferAppend( pipe, &g->buffer );
		wireGLSendPipeServerBuffer( pipe );

		g->buffer.data_current = g->buffer.data_start;
		g->buffer.opcode_current = g->buffer.opcode_start;
	}

	for ( i = 0; i < g->num_servers; i++ )
	{
		WireGLPipeServer *pipe = &g->servers[i];

		while ( pipe->conn->pending_writebacks > 0 )
		{
			wireGLNetRecv( );
		}
	}
}
