/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_common.h"

/* Structures
 * ==========
 *
 * CompressedNormalT
 * -----------------
 * This structure stores the compressed data that is produced by the
 * compression algorithm. The symmetry field contains information
 * regarding in which octant and sextant the normal lies, and the left
 * and right values contain a compressed version of nx/nz and ny/nz.
 *
 * NormalDataT
 * -----------
 * This type is used to force the compiler to write the
 * CompressedNormalT to the data buffer as a specific number of
 * bytes. Normally, the CompressedNormalT would take up four bytes
 * even if only the first two are being used. So, we declare a
 * separate type that takes up the amount of room we want to use in
 * the data buffer and use a pointer cast to write only that much
 * data.
 *
 * COMPRESSION_SHIFT
 * -----------------
 * This value dictates how many bits should be removed from the
 * mantissa of the floating point value in the quantization step.  
 */

typedef struct
{
	unsigned symmetry : 6;
	unsigned leftValue : 5;
	unsigned rightValue : 5;
} CompressedNormalT;

typedef unsigned short NormalDataT;

#define QUANTIZATION_LEVEL 5

#include <limits.h>
#include <math.h>
#include "wiregl/util/floatlib.c"

// keep the data word-aligned
#define NORMAL_SIZE ((sizeof(NormalDataT) + 3) & ~0x3)

#define NEGATIVE_X 0x0001
#define NEGATIVE_Y 0x0002
#define NEGATIVE_Z 0x0004
#define SEXTANT_XY 0x0008
#define SEXTANT_XZ 0x0010
#define SEXTANT_YZ 0x0020

static GLfloat compressionMultiplier = (1 << QUANTIZATION_LEVEL) - 1;

#ifdef WINDOWS
#pragma warning( disable : 4725 )
#define QUANTIZE_VALUES() __asm						\
	{												\
		__asm fld nx								\
		__asm fdiv nz								\
		__asm fmul compressionMultiplier			\
		__asm fistp shortMantissa					\
		__asm mov   eax, shortMantissa				\
	}												\
													\
	__asm											\
	{												\
		__asm fld ny								\
		__asm fdiv nz								\
		__asm fmul compressionMultiplier			\
		__asm fistp shortMantissa					\
		__asm mov   ebx, shortMantissa				\
	}												\
													\
	__asm											\
	{												\
		__asm shl eax, 6							\
		__asm shl ebx, 11							\
		__asm or  outData, ax						\
		__asm or  outData, bx						\
	}

#else	/* !WINDOWS */

#define QUANTIZE_VALUES() outNormal->leftValue = (unsigned char)(nx / nz * compressionMultiplier); \
	outNormal->rightValue = (unsigned char)(ny / nz * compressionMultiplier); \

#endif	/* !WINDOWS */

/* swap
 * ----
 * This function swaps two floating point values.
 */

static __inline void swap(GLfloat *in1, GLfloat *in2)
{
	GLfloat temp;
	temp = *in1;
	*in1 = *in2;
	*in2 = temp;
}

/* __assignOctantAndSextant
 * ------------------------
 * This function takes as arguments three floating point values that
 * represent the normal to be compressed and a CompressedNormalT
 * structure that is the target of the compressed data.  The three
 * values sent in will be rearranged so that the largest of the three
 * will be stored in nx, and the function will replace each value with
 * its absolute value. Additionally, the function will store the
 * symmetry data in the symmetry field of the CompressedNormalT
 * structure.  
 */

#ifdef WINDOWS

static __inline void __assignOctantAndSextant( CompressedNormalT *compressedNormal, GLfloat *nx, GLfloat *ny, GLfloat *nz)
{
	NormalDataT *normal = (NormalDataT*)compressedNormal;

	/* Use faster versions of __isNegative and __absoluteValue that use bit arithmetic. */
	
	if(__isNegative(*nx))
	{
		*normal |= NEGATIVE_X;
		__absoluteValue(nx);
	}
	if(__isNegative(*ny))
	{
		*normal |= NEGATIVE_Y;
		__absoluteValue(ny);
	}
	if(__isNegative(*nz))
	{
		*normal |= NEGATIVE_Z;
		__absoluteValue(nz);
	}

	/* Assign sextants and sort normal indices */
	
	if(*(int*)nx < *(int*)ny)				// NX < NY
	{
		*normal |= SEXTANT_XY;
		if(*(int*)nx < *(int*)nz)			// NX < NY && NX < NZ
		{
			*normal |= SEXTANT_XZ;
			if(*(int*)ny < *(int*)nz)		// NX < NY < NZ
				*normal |= SEXTANT_YZ;
			else							// NX < NZ < NY
				swap(ny,nz);
		}
		else								// NZ < NX < NY
		{
			swap(ny,nz);
		}
	}
	else									// NY < NX
	{
		if(*(int*)ny < *(int*)nz)			// NY < NX && NY < NZ
		{
			*normal |= SEXTANT_YZ;
			if(*(int*)nx < *(int*)nz)		// NY < NX < NZ
			{
				*normal |= SEXTANT_XZ;
			}
			else							// NY < NZ < NX
			{
				swap(nx, nz);
			}
		}
		else								// NZ < NY < NX
		{
			swap(nx, nz);
		}
	}
}

#else /* WINDOWS */

static __inline void __assignOctantAndSextant( CompressedNormalT *normal, GLfloat *nx, GLfloat *ny, GLfloat *nz)
{
	unsigned short outData = 0;
	
	/* Use faster versions of __isNegative and __absoluteValue that use bit arithmetic. */
	
	if(*nx < 0)
	{
		outData |= NEGATIVE_X;
		*nx = -*nx;
	}
	if(*ny < 0)
	{
		outData |= NEGATIVE_Y;
		*ny = -*ny;
	}
	if(*nz < 0)
	{
		outData |= NEGATIVE_Z;
		*nz = -*nz;
	}

	/* Assign sextants and sort normal indices */
	
	if(*nx < *ny)				// NX < NY
	{
		outData |= SEXTANT_XY;
		if(*nx < *nz)			// NX < NY && NX < NZ
		{
			outData |= SEXTANT_XZ;
			if(*ny < *nz)		// NX < NY < NZ
				outData |= SEXTANT_YZ;
			else							// NX < NZ < NY
				swap(ny,nz);
		}
		else								// NZ < NX < NY
		{
			swap(ny,nz);
		}
	}
	else									// NY < NX
	{
		if(*ny < *nz)			// NY < NX && NY < NZ
		{
			outData |= SEXTANT_YZ;
			if(*nx < *nz)		// NY < NX < NZ
			{
				outData |= SEXTANT_XZ;
			}
			else							// NY < NZ < NX
			{
				swap(nx, nz);
			}
		}
		else								// NZ < NY < NX
		{
			swap(nx, nz);
		}
	}

	normal->symmetry |= outData;
}

#endif /* !WINDOWS */

/* __handleNormalData
 * ------------------
 * This function is responsible for compressing the normal data sent
 * in by the GL client.  It first stores the symmetry data by calling
 * __assignOctantAndSextant and then compresses nx/nz and ny/nz by
 * quantizing the values to 5 bits.  */

static __inline void __handleNormalData( GLfloat nx, GLfloat ny, GLfloat nz )
{
	unsigned char *data_ptr;

#ifdef WINDOWS
	unsigned int shortMantissa;
#endif /* WINDOWS */

	NormalDataT outData = 0;
	CompressedNormalT *outNormal = (CompressedNormalT*)&outData;

	GET_BUFFERED_POINTER(NORMAL_SIZE);

	__assignOctantAndSextant(outNormal, &nx, &ny, &nz);

	QUANTIZE_VALUES();
	
	WRITE_DATA(0, NormalDataT, outData);
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Normal3dc )( GLdouble nx, GLdouble ny, GLdouble nz )
{
	__handleNormalData((GLfloat)nx, (GLfloat)ny, (GLfloat)nz);
	WRITE_OPCODE( WIREGL_NORMAL3COMPRESSED_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Normal3dvc )( const GLdouble *v )
{
	__handleNormalData((GLfloat)v[0], (GLfloat)v[1], (GLfloat)v[2]);
	WRITE_OPCODE( WIREGL_NORMAL3COMPRESSED_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Normal3fc )( GLfloat nx, GLfloat ny, GLfloat nz )
{
	__handleNormalData(nx, ny, nz);
	WRITE_OPCODE( WIREGL_NORMAL3COMPRESSED_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Normal3fvc )( const GLfloat *v )
{
	__handleNormalData(v[0], v[1], v[2]);
	WRITE_OPCODE( WIREGL_NORMAL3COMPRESSED_OPCODE );
}

/* glNormal3x
 * ----------
 * The functions that take in integer values require conversion to
 * floating point values before compressing them. GL requires this
 * conversion at some point in the pipeline, so this step does not
 * require a significant amount of time.  
 */

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Normal3ic )( GLint nx, GLint ny, GLint nz )
{
	__handleNormalData((GLfloat)(2.0f * nx + 1) / UINT_MAX,
		(GLfloat)(2.0f * ny + 1) / UINT_MAX,
		(GLfloat)(2.0f * nz + 1) / UINT_MAX);

	WRITE_OPCODE( WIREGL_NORMAL3COMPRESSED_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Normal3ivc )( const GLint *v )
{
	__handleNormalData((GLfloat)(2.0f * v[0] + 1) / UINT_MAX,
		(GLfloat)(2.0f * v[1] + 1) / UINT_MAX,
		(GLfloat)(2.0f * v[2] + 1) / UINT_MAX);

	WRITE_OPCODE( WIREGL_NORMAL3COMPRESSED_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Normal3sc )( GLshort nx, GLshort ny, GLshort nz )
{
	__handleNormalData((GLfloat)(2.0f * nx + 1) / USHRT_MAX,
		(GLfloat)(2.0f * ny + 1) / USHRT_MAX, 
		(GLfloat)(2.0f * nz + 1) / USHRT_MAX);

	WRITE_OPCODE( WIREGL_NORMAL3COMPRESSED_OPCODE );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Normal3svc )( const GLshort *v )
{
	__handleNormalData((GLfloat)(2.0f * v[0] + 1) / USHRT_MAX,
		(GLfloat)(2.0f * v[1] + 1) / USHRT_MAX,
		(GLfloat)(2.0f * v[2] + 1) / USHRT_MAX);

	WRITE_OPCODE( WIREGL_NORMAL3COMPRESSED_OPCODE );
}

GLapi wiregl_compresspack_api[] = {
	{ "glNormal3d", (GLfunc) __glpack_Normal3dc },
	{ "glNormal3dv", (GLfunc) __glpack_Normal3dvc },
	{ "glNormal3f", (GLfunc) __glpack_Normal3fc },
	{ "glNormal3fv", (GLfunc) __glpack_Normal3fvc },
	{ "glNormal3i", (GLfunc) __glpack_Normal3ic },
	{ "glNormal3iv", (GLfunc) __glpack_Normal3ivc },
	{ "glNormal3s", (GLfunc) __glpack_Normal3sc },
	{ "glNormal3sv", (GLfunc) __glpack_Normal3svc },
	{ NULL, NULL}
};
