/*
** 3/06/2001
** http://graphics.stanford.edu/software/wiregl
**
** Copyright 2001
** The Board of Trustees of The Leland Stanford Junior University.
** All rights reserved.
**
** Except for commercial resale, lease, license or other commercial
** transactions, permission is hereby given to use, copy, and/or
** modify this software, provided that the above copyright notice and
** this permission notice appear in all copies of this software.  No
** part of this software or any derivatives thereof may be used in
** graphics systems for resale or for use in a commercial product.
**
** This software is provided "as is" and without warranty of any kind,
** express, implied or otherwise, including without limitation, any
** warranty of merchantability or fitness for a particular purpose.
*/

#include "wiregl/include/wireGL.h"
#include "wiregl/include/wiregl_protocol.h"
#include "wiregl/include/wiregl_client.h"
#include "wiregl/include/wiregl_util.h"

/* Note -- for these packets, the ustride and vstride are implicit,
 * and are computed into the packet instead of copied.
 */

int __gl_Map2NumComponents( GLenum target )
{
	switch( target )
	{
	case GL_MAP2_VERTEX_3:
	case GL_MAP2_NORMAL:
	case GL_MAP2_TEXTURE_COORD_3:
		return 3;
	case GL_MAP2_VERTEX_4:
	case GL_MAP2_COLOR_4:
	case GL_MAP2_TEXTURE_COORD_4:
		return 4;
	case GL_MAP2_INDEX:
	case GL_MAP2_TEXTURE_COORD_1:
		return 1;
	case GL_MAP2_TEXTURE_COORD_2:
		return 2;
	default:
		wireGLSimpleError( "Invalid map target: %d", target );
	}
	return -1;
}

int __gl_Map1NumComponents( GLenum target )
{
	switch( target )
	{
	case GL_MAP1_VERTEX_3:
	case GL_MAP1_NORMAL:
	case GL_MAP1_TEXTURE_COORD_3:
		return 3;
	case GL_MAP1_VERTEX_4:
	case GL_MAP1_COLOR_4:
	case GL_MAP1_TEXTURE_COORD_4:
		return 4;
	case GL_MAP1_INDEX:
	case GL_MAP1_TEXTURE_COORD_1:
		return 1;
	case GL_MAP1_TEXTURE_COORD_2:
		return 2;
	default:
		wireGLSimpleError( "Invalid map target: %d", target );
	}
	return -1;
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Map2d  )(GLenum target, GLdouble u1, 
		GLdouble u2, GLint ustride, GLint uorder, GLdouble v1, GLdouble v2, 
		GLint vstride, GLint vorder, const GLdouble *points)
{
	unsigned char *data_ptr;
	int u,v;
	GLdouble *dest_data, *src_data;
	int packet_length = 
		sizeof( target ) + 
		sizeof( u1 ) +
		sizeof( u2 ) +
		sizeof( uorder ) +
		sizeof( ustride ) +
		sizeof( v1 ) +
		sizeof( v2 ) + 
		sizeof( vorder ) +
		sizeof( vstride );

	int num_components = __gl_Map2NumComponents( target );
	packet_length += num_components*uorder*vorder*sizeof( *points );

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );

	WRITE_DATA( 0, GLenum, target );
	WRITE_DOUBLE( 4, u1 );
	WRITE_DOUBLE( 12, u2 );
	WRITE_DATA( 20, GLint, num_components );
	WRITE_DATA( 24, GLint, uorder );
	WRITE_DOUBLE( 28, v1 );
	WRITE_DOUBLE( 36, v2 );
	WRITE_DATA( 44, GLint, num_components*uorder );
	WRITE_DATA( 48, GLint, vorder );

	dest_data = (GLdouble *) (data_ptr + 52);
	src_data = (GLdouble *) points;
	for (v = 0 ; v < vorder ; v++)
	{
		for (u = 0 ; u < uorder ; u++)
		{
			memcpy( dest_data, src_data, num_components * sizeof( *points ) );
			dest_data += num_components;
			src_data += ustride;
		}
		src_data += vstride - ustride*uorder;
	}

	wireGLSendPacket( WIREGL_MAP2D_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Map2f  )(GLenum target, GLfloat u1, 
		GLfloat u2, GLint ustride, GLint uorder, GLfloat v1, GLfloat v2, 
		GLint vstride, GLint vorder, const GLfloat *points)
{
	unsigned char *data_ptr;
	int u,v;
	GLfloat *dest_data, *src_data;
	int packet_length = 
		sizeof( target ) + 
		sizeof( u1 ) +
		sizeof( u2 ) +
		sizeof( uorder ) +
		sizeof( ustride ) +
		sizeof( v1 ) +
		sizeof( v2 ) + 
		sizeof( vorder ) +
		sizeof( vstride );

	int num_components = __gl_Map2NumComponents( target );
	packet_length += num_components*uorder*vorder*sizeof( *points );

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );

	WRITE_DATA( 0, GLenum, target );
	WRITE_DATA( 4, GLfloat, u1 );
	WRITE_DATA( 8, GLfloat, u2 );
	WRITE_DATA( 12, GLint, num_components );
	WRITE_DATA( 16, GLint, uorder );
	WRITE_DATA( 20, GLfloat, v1 );
	WRITE_DATA( 24, GLfloat, v2 );
	WRITE_DATA( 28, GLint, num_components*uorder );
	WRITE_DATA( 32, GLint, vorder );

	dest_data = (GLfloat *) (data_ptr + 36);
	src_data = (GLfloat *) points;
	for (v = 0 ; v < vorder ; v++)
	{
		for (u = 0 ; u < uorder ; u++)
		{
			memcpy( dest_data, src_data, num_components * sizeof( *points ) );
			dest_data += num_components;
			src_data += ustride;
		}
		src_data += vstride - ustride*uorder;
	}

	wireGLSendPacket( WIREGL_MAP2F_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Map1d  )( GLenum target, GLdouble u1,
		GLdouble u2, GLint stride, GLint order, const GLdouble *points )
{
	unsigned char *data_ptr;
	int packet_length = 
		sizeof( target ) + 
		sizeof( u1 ) +
		sizeof( u2 ) + 
		sizeof( stride ) + 
		sizeof( order );

	int num_components = __gl_Map1NumComponents( target );
	GLdouble *src_data, *dest_data;
	int u;

	packet_length += num_components * order * sizeof( *points );

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );

	WRITE_DATA( 0, GLenum, target );
	WRITE_DOUBLE( 4, u1 );
	WRITE_DOUBLE( 12, u2 );
	WRITE_DATA( 20, GLint, num_components );
	WRITE_DATA( 24, GLint, order );

	dest_data = (GLdouble *) (data_ptr + 28);
	src_data = (GLdouble *) points;
	for (u = 0 ; u < order ; u++)
	{
		memcpy( dest_data, src_data, num_components * sizeof( *points ) );
		dest_data += num_components;
		src_data += stride;
	}

	wireGLSendPacket( WIREGL_MAP1D_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}

void OPENGL_APIENTRY WIREGL_PACK_FUNCTION( Map1f  )( GLenum target, GLfloat u1,
		GLfloat u2, GLint stride, GLint order, const GLfloat *points )
{
	unsigned char *data_ptr;
	int packet_length = 
		sizeof( target ) + 
		sizeof( u1 ) +
		sizeof( u2 ) + 
		sizeof( stride ) + 
		sizeof( order );

	int num_components = __gl_Map1NumComponents( target );
	GLfloat *src_data, *dest_data;
	int u;

	packet_length += num_components * order * sizeof( *points );

	data_ptr = (unsigned char *) wireGLAllocPacket( packet_length );

	WRITE_DATA( 0, GLenum, target );
	WRITE_DATA( 4, GLfloat, u1 );
	WRITE_DATA( 8, GLfloat, u2 );
	WRITE_DATA( 12, GLint, num_components );
	WRITE_DATA( 16, GLint, order );

	dest_data = (GLfloat *) (data_ptr + 20);
	src_data = (GLfloat *) points;
	for (u = 0 ; u < order ; u++)
	{
		memcpy( dest_data, src_data, num_components * sizeof( *points ) );
		dest_data += num_components;
		src_data += stride;
	}

	wireGLSendPacket( WIREGL_MAP1F_OPCODE, data_ptr );
	wireGLFreePacket( data_ptr );
}
